/*	AppDelegate.m - Handles launch and termination of application
 
 $Id: AppDelegate.m,v 1.17 2006/02/28 10:27:15 willthimbleby Exp $
 
 Copyright 2002 Will Thimbleby (will@thimbleby.net)
 
 This program is free software; you can redistribute it and/or modify it under the terms of the
 GNU General Public License as published by the Free Software Foundation version 2.
 
 This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
 */

#import "AppDelegate.h"
#import "WebKit/WebPreferences.h"
#import "CocoLogger/CocoLogger.h"
#import "MBThreadedProgressSheetController.h"
#import "SwordInstallSourceController.h"
#import "IndexingManager.h"
#import "ModuleManager.h"
#import "BibleService.h"
#import "MyDocument.h"
#import "SwordURLProtocol.h"
#import "NSString_Extensions.h"
#import "FindControl.h"
#import "EditView.h"
#import "globals.h"
#import "SwordManager.h"
#import "SwordBible.h"
#import "PrefHandler.h"

// Globals
NSArray	*daysPerMonth;

NSString* pathForFolderType(OSType dir,short domain,BOOL createFolder) {
	OSStatus err = 0;
	FSRef folderRef;
	NSString *path = nil;
	NSURL *url = nil;
	
	err = FSFindFolder(domain,dir,createFolder,&folderRef);
	if(err == 0) {
		url = (NSURL *)CFURLCreateFromFSRef(kCFAllocatorSystemDefault, &folderRef);
		if(url) {
			path = [NSString stringWithString:[url path]];
			[url release];
		}
	}
    
	return path;
}

@implementation AppDelegate

/**
 \brief this method is called before any message is sent to this object
 Do low level inits here
 */
+ (void)initialize {
	// get path to "Logs" folder of current user
	NSString *logPath = @"~/Library/Logs/MacSword.log";
	// expand tilde
	logPath = [logPath stringByExpandingTildeInPath];
	
#ifdef DEBUG
	// init the logging facility in first place
	[MBLogger initLogger:logPath logPrefix:@"[MacSword]" logFilterLevel:MBLOG_DEBUG appendToFile:NO logToConsole:YES];
#endif
#ifdef RELEASE
	// init the logging facility in first place
	[MBLogger initLogger:logPath logPrefix:@"[MacSword]" logFilterLevel:MBLOG_WARN appendToFile:YES logToConsole:NO];
#endif
}

/**
 sets up all needed folders so the application can work
 */
- (BOOL)setupFolders {
    BOOL ret = YES;
    
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    
    // get app support path
	NSString *path = pathForFolderType(kApplicationSupportFolderType, kUserDomain, true);
	if(path == nil) {
		MBLOG(MBLOG_ERR, @"Cannot get path to Application Support!");
	} else {
        MBLOG(MBLOG_INFO, @"Have path to AppSupport, ok.");
        
        // add path for application path in Application Support
        path = [path stringByAppendingPathComponent:APPNAME];
        
        // check if dir for application exists
        NSFileManager *manager = [NSFileManager defaultManager];
        if([manager fileExistsAtPath:path] == NO) {
            MBLOG(MBLOG_INFO, @"path to MacSword does not exist, creating it!");
            // create
            if([manager createDirectoryAtPath:path attributes:nil] == NO) {
                MBLOG(MBLOG_ERR,@"Cannot create folder in Application Support!");
                ret = NO;
            }
        }
        
        // on no error continue
        if(ret) {
            // create IndexFolder folder
            NSString *indexPath = [path stringByAppendingPathComponent:@"Index"];
            if([manager fileExistsAtPath:indexPath] == NO) {
                MBLOG(MBLOG_INFO, @"path to IndexFolder does not exist, creating it!");
                if([manager createDirectoryAtPath:indexPath attributes:nil] == NO) {
                    MBLOG(MBLOG_ERR,@"Cannot create installmgr folder in Application Support!");
                }                
            }
            // put to defaults
            [defaults setObject:indexPath forKey:DEFAULTS_SWINDEX_PATH_KEY];
            [defaults synchronize];
        }

        // create default modules folder which is Sword
        path = DEFAULT_MODULE_PATH;
        if([manager fileExistsAtPath:path] == NO) {
            MBLOG(MBLOG_INFO, @"path to swmodules does not exist, creating it!");
            if([manager createDirectoryAtPath:path attributes:nil] == NO) {
                MBLOG(MBLOG_ERR,@"Cannot create swmodules folder in Application Support!");
                ret = NO;
            }
            
            // check for "mods.d" folder
            NSString *modsFolder = [path stringByAppendingPathComponent:@"mods.d"];
            if([manager fileExistsAtPath:modsFolder] == NO) {
                // create it
                if([manager createDirectoryAtPath:modsFolder attributes:nil] == NO) {
                    MBLOG(MBLOG_ERR, @"Could not create mods.d folder!");
                }
            }
        }
        // put to defaults
        [defaults setObject:path forKey:DEFAULTS_SWMODULE_PATH_KEY];
        [defaults synchronize];                    
        
        // on no error continue
        if(ret) {
            // create InstallMgr folder
            NSString *installMgrPath = [path stringByAppendingPathComponent:SWINSTALLMGR_NAME];
            if([manager fileExistsAtPath:installMgrPath] == NO) {
                MBLOG(MBLOG_INFO, @"path to imstallmgr does not exist, creating it!");
                if([manager createDirectoryAtPath:installMgrPath attributes:nil] == NO) {
                    MBLOG(MBLOG_ERR,@"Cannot create installmgr folder in Application Support!");
                    ret = NO;
                }                
            }
            // put to defaults
            [defaults setObject:installMgrPath forKey:DEFAULTS_SWINSTALLMGR_PATH_KEY];
            [defaults synchronize];
        }
	}
    
    return ret;
}

/*" Startup code
 "*/
- (void)applicationDidFinishLaunching:(NSNotification *)notification {
	// initialise services
    BibleService *serviceProvider = [[BibleService alloc] init];
	[NSApp setServicesProvider:serviceProvider];
    
	// know the number of days per month
	daysPerMonth = [[NSArray arrayWithObjects:
                     [NSNumber numberWithInt:1],		// about
                     [NSNumber numberWithInt:31],
                     [NSNumber numberWithInt:29],	// always assume a leap year
                     [NSNumber numberWithInt:31],
                     [NSNumber numberWithInt:30],
                     [NSNumber numberWithInt:31],
                     [NSNumber numberWithInt:30],
                     [NSNumber numberWithInt:31],
                     [NSNumber numberWithInt:31],
                     [NSNumber numberWithInt:30],
                     [NSNumber numberWithInt:31],
                     [NSNumber numberWithInt:30],
                     [NSNumber numberWithInt:31], nil] retain];
	
    // init AppSupportFolder
    BOOL success = [self setupFolders];
    if(!success) {
        MBLOG(MBLOG_ERR, @"[AppController -init] could not initialize AppSupport!");
    } else {
        // initialize ThreadedProgressSheet
        [MBThreadedProgressSheetController standardProgressSheetController];
        
        // init install manager
        SwordInstallSourceController *sim = [SwordInstallSourceController defaultController];
        [sim setConfigPath:[userDefaults stringForKey:DEFAULTS_SWINSTALLMGR_PATH_KEY]];

        // init indexingmanager, set base index path
        IndexingManager *im = [IndexingManager sharedManager];
        [im setBaseIndexPath:[userDefaults stringForKey:DEFAULTS_SWINDEX_PATH_KEY]];
        
        // load manager - hmmm usless comments
        [self loadManager];
        
        // post notifications to other objects to load up
        [[NSNotificationCenter defaultCenter] postNotificationName:@"start" object:NULL];
        
        // set version in about box
        [versionTextBox setStringValue:[[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleVersion"]];
        
        //initialise url handlers
        [[NSAppleEventManager sharedAppleEventManager] setEventHandler:self 
                                                           andSelector:@selector( handleURLEvent:withReplyEvent: ) 
                                                         forEventClass:kInternetEventClass 
                                                            andEventID:kAEGetURL];
        [SwordURLProtocol setup];
        
        //open saved documents
        if([[NSUserDefaults standardUserDefaults] boolForKey:@"remember windows"]) {
            [prefHandler openWindows];
        }
        
        // automatically open todays devotion if desired
        if([[NSUserDefaults standardUserDefaults] boolForKey:@"auto devote"]) {
            [self todaysDevotion:self];
        }
        
        //if no modules
        if([[[SwordManager defaultManager] allModuleNames] count] == 0) {
            NSRunInformationalAlertPanel(
                                         NSLocalizedString(@"No modules title", @"Title for warning on starting app for the first time without modules."),
                                         NSLocalizedString(@"No modules info", @"Information for warning on starting app for the first time without modules."),
                                         @"OK", NULL, NULL);
        }
        
        //save webkit stuff
        [[WebPreferences standardPreferences] setAutosaves:YES];
    }
}

/*"	load up the sword manager
 from the location specified in the user preferences
 "*/
- (void)loadManager {
	NSString *location;
	
	// gets location and removes things like ~
	location = [[NSUserDefaults standardUserDefaults] stringForKey:@"moduleLocation"];
	location = [location stringByStandardizingPath];
	
	if([location hasPrefix:@"."] || (![location hasPrefix:@"/"] && ![location hasPrefix:@"~"])) {
		location = [NSString stringWithFormat:@"%@/%@", [[[NSBundle mainBundle] bundlePath] stringByDeletingLastPathComponent], location];
	}
    
	// load up the sword manager in this directory - use a different manager for the searching
	[SwordManager initManagersWithPath:location];
}

- (void)applicationWillTerminate:(NSNotification *)aNotification {
	// get prefs one last time
	[prefHandler updatePreferences:self];
	[prefHandler saveWindows];
}

- (void)todaysDevotion:(id)sender {
	NSString *moduleName;
	NSArray *documents;
	MyDocument *d = nil;
	
	// get the devotional module
	moduleName = [[NSUserDefaults standardUserDefaults] objectForKey:@"devotional"];
	
	if(!moduleName || [moduleName isEqualToString:@"None"]) {
		return;
	}
	
	//see if we alreday have a document to use
	documents = [[NSDocumentController sharedDocumentController] documents];
	
	for(int i = 0;i < [documents count];i++) {
		d = (MyDocument*)[documents objectAtIndex:i];
        
		if([[d getTag] isEqualToString:@"daily"]) {
			[d openURLString:[NSString stringWithFormat:@"sword://today@%@/%@", moduleName,[[NSCalendarDate calendarDate] descriptionWithCalendarFormat:@"%m.%d"]]];
			return;
		}
	}
	
	// open a new document
	MyDocument *mdoc = [MyDocument openDocumentWithURL:[NSURL URLWithString:[NSString stringWithFormat:@"sword://today@%@/%@", moduleName,[[NSCalendarDate calendarDate] descriptionWithCalendarFormat:@"%m.%d"]]]];
	[mdoc setTag:@"daily"];
}

- (void) handleURLEvent:(NSAppleEventDescriptor *) event withReplyEvent:(NSAppleEventDescriptor *) replyEvent {
	[MyDocument openDocumentWithURLString:[[event descriptorAtIndex:1] stringValue]];
}

- (void)newWindow:(id)sender {
	NSURL *url;
	NSString *mod = [[NSUserDefaults standardUserDefaults] objectForKey:@"serviceBible"];
	
	if([[NSDocumentController sharedDocumentController] currentDocument]) {
		url = [[[NSDocumentController sharedDocumentController] currentDocument] url];
	} else {
		url = [NSURL URLWithString:[[NSString stringWithFormat:@"sword://%@/", mod] addPercentEscapes]];
	}
	
	// open a new document
	if(url && mod && ![mod isEqualToString:@"None"]) {
		[MyDocument openDocumentWithURL:url];
	}
}

- (void)newModule:(id)sender {
    NSSavePanel		*oPanel = [NSSavePanel savePanel];
	
	[oPanel setRequiredFileType:@"swd"];
	int returnCode = [oPanel runModalForDirectory:[[NSUserDefaults standardUserDefaults] stringForKey:@"moduleLocation"] file:nil];
	
	// if click ok
    if (returnCode == NSOKButton) {
        NSString *fileName = [oPanel filename];
        NSString *modName = [[fileName lastPathComponent] stringByDeletingPathExtension];
		
		NSFileWrapper *newMod = [[NSFileWrapper alloc] initWithPath:[[[NSBundle mainBundle] bundlePath] stringByAppendingPathComponent:@"Contents/Resources/Personal"]];
		NSFileWrapper *temp;
		
		temp = [[newMod fileWrappers] objectForKey:@"modules"];
		temp = [[temp fileWrappers] objectForKey:@"comments"];
		temp = [[temp fileWrappers] objectForKey:@"rawfiles"];
		temp = [[temp fileWrappers] objectForKey:@"personal"];
		[temp setPreferredFilename:modName];
		
		temp = [[newMod fileWrappers] objectForKey:@"mods.d"];
		[temp removeFileWrapper:[[temp fileWrappers] objectForKey:@"personal.conf"]];
		
		//create the conf file
		NSString *confStr = [NSString stringWithContentsOfFile:[[[NSBundle mainBundle] bundlePath] stringByAppendingPathComponent:@"Contents/Resources/Personal/mods.d/personal.conf"]];
		confStr = [NSString stringWithFormat:confStr, modName, modName];
        
		NSFileWrapper *confFW = [[NSFileWrapper alloc] initRegularFileWithContents:[confStr dataUsingEncoding:NSUTF8StringEncoding]];
		[confFW setPreferredFilename:[NSString stringWithFormat:@"%@.conf", modName]];
		
		[temp addFileWrapper:confFW];
		
		[newMod setFilename:modName];
		
		[newMod writeToFile:fileName atomically:YES updateFilenames:NO];
		
		//now open the document
		[[SwordManager defaultManager] addPath:fileName];
        
		[MyDocument openDocumentWithURLString:[NSString stringWithFormat:@"sword://%@/", modName]];
    }
}

- (void)newFindWindow:(id)sender {
	[[[FindControl allocWithZone:[self zone]] init] showWindow:self];
}

/*
 - (void)askForKey
 {
 [NSApp beginSheet: cipherKeyDialog
 modalForWindow: [NSApp mainWindow]
 modalDelegate: self
 didEndSelector: @selector(didEndKeySheet:returnCode:contextInfo:)
 contextInfo: nil];
 }
 
 - (IBAction)okKeySheet: (id)sender
 {
 NSBeep();
 [NSApp endSheet:cipherKeyDialog];
 }
 
 - (void)didEndKeySheet:(NSWindow *)sheet returnCode:(int)returnCode contextInfo:(void *)contextInfo
 {
 [cipherKeyDialog orderOut:self];
 }
 */

- (BOOL)validateMenuItem:(NSMenuItem *)anItem {
	int tag = [anItem tag];
    
    switch(tag) {
		case 15: {
			// get the devotional module
			NSString *mod = [[NSUserDefaults standardUserDefaults] objectForKey:@"devotional"];
			// valid if module exists
			return (mod && ![mod isEqualToString:@"None"]);
		}
		default:
			return YES;
	}
}

// a couple of delegate methods that stop the application opening blank windows
- (BOOL)applicationShouldHandleReopen:(NSApplication *)theApplication hasVisibleWindows:(BOOL)flag {
	return NO;
}

- (BOOL)applicationShouldOpenUntitledFile:(NSApplication *)sender {
	return NO;
}

// ------------------- actions --------------------------
- (IBAction)openModuleInstaller:(id)sender {
    if(moduleManager == nil) {
        moduleManager = [[ModuleManager alloc] init];
    }
    
    // open window
    [moduleManager showWindow:self];
}

- (IBAction)menuGiveFeedback:(id)sender {
	[[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:@"mailto:macsword-user@lists.sourceforge.net"]];
}

- (IBAction)menuVisitWebsite:(id)sender {
	[[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:@"http://www.macsword.com"]];
}


@end
