//
//  MBLogger.m
//  CocoLogger
//
//  Created by Manfred Bergmann on 02.06.05.
//  Copyright 2005 mabe. All rights reserved.
//

// $Author: asrael $
// $HeadURL: file:///REPOSITORY/private/cocoa/CocoLogger/releases/CocoLogger-0.4/src/MBLogger.m $
// $LastChangedBy: asrael $
// $LastChangedDate: 2006-01-11 02:05:48 +0100 (Wed, 11 Jan 2006) $
// $Rev: 477 $

#import "MBLogger.h"

// some static variables
static FILE *cFHandle = nil;
static int logFilterLevel = MBLOG_WARN;
static NSString *logPrefix = nil;
static BOOL logToConsole = YES;
 
void MBLog(MBLoggingLevel level, NSString *msg, ...)
{
	// make mutable string out of input
	//NSString *str = [NSString stringWithFormat:msg,...];
	
	// call real logging method
	//[MBLogger log:str level:level];
}

@implementation MBLogger

/**
 \brief initialize the logger
 
 @param[in] logPath specify the path to where the outputgoes
 @param[in] aPrefix specify a log prefix e.g. the AppName
 @param[in] aLevel specify the log filter level to which the logger should be set to, can be changed with setLogFilterLevel:
 @param[in] fileAppend each time create a new file? YES or NO
 @param[in] consoleLogging specify if additionally the logging should go to the console
*/
+ (int)initLogger:(NSString *)logPath 
		logPrefix:(NSString *)aPrefix 
   logFilterLevel:(int)aLevel
	 appendToFile:(BOOL)fileAppend
	 logToConsole:(BOOL)consoleLogging
{
	int ret = 0;
	
	// copy logPrefix
	if(aPrefix != nil)
	{
		logPrefix = [aPrefix copy];
	}

	// set logFilterLevel
	logFilterLevel = aLevel;
	
	// set console logging
	logToConsole = consoleLogging;

	// check, if we have a path
	if(logPath != nil)
	{
		// check, if this file exists
		NSFileManager *fManager = [NSFileManager defaultManager];
		if(fManager != nil)	// if something went wrong
		{
			if([fManager fileExistsAtPath:logPath] == YES)
			{
				if(fileAppend == NO)
				{
					// if the file exists and we do not append, then we have to delete the file
					if([fManager removeFileAtPath:logPath handler:nil] == NO)
					{
						// we could not delete the file, but we go on appending to this file
						NSLog(@"Could not delete file at %@, but proceeding anyway!",logPath);
						
						char cString[256];
						[logPath getCString:cString maxLength:256];
						syslog(LOG_ERR,"Could not delete file at %s, but proceeding anyway!",cString); 
					}
				}
			}
			
			if([fManager fileExistsAtPath:logPath] == NO)
			{
				// then create it
				if([fManager createFileAtPath:logPath contents:nil attributes:nil] == NO)
				{
					NSLog(@"Cannot create logFile at %@!",logPath);
					
					char cString[256];
					[logPath getCString:cString maxLength:256];
					syslog(LOG_ERR,"Cannot create logFile at %s!",cString); 
					
					ret = -1;
				}
			}
			
			// do it the c way
			char cLogFile[256];
			[logPath getCString:cLogFile maxLength:256];
			cFHandle = fopen(cLogFile,"a+");
			if(cFHandle == nil)
			{
				// an error has occured
				NSLog(@"Cannot get cfilehandle from %@!",logPath);
				
				char cString[256];
				[logPath getCString:cString maxLength:256];
				syslog(LOG_ERR,"Cannot get cfilehandle from %s!",cString); 
				
				ret = -1;			
			}
		}
		else
		{
			// an error has occured
			NSLog(@"Cannot NSFileManager instance!",logPath);
			
			char cString[256];
			[logPath getCString:cString maxLength:256];
			syslog(LOG_ERR,"Cannot NSFileManager instance!",cString); 
			
			ret = -1;
		}
	}
	
	return ret;
}

+ (int)closeLogger
{
	int ret = 0;
	
	// release our prefix string
	[MBLogger setLogPrefix:nil];
	
	if(cFHandle != nil)
	{
		// close filehandle
		ret = fclose(cFHandle);
	}
	
	return ret;
}

/**
 \brief Set the logging filter level
 
 @param aLevel
*/
+ (void)setLogFilterLevel:(int)aLevel
{
	// set logFilterLevel
	logFilterLevel = aLevel;
}

/**
\brief Get the current log filter level
 */
+ (int)logFilterLevel
{
	return logFilterLevel;
}

/**
\brief Set the log prefix
 
 @param aPrefix
 */
+ (void)setLogPrefix:(NSString *)aPrefix
{
	aPrefix = [aPrefix copy];	// make immutable out of mutable
	[logPrefix release];
	logPrefix = aPrefix;
}

/**
\brief Get the log prefix
 */
+ (NSString *)logPrefix
{
	return logPrefix;
}

/**
\brief Log a message to file and/or console
 
 @param message the logging message
 @param aLevel the level of this message
 @returns error status code != 0
 */
+ (int)log:(NSString *)message level:(int)aLevel
{
	int ret = 0;
	
	// check for error or unposible setting
	if(cFHandle == nil)
	{
		if(logToConsole == NO)
		{
			// an error has occured
			NSLog(@"[MBLogger] Have no open filehandle and console logging is disabled!");

			syslog(LOG_ERR,"[MBLogger] Have no open filehandle and console logging is disabled!");
			
			return -1;
		}
	}
	
	// check, if message is nil
	if(message != nil)
	{
		// check, if we have to write a message
		if(aLevel <= logFilterLevel)
		{
			// add log level
			NSString *level = nil;
			switch(aLevel)
			{
				case MBLOG_CRIT:
					level = @"CRIT";
					break;
				case MBLOG_ERR:
					level = @"ERR";
					break;
				case MBLOG_WARN:
					level = @"WARN";
					break;
				case MBLOG_INFO:
					level = @"INFO";
					break;
				case MBLOG_DEBUG:
					level = @"DEBUG";
					break;
			}
			
			// insert prefix
			if(logPrefix == nil)
			{
				logPrefix = @"";
			}

			// build log str
			NSString *printString = [NSString stringWithFormat:@"%@ %@ %@ %@\n",
				[[NSDate date] description],
				level,
				logPrefix,
				message];
			
			// print string
			char cBytes[[printString length]];
			[printString getCString:cBytes];
			
			// check if cFHandle is not nil
			if(cFHandle != nil)
			{
				fprintf(cFHandle,"%s",cBytes);
				fflush(cFHandle);
			}
			
			// log to console as well if we have to
			if(logToConsole == YES)
			{
				fprintf(stdout,"%s",cBytes);
				fflush(stdout);
			}
		}
	}
	
	return ret;
}

@end
