
/*
 * Copyright (c) 2002 Packet Design, LLC.
 * All rights reserved.
 * 
 * Subject to the following obligations and disclaimer of warranty,
 * use and redistribution of this software, in source or object code
 * forms, with or without modifications are expressly permitted by
 * Packet Design; provided, however, that:
 * 
 *    (i)  Any and all reproductions of the source or object code
 *         must include the copyright notice above and the following
 *         disclaimer of warranties; and
 *    (ii) No rights are granted, in any manner or form, to use
 *         Packet Design trademarks, including the mark "PACKET DESIGN"
 *         on advertising, endorsements, or otherwise except as such
 *         appears in the above copyright notice or in the software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED BY PACKET DESIGN "AS IS", AND
 * TO THE MAXIMUM EXTENT PERMITTED BY LAW, PACKET DESIGN MAKES NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, REGARDING
 * THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY AND ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT.  PACKET DESIGN DOES NOT WARRANT, GUARANTEE,
 * OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF, OR THE RESULTS
 * OF THE USE OF THIS SOFTWARE IN TERMS OF ITS CORRECTNESS, ACCURACY,
 * RELIABILITY OR OTHERWISE.  IN NO EVENT SHALL PACKET DESIGN BE
 * LIABLE FOR ANY DAMAGES RESULTING FROM OR ARISING OUT OF ANY USE
 * OF THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE, OR CONSEQUENTIAL
 * DAMAGES, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, LOSS OF
 * USE, DATA OR PROFITS, HOWEVER CAUSED AND UNDER ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF PACKET DESIGN IS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Archie Cobbs <archie@packetdesign.com>
 */

#include "lws_global.h"
#include "lws_tmpl.h"
#include "lws_tmpl_http.h"

/***********************************************************************
			HTTP TMPL FUNCTIONS
***********************************************************************/

/* Our user-defined template functions */
static tmpl_handler_t	lws_tf_http_path;
static tmpl_handler_t	lws_tf_query_count;
static tmpl_handler_t	lws_tf_query_name;
static tmpl_handler_t	lws_tf_query_value;
static tmpl_handler_t	lws_tf_http_remote_ip;
static tmpl_handler_t	lws_tf_http_remote_port;

#define SERVLET_TMPL_FUNC(name, min, max, params, desc)		\
	LWS_TMPL_FUNC2(name, http_servlet_tmpl_func_, min, max, params, desc)

const struct lws_tmpl_func lws_tmpl_http_functions[] = {
    SERVLET_TMPL_FUNC(query, 1, 1, "fieldname",
	"Returns the value of the HTML GET or POST form field $1, or the"
"\n"	"empty string if no such field was submitted."),
    SERVLET_TMPL_FUNC(query_exists, 1, 1, "fieldname",
	"Returns \"1\" if an HTML GET or POST form field named $1 was"
"\n"	"submitted, otherwise \"0\"."),
    SERVLET_TMPL_FUNC(query_string, 0, 0, "",
	"Returns the query string from the HTTP request."),
    LWS_TMPL_FUNC(query_count, 0, 0, "",
	"Returns the number of form fields submitted by the browser."),
    LWS_TMPL_FUNC(http_path, 0, 0, "",
	"Returns the path component of the requested URL."),
    LWS_TMPL_FUNC(query_name, 1, 1, "index",
	"Returns the name of the $1'th field submitted by the browser."),
    LWS_TMPL_FUNC(query_value, 1, 1, "index",
	"Returns the value of the $1'th field submitted by the browser."),
    SERVLET_TMPL_FUNC(get_header, 1, 1, "name",
	"Returns the value of the HTTP request header named $1."),
    SERVLET_TMPL_FUNC(set_header, 2, 2, "name:value",
	"Sets the value of the HTTP response header named $1 to $2."),
    SERVLET_TMPL_FUNC(remove_header, 1, 1, "name",
	"Removes the HTTP response header named $1."),
    SERVLET_TMPL_FUNC(redirect, 1, 1, "url",
	"Sends back an HTTP redirect response, redirecting the requesting"
"\n"	"browser to the URL $1."),
    SERVLET_TMPL_FUNC(unbuffer, 0, 0, "",
	"Unbuffers the HTTP response body. This causes the HTTP response"
"\n"	"headers to be sent and means the response body does not have to be"
"\n"	"all gathered up in memory being sent to the browser."),
    LWS_TMPL_FUNC(http_remote_ip, 0, 0, "",
	"Returns the IP address of the requesting HTTP client."),
    LWS_TMPL_FUNC(http_remote_port, 0, 0, "",
	"Returns the TCP port of the requesting HTTP client."),
    { { NULL } }
};

/************************************************************************
			    HTTP STUFF
************************************************************************/

/* 
 * Get field count.
 *
 * Usage: @query_count()
 */
static char *
lws_tf_query_count(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);
	char buf[32];

	snprintf(buf, sizeof(buf),
	    "%d", http_request_get_num_values(targ->req));
	return (STRDUP(mtype, buf));
}

/* 
 * Get field name by index.
 *
 * Usage: @query_name(index)
 */
static char *
lws_tf_query_name(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);
	const char *name;

	if (http_request_get_value_by_index(targ->req,
	    atoi(av[1]), &name, NULL) == -1)
		return (NULL);
	return (STRDUP(mtype, name));
}

/* 
 * Get field value by index.
 *
 * Usage: @query_value(index)
 */
static char *
lws_tf_query_value(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);
	const char *value;

	if (http_request_get_value_by_index(targ->req,
	    atoi(av[1]), NULL, &value) == -1)
		return (NULL);
	return (STRDUP(mtype, value));
}

/* 
 * Get peer's remote IP address.
 *
 * Usage: @http_remote_ip()
 */
static char *
lws_tf_http_remote_ip(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);

	return (STRDUP(mtype,
	    inet_ntoa(http_request_get_remote_ip(targ->req))));
}

/* 
 * Get peer's remote TCP port.
 *
 * Usage: @http_remote_port()
 */
static char *
lws_tf_http_remote_port(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);
	char buf[16];

	snprintf(buf, sizeof(buf),
	    "%u", http_request_get_remote_port(targ->req));
	return (STRDUP(mtype, buf));
}

/* 
 * Get path.
 *
 * Usage: @http_path()
 */
static char *
lws_tf_http_path(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const path = http_request_get_path(targ->req);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);

	return (STRDUP(mtype, path));
}

