
/*
 * Copyright (c) 2002 Packet Design, LLC.
 * All rights reserved.
 * 
 * Subject to the following obligations and disclaimer of warranty,
 * use and redistribution of this software, in source or object code
 * forms, with or without modifications are expressly permitted by
 * Packet Design; provided, however, that:
 * 
 *    (i)  Any and all reproductions of the source or object code
 *         must include the copyright notice above and the following
 *         disclaimer of warranties; and
 *    (ii) No rights are granted, in any manner or form, to use
 *         Packet Design trademarks, including the mark "PACKET DESIGN"
 *         on advertising, endorsements, or otherwise except as such
 *         appears in the above copyright notice or in the software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED BY PACKET DESIGN "AS IS", AND
 * TO THE MAXIMUM EXTENT PERMITTED BY LAW, PACKET DESIGN MAKES NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, REGARDING
 * THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY AND ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT.  PACKET DESIGN DOES NOT WARRANT, GUARANTEE,
 * OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF, OR THE RESULTS
 * OF THE USE OF THIS SOFTWARE IN TERMS OF ITS CORRECTNESS, ACCURACY,
 * RELIABILITY OR OTHERWISE.  IN NO EVENT SHALL PACKET DESIGN BE
 * LIABLE FOR ANY DAMAGES RESULTING FROM OR ARISING OUT OF ANY USE
 * OF THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE, OR CONSEQUENTIAL
 * DAMAGES, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, LOSS OF
 * USE, DATA OR PROFITS, HOWEVER CAUSED AND UNDER ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF PACKET DESIGN IS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Archie Cobbs <archie@packetdesign.com>
 */

#include "lws_global.h"
#include "lws_config.h"
#include "lws_tmpl.h"
#include "lws_tmpl_config.h"

/***********************************************************************
			CONFIG OBJECT
***********************************************************************/

static tinfo_init_t	lws_config_object_init;

struct tinfo lws_tmpl_config_tinfo = TINFO_INIT(&lws_config_type,
	"tmpl_config", lws_config_object_init);

/*
 * Initialize this thread's configuration object.
 */
static int
lws_config_object_init(struct tinfo *t, void *data)
{
	void *config;

	if ((config = app_config_get(lws_config_ctx, 1)) == NULL)
		return (-1);
	if (structs_get(t->type, NULL, config, data) == -1) {
		app_config_free(lws_config_ctx, &config);
		return (-1);
	}
	app_config_free(lws_config_ctx, &config);
	return (0);
}

/***********************************************************************
			CONFIG TMPL FUNCTIONS
***********************************************************************/

/* Our user-defined template functions */
static tmpl_handler_t	lws_tf_config_apply;
static tmpl_handler_t	lws_tf_config_server;
static tmpl_handler_t	lws_tf_config_vhost;
static tmpl_handler_t	lws_tf_config_servlet;

/* User-defined template function descriptor list */
const struct lws_tmpl_func lws_tmpl_config_functions[] = {
    LWS_TMPL_FUNC(config_apply, 1, 1, "delay",
	"Make the current configuration object the new configuration after"
"\n"	"a delay of $1 seconds. Returns the empty string if successful,"
"\n"	"otherwise an error message is returned."),
    LWS_TMPL_FUNC(config_server, 0, 0, "",
	"Returns the config index of the server that the currently executing"
"\n"	"servlet is running in."),
    LWS_TMPL_FUNC(config_vhost, 0, 0, "",
	"Returns the config index of the virtual host that the currently"
"\n"	"executing servlet is running in."),
    LWS_TMPL_FUNC(config_servlet, 0, 0, "",
	"Returns the config index of the currently executing servlet."),
	{ { NULL } }
};

/*
 * Apply this thread's configuration object.
 */
static char *
lws_tf_config_apply(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	const char *const mtype = tmpl_ctx_get_mtype(ctx);
	const void *config;
	char ebuf[512];

	/* Get config structure */
	if ((config = tinfo_get(&lws_tmpl_config_tinfo)) == NULL)
		return (NULL);

	/* Apply it */
	if (app_config_set(lws_config_ctx, config,
	      atoi(av[1]), ebuf, sizeof(ebuf)) == -1) {
		if (errno != EINVAL)
			return (NULL);
		return (STRDUP(mtype, ebuf));
	}

	/* Done */
	return (STRDUP(mtype, ""));
}

/*
 * Get servlet's server index.
 */
static char *
lws_tf_config_server(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);
	const struct lws_tmpl_info *const tfi = targ->arg;
	char buf[32];

	snprintf(buf, sizeof(buf), "%d", tfi->sidx);
	return (STRDUP(mtype, buf));
}

/*
 * Get servlet's virtual host index.
 */
static char *
lws_tf_config_vhost(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);
	const struct lws_tmpl_info *const tfi = targ->arg;
	char buf[32];

	snprintf(buf, sizeof(buf), "%d", tfi->vidx);
	return (STRDUP(mtype, buf));
}

/*
 * Get servlet's servlet index.
 */
static char *
lws_tf_config_servlet(struct tmpl_ctx *ctx, char **errmsgp, int ac, char **av)
{
	struct http_servlet_tmpl_arg *const targ = tmpl_ctx_get_arg(ctx);
	const char *const mtype = tmpl_ctx_get_mtype(ctx);
	const struct lws_tmpl_info *const tfi = targ->arg;
	char buf[32];

	snprintf(buf, sizeof(buf), "%d", tfi->lidx);
	return (STRDUP(mtype, buf));
}

