/* checkpred.c
 *
 * Given an H-representation with redund/redund_list and project/eliminate
 * options, this program outputs a logical formula satisfiable iff the first
 * inequality is NON-redundant when projecting/eliminating as specified.
 * The formula is in SMT-LIB standard 2.6, with quantifiers using logic LRA
 * (closed linear formulas in linear real arithmetic), suitable for use with
 * Z3 or other solvers.  Also, generates lrs certificates (1&2) to verify
 * witnesses of redundancy; cert 1 should be feasible, cert 2 infeasible.
 * Define USEEXISTS to use an inner existential quantifier to not get the
 * projected variables as constants (so not in the witness for non-redundancy).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define VERSION "20210309"

#ifdef LRSLONG
#define ARITH "lrslong.h"    /* lrs long integer arithmetic package */
#else
#if defined(GMP) || defined(FLINT)
#define ARITH "lrsgmp.h"     /* lrs wrapper for gmp multiple precsion arithmetic    */
#else
#define ARITH "lrsmp.h"      /* lrs multiple precsion arithmetic    */
#endif
#endif

#include ARITH

#define printusage() \
  printf("Usage: %s <file.ine>\n" \
	 "To check redundancy of the first inequality in redund_list or redund when eliminating/projecting as specified by the eliminate/project option.\n\n%s -c <1/2> <file.ine>\nTo generate lrs input file <1/2> checking a witness (given on stdin) of non-redundancy generated by z3\n", \
	 argv[0],argv[0]);

int debug = 0;
FILE *lrs_ifp; /* hook to make readrat work */
FILE *lrs_ofp; /* hook to be able to link to lrsgmp/etc */
int mode = 0; /* 0: normal, 1: generate certificate 1, 2: generate certificate 2 */


struct mat {
	lrs_mp_matrix num, den;
	unsigned int m, n;	/* m inequalities, n variables x1...xn */
	unsigned int nlinearity; /* number of linearities */
	unsigned int *linearities; /* indices here are equations */
	unsigned int *linmask; /* Boolean flags for equations */
	unsigned int *varmask; /* Boolean flags for variables to eliminate */
	};

struct mat *parse_input(FILE *, unsigned int *, unsigned long **,
			unsigned long *);
void print_formula(struct mat*, unsigned int, unsigned long *, unsigned long, const char *, const int, char **);
void free_mat(struct mat*);
void print_cert(struct mat*, unsigned int, unsigned long *, unsigned long, const char *, const int, char **);

int main(int argc, char **argv)
{
	FILE *f;
	struct mat *mat;
	unsigned long *vars=NULL;
	unsigned long ineq=0; /* ineq ineq when projecting out vars */
	unsigned int i, nv=0, flag=0;
	lrs_ofp = stdout;

	if (argc != 2 && argc != 4)
	{
		printusage();
		return 0;
	}

	if (!strcmp(argv[1], "-c"))
	{
		mode = atoi(argv[2]);
		if (mode<0 || mode>2)
		{
			printusage();
			return 0;
		}
		argv+=2; /* -c <int> */
		argc-=2;
	}
	f = fopen(argv[1], "r");
	if (f==NULL)
	{
		printusage();
		return 0;
	}

	mat = parse_input(f, &nv, &vars, &ineq);
	fclose(f);

	if (mat == NULL)
	{
		fprintf(stderr,"Error parsing input\n");
		return 0;
	}

	for (i=0; i<nv; i++)
		if (vars[i]<1 || vars[i] >= mat->n)
			flag = 1;
	if (flag || ineq<1 || ineq>mat->m)
	{
		fprintf(stderr,"Bounds error: variables 1...%d, inequalities 1...%d\n", mat->n, mat->m);
		free_mat(mat);
		free(vars);
		return 0;
	}

	mat->varmask = calloc(mat->n+1, sizeof(unsigned int));

	fprintf(stderr, "*Mode %d checking non-redundancy of inequality %lu after eliminating variables", mode, ineq);
	for (i=0; i<nv; i++)
	{
		mat->varmask[vars[i]] = 1;
		fprintf(stderr, " %lu", vars[i]);
	}
	fprintf(stderr, " of %s\n", argv[1]);

	if (mode == 0)
		print_formula(mat, nv, vars, ineq, argv[1], argc, argv);
	else /* print a certificate */
		print_cert(mat, nv, vars, ineq, argv[1], argc, argv);
	free_mat(mat);
	free(vars);

	return 0;
}

void parse_error(struct mat *mat, const char *s)
{
	fprintf(stderr, "Parse error: %s\n", s);
	if (mat == NULL)
		return;
	if (mat->num != NULL)
		lrs_clear_mp_matrix(mat->num,mat->m,mat->n);
	if (mat->den != NULL)
		lrs_clear_mp_matrix(mat->den,mat->m,mat->n);
	free(mat);
	exit(0);
}

void eatspace(FILE *f)
{
	int c;
	while (isspace(c=getc(f)));
	ungetc(c,f);
}

/* just read name (either "redund" or "redund_list"
 * handle the line, put the first ineq in ineq
 */
void handle_redund(FILE *f, char *name, struct mat *mat, unsigned long *ineq)
{
	unsigned int i, j, tmp;
	int ret;
	eatspace(f);
	if (!strcmp("redund", name))
	{
		*ineq = 1;
		return;
	}
	/* redund_list */
	ret = fscanf(f, "%u", &j);
	if (ret!=1 || j==0)
		parse_error(mat, "broken redund_list line");
	for (i=0; i<j; i++)
	{
		eatspace(f);
		ret = fscanf(f, "%u", &tmp);
		if (ret!=1)
			parse_error(mat, "broken redund_list line");
		if (i==0)
			*ineq = tmp;
	}
	eatspace(f);
	return;
}

/* just read name (either "project" or "eliminate"
 * handle the line, set nv, allocate and set vars
 * note: could be multiple lines, last over-writes earlier ones
 * so realloc as needed
 */
void handle_project(FILE *f, char *name, struct mat *mat, unsigned int* nv,
		    unsigned long **vars)
{
	unsigned int i, j, num;
	unsigned long *tmp;
	unsigned long *tmp2;
	int ret;

	eatspace(f);
	ret = fscanf(f, "%u", &num);
	if (ret!=1 || num==0)
		parse_error(mat, "broken project/eliminate line");
	tmp = malloc(sizeof(unsigned long) * num);
	for (i=0; i<num; i++)
	{
		eatspace(f);
		ret = fscanf(f, "%lu", tmp+i);
		if (ret!=1 || tmp[i]<1 || tmp[i]>mat->m)
			parse_error(mat, "invalid row index in project/eliminate");
	}
	eatspace(f);
	if (!strcmp(name, "eliminate"))
	{
		*nv = num;
		tmp2 = realloc(*vars, sizeof(unsigned long)*num);
		if (tmp2 == NULL)
			parse_error(mat, "memory allocation error");
		*vars = tmp2;
		for (i=0; i<num; i++)
			tmp2[i] = tmp[i];
		free(tmp);
		return;
	}
	/* project, so flip indices */
	tmp2 = calloc(mat->n, sizeof(unsigned long));
	for (i=0; i<num; i++)
		tmp2[tmp[i]] = 1;
	/* tmp2[i] is 1 if i is to be kept, 0 if to be eliminated */
	free(tmp);
	*nv = mat->n - 1 - num;
	tmp = realloc(*vars, sizeof(unsigned long)* *nv);
	if (tmp == NULL)
		parse_error(mat, "memory allocation error");
	*vars = tmp;
	j=0;
	for (i=1; i<mat->n; i++)
		if (tmp2[i] == 0)
			tmp[j++] = i;
	if (j!=*nv)
		parse_error(mat, "horribly broken");
	free(tmp2);	
	return;
}

struct mat *parse_input(FILE *f, unsigned int *nv, unsigned long **vars,
			unsigned long *ineq)
{
	struct mat *mat;
	char *p, name[4096]={0}; /* lrslib uses 1000 */
	unsigned int cur, i, j, k, offs=0;
	int ret, end=0, redund=0, elim=0;

	mat = calloc(1, sizeof(struct mat));	

	p = fgets(name, 4000, f);
	if (p == NULL)
		parse_error(mat, "file error");
	if (debug)
		fprintf(stderr, "%s", name);
	while (strncmp(name,"begin",5))
	{
		if (!strncmp(name, "V-representation", 16))
			parse_error(mat,"V-representation not supported");
		if (!strncmp(name, "linearity ", 10))
		{
			/* read linearity option */
			p = name + 10;
			ret = sscanf(name+10, "%u%n", &k, &cur);
			if (ret == 0)
				parse_error(mat,"linearity error");
			mat->nlinearity = k;
			mat->linearities = calloc(k, sizeof(unsigned int));
			for (offs=cur, i=0; i<k; i++)
			{
				ret = sscanf(name+10+offs, "%u%n", 
					     &mat->linearities[i], &cur);
				if (ret == 0)
					parse_error(mat,"linearity error");
				offs += cur;
			}
			if (debug)
			{
				fprintf(stderr, "linearities:");
				for (i=0; i<mat->nlinearity; i++)
					fprintf(stderr, " %u", 
						mat->linearities[i]);
				fprintf(stderr, "\n");
			}
		} /* linearity option */
		fgets(name, 4000, f);
	}
	/* got begin */
	ret = fscanf(f, "%u %u", &mat->m, &mat->n);
	if (ret != 2)
		parse_error(mat,"missing or broken dimensions");
	mat->linmask = calloc(mat->m+1, sizeof(unsigned int));
	for (i=0; i<mat->nlinearity; i++)
		mat->linmask[mat->linearities[i]] = 1;
	fgets(name, 4000, f);
	for (i=0; isspace((int)name[i]); i++);
	if (strncmp(name+i, "integer", 7)!=0 && strncmp(name+i, "rational", 8)!=0)
		parse_error(mat,"data type must be integer or rational");

	mat->num = lrs_alloc_mp_matrix(mat->m, mat->n);
	mat->den = lrs_alloc_mp_matrix(mat->m, mat->n);

	lrs_ifp = f; /* hook to make readrat work */

	/* read the matrix */
	for (i=1; i<=mat->m; i++)
		for (j=0; j<mat->n; j++)
			readrat(mat->num[i][j], mat->den[i][j]);

	/* read options - required project or eliminate and redund or
	 * redund_list after end. they could be missing or too many
	 * numbers
	 */
	ret = 0;
	
	while (ret != EOF)
	{
		eatspace(f);
		ret = fscanf(f, "%s", name);
		if (ret != 1)
			break;
		else if (!strcmp("end", name))
		{
			end++;
			continue;
		}
		else if (!strcmp("redund_list", name)|| !strcmp("redund", name))
			redund++,handle_redund(f, name, mat, ineq);
		else if (!strcmp("eliminate", name) || !strcmp("project", name))
			elim++,handle_project(f, name, mat, nv, vars);
	}
	if (end!=1)
		parse_error(mat, "single end line required");
	if (redund == 0)
		parse_error(mat, "redund/redund_list line missing");
	if (elim == 0)
		parse_error(mat, "eliminate/project line missing");

	return mat;
}

/* 2020.8.21 do negation the annoying way to make MathSAT happy */
void print_prat(lrs_mp Nt, lrs_mp Dt)
{
	lrs_mp pNt, pDt;
	char *tmp1 = cpmp("", Nt), *tmp2=cpmp("", Dt);
	int neg1 = negative(Nt), neg2 = negative(Dt);
	int neg = (neg1 != neg2); /* Nt/Dt negative? */

	lrs_alloc_mp(pNt); lrs_alloc_mp(pDt);
	copy(pNt, Nt); copy(pDt, Dt);
	if (neg1)
		changesign(pNt);
	if (neg2)
		changesign(pDt);

	tmp1 = cpmp("", pNt);
	tmp2 = cpmp("", pDt);

	if (neg)
		printf("(- ");

	if (one(Dt))
		printf("%s", tmp1);
	else
		printf("(/ %s %s)", tmp1, tmp2);

	if (neg)
		printf(")"); /* close negation */
	free(tmp1); free(tmp2);
	lrs_clear_mp(pNt); lrs_clear_mp(pDt);
}

void print_header(unsigned long *vars, unsigned int nv, unsigned long ineq, const char *filename, const int argc, char **argv)
{
	unsigned int i;
	printf("; checkpred %s formula for checking non-redundancy of inequality %lu after eliminating variables", VERSION, ineq);
	for (i=0; i<nv; i++)
		printf(" %lu", vars[i]);
	printf(", using file %s\n", filename);
	printf("; invocation was:");
	for (i=0; i<argc; i++)
		printf(" %s", argv[i]);
	printf("\n");

	printf("(set-logic LRA)\n\n");
	return;
}

/* bool, is inequality i actually an equality? */
int islin(struct mat *mat, unsigned int i)
{
	return mat->linmask[i];
}

/* print inequality i, ax + by <= c, but done as 0 <= ...
 * invert a flag, print the <= if false, print > if true
 * also handle linearities.
 */
void printineq(struct mat *mat, unsigned int i, int invert)
{
	unsigned int j, n = mat->n;
	int lin = islin(mat, i);
	int one;

	printf("\n         (");
	if (invert == 0)
		printf("%s", (lin ? "=" : "<="));
	else /* inverting, so <= -> >, = -> noteq */
		printf("%s", (lin ? "not (=" : ">"));
	printf(" 0\n          (+ ");
	print_prat(mat->num[i][0], mat->den[i][0]);
	for (j=1; j<n; j++)
	{
		if (zero(mat->num[i][j]))
			continue; /* don't bother: 0 * x */
		one = 0;
		if (!mpz_cmp(mat->num[i][j], mat->den[i][j]))
			one = 1;
		
		if (!one) /* just do x_j instead of (* (/ a a) x_j) */
		{
			printf(" (* ");
			print_prat(mat->num[i][j], mat->den[i][j]);
		}
		printf(" x_%u", j);
		if (!one)
			printf(")");
	}
	if (invert && lin) /* printed "not (=, so close the extra paren here */
		printf(")");
	printf("))"); /* close + and <= */
}

/* print Ax + By <= C from diagram, but 
 * done as 0 <= ...
 * Omit inequality ineq per diagram
 */
void printsystem(struct mat *mat, unsigned long ineq)
{
	unsigned int i, m=mat->m;

	printf("      (and"); 
	for (i=1; i<=m; i++)
	{
		if (i == ineq)
			continue;
		printineq(mat, i, 0);
	}
	printf(")"); /* close and */
}

int projvar(struct mat *mat, unsigned int i)
{
	return mat->varmask[i];
}

/* rows give a_0 + a_1x_1 + a_2x_2 ... a_{n-1}x_{n_1} \ge 0 */
void print_formulabody(struct mat *mat, unsigned int nv, unsigned long *vars,
		       unsigned long ineq)
{
	unsigned int i, n=mat->n;

	/* declare constants x_1 ... x_{n-1} so (get-model) gives model */
	for (i=1; i<n; i++)
#ifdef USEEXISTS
		if (!projvar(mat, i)) /* omit variables projecting out */
#endif
			printf("(declare-const x_%u Real)\n", i);

	printf("\n(assert (and\n");

	/* \exists y st Ax + By \le C from diagram */
	/* in loop in case we can generalize to projecting out multiple vars */
#ifdef USEEXISTS
	printf("  (exists (");
	for (i=1; i<n; i++)
		if (projvar(mat, i))
			printf("(x_%u Real) ", i);
	printf(")");
#endif
	printsystem(mat, ineq);
#ifdef USEEXISTS
	printf("\n  )\n"); /* close exists */
#endif

	/* forall y : Ax+By\le C from diagram */
	printf("  (forall (");
	for (i=1; i<n; i++)
		if (projvar(mat, i))
			printf("(x_%u Real) ", i);
	printf(")");
	printf("\n    (=> ");
	printsystem(mat, ineq);

	/* implies  ax + by > c from diagram */
	printineq(mat, ineq, 1);
	printf("    )"); /* close => */
	printf("\n  )\n))"); /* close forall, and,assert */

	return;
}

void print_formula(struct mat *mat, unsigned int nv, unsigned long *vars,
		   unsigned long ineq, const char *filename, const int argc,
		   char **argv)
{
	print_header(vars, nv, ineq, filename, argc, argv);
	print_formulabody(mat, nv, vars, ineq);
	printf("\n(check-sat)\n(get-model)\n");
	return;
}

/* f is pointing at a floating point representation of a number,
 * read it into v[i]
 */
void readmp_float(FILE *f, lrs_mp_vector v, unsigned int i)
{
	char s[1024];
	char c;

	eatspace(f);

	fscanf(f, "%[0-9]", s);
	atomp(s, v[i]);

	eatspace(f);
	
	c = getc(f);
	if (c!='.')
	{
		if (isdigit(c)) /* cvc4 does not always add .0 to ints */
		{
			ungetc(c,f);
			c = ' ';
		}
		else if (c!=')') /* cvc4 does not always add .0 to ints */
		{
			printf("parse error 1 %c\n", c);
			exit(1);
		}
	}
	while (!isspace(c) && c!=')')
	{
		c = getc(f);
		if (isdigit(c) && c!='0')
		{
			printf("unexpected decimal part %c\n", c);
			exit(1);
		}
	}
}

/* f is pointing at a z3 representation of a number, 
 * e.g. 15.0 or (/ 3.0 7.0)
 * or (- a b) etc.
 * read this into num[i] / den[i]
 */
void readmp_z3(FILE *f, lrs_mp_vector num, lrs_mp_vector den, unsigned int i)
{
	char c;

	eatspace(f);
	c=getc(f);
	if (isdigit(c))
	{
		ungetc(c, f);
		/* now pointing at a floating point representation */
		readmp_float(f, num, i);
		itomp(ONE, den[i]);
		return;
	}
	/* now pointing at "/ a b)" */
	c = getc(f);
	if (c == '/')
	{
		readmp_float(f, num, i);
		readmp_float(f, den, i);
		c = getc(f);
		if (c != ')')
		{
			printf("parse error 2 %c\n", c);
			exit(1);
		}
	}
	else if (c == '-')
	{
		readmp_z3(f, num, den, i);
		changesign(num[i]);
		c = getc(f);
		if (c != ')')
		{
			printf("parse error 3 %c\n", c);
			exit(1);
		}
	}
	else
	{
		printf("unknown character %c\n", c);
		exit(1);
	}
	
}

/* read z3 output from stdin, put it in num/den
 * notes: x_i read to index i, index 0 unused.
 *        z3 outputs reals, we read them in as naturals or rationals
 */
void parse_witness(lrs_mp_vector num, lrs_mp_vector den, unsigned int n)
{
	FILE *f = stdin;
	unsigned int i;
	unsigned int index=0;

	lrs_ifp = f; /* hook to make readmp work */

	eatspace(f); fscanf(f, "sat"); eatspace(f); fscanf(f, "(model");
	for (i=1; i<n; i++)
	{
		eatspace(f);
		fscanf(f, "(define-fun x_%u () Real", &index);
		readmp_z3(f, num, den, index);
	}
	
}

/* print the linearity line if needed */
void cert_print_linearity(struct mat *mat, unsigned int nv, unsigned long ineq)
{
	unsigned int i;
	unsigned int bound=0; /* set for warning removal */
	unsigned int nlin;
	unsigned int lin;

	if (mode == 1)
		bound = mat->n-1;
	if (mode == 2)
		bound = mat->n-1 - nv;

	nlin = mat->nlinearity + bound;

	/* if we're checking a linearity, it gets omitted in certificate 1 */
	if (mode == 1 && islin(mat, ineq))
		nlin--;

	printf("linearity %u", nlin);

	for (i=0; i<mat->nlinearity; i++)
	{
		if (mode==1 && mat->linearities[i] == ineq)
			continue;
		lin = mat->linearities[i];
		/* mode 1 removes ineq, so later linearities shift down */
		if (mode==1 && lin>ineq)
			lin--;
		printf(" %u", lin);
	}

	for (i=1; i<=bound; i++)
		printf(" %u", mat->m + (mode==1?i-1:i));

	printf("\n");
}

/* print the header for the lrs certificate, which may have linearities */
void cert_print_header(struct mat *mat, unsigned int nv, const char *filename,
		       const int argc, char **argv, unsigned long ineq)
{
	printf("certificate %d for %s\nH-representation\n", mode, filename);
	cert_print_linearity(mat, nv, ineq);
	printf("begin\n");
	if (mode == 1)
		printf("%u", mat->m -1 + mat->n-1);
	else /* mode == 2 */
		printf("%u", mat->m + mat->n-1 - nv);
	printf(" %u rational\n", mat->n);
}

/* print the original system: note omit the inequality if mode==1 */
void lrs_print_system(struct mat *mat, unsigned long ineq)
{
	unsigned int i, j;

	for (i=1; i<=mat->m; i++)
	{
		if (i == ineq && mode == 1) /* omit this one for mode==1 */
			continue;
		for (j=0; j<mat->n; j++)
			prat("", mat->num[i][j], mat->den[i][j]);
		printf("\n");
	}
}

/* print the witness part */
void lrs_print_witness(struct mat *mat, lrs_mp_vector wn, lrs_mp_vector wd)
{
	unsigned int i, j;

	for (i=1; i<mat->n; i++)
	{
		if (mode == 2 && mat->varmask[i]) /* don't include y */
			continue;
		prat("", wn[i], wd[i]);
		for (j=1; j<mat->n; j++)
			printf(" %s", (j==i? "-1" : "0"));
		printf("\n");
	}
}

void print_cert(struct mat *mat, unsigned int nv, unsigned long *vars,
                   unsigned long ineq, const char *filename, const int argc,
                   char **argv)
{
	lrs_mp_vector witness_n = lrs_alloc_mp_vector(mat->n);
	lrs_mp_vector witness_d = lrs_alloc_mp_vector(mat->n);
	parse_witness(witness_n, witness_d, mat->n);

	/* print new header */
	cert_print_header(mat, nv, filename, argc, argv, ineq);
	/* print original input, omitting one inequality if mode==1 */
	lrs_print_system(mat, ineq);
	/* print new part */
	lrs_print_witness(mat, witness_n, witness_d);

	printf("end\nlponly\n");

	lrs_clear_mp_vector(witness_n,mat->n);
	lrs_clear_mp_vector(witness_d,mat->n);
	return;
}

void free_mat(struct mat *mat)
{
	lrs_clear_mp_matrix(mat->num, mat->m, mat->n);
	lrs_clear_mp_matrix(mat->den, mat->m, mat->n);
	free(mat->linearities);
	free(mat->linmask);
	free(mat->varmask);
	free(mat);
	return;
}
