/* lsysinfo:  Prints out various pieces of system information.

   Copyright (C) 2001-2021 by Brian E. Lindholm.
   This file is part of the littleutils utility set.

   The lsysinfo utility is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option) any
   later version.

   The lsysinfo utility is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   the littleutils.  If not, see <https://www.gnu.org/licenses/>. */


#include <config.h>

#include <stdio.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#if defined(HAVE_SYS_SYSINFO_H)
#include <sys/sysinfo.h>

#if defined(__CYGWIN__)
#define SYSINFO_SHIFT 8
#else
#define SYSINFO_SHIFT 20
#endif

int
main (int argc, char *argv[])
{
  struct sysinfo info;
  int rc;

  rc = sysinfo (&info);
  if (rc == 0)
    {
      fprintf (stdout, "sysinfo information:\n");
      fprintf (stdout, "  total RAM  = %5ld MiB\n", info.totalram >> SYSINFO_SHIFT);
      fprintf (stdout, "  free RAM   = %5ld MiB\n", info.freeram >> SYSINFO_SHIFT);
#if !defined(__CYGWIN__)
      fprintf (stdout, "  shared RAM = %5ld MiB\n", info.sharedram >> SYSINFO_SHIFT);
      fprintf (stdout, "  buffer RAM = %5ld MiB\n", info.bufferram >> SYSINFO_SHIFT);
#endif
      fprintf (stdout, "  total SWAP = %5ld MiB\n", info.totalswap >> SYSINFO_SHIFT);
      fprintf (stdout, "  free SWAP  = %5ld MiB\n", info.freeswap >> SYSINFO_SHIFT);
      fprintf (stdout, "  uptime = %ld seconds\n", info.uptime);
      fprintf (stdout, "  processes = %d\n", info.procs);
      fprintf (stdout, "  loadavg = %.3f, %.3f, %.3f\n", info.loads[0] / 65536.0,
        info.loads[1] / 65536.0, info.loads[2] / 65536.0);
    }
  else
    fprintf (stderr, "sysinfo error: bad sysinfo call\n");

  fprintf (stdout, "sysconf information:\n");
  fprintf (stdout, "  MEMORY = %ld MiB\n", (sysconf(_SC_PAGE_SIZE) >> 10) * (sysconf(_SC_PHYS_PAGES) >> 10));
  fprintf (stdout, "  PAGE_SIZE = %ld bytes\n", sysconf(_SC_PAGE_SIZE));
  fprintf (stdout, "  AVAIL_PHYS_PAGES = %ld\n", sysconf(_SC_AVPHYS_PAGES));
  fprintf (stdout, "  TOTAL_PHYS_PAGES = %ld\n", sysconf(_SC_PHYS_PAGES));
  fprintf (stdout, "  NUM_OF_PROCESSORS_CONF = %lu\n", sysconf(_SC_NPROCESSORS_CONF));
  fprintf (stdout, "  NUM_OF_PROCESSORS_ONLN = %lu\n", sysconf(_SC_NPROCESSORS_ONLN));
  fprintf (stdout, "  CLK_TCK = %lu\n", sysconf(_SC_CLK_TCK));
  fprintf (stdout, "  OPEN_MAX = %lu\n", sysconf(_SC_OPEN_MAX));

  return (0);
}

#elif defined(sun)
#include <sys/systeminfo.h>

int
main (int argc, char *argv[])
{
  char buf[1024];
  int count = 1024;

  fprintf (stdout, "sysinfo information:\n");
  sysinfo (SI_SYSNAME, buf, count);
  fprintf (stdout, "  SYSNAME = %s\n", buf);
  sysinfo (SI_HOSTNAME, buf, count);
  fprintf (stdout, "  HOSTNAME = %s\n", buf);
  sysinfo (SI_RELEASE, buf, count);
  fprintf (stdout, "  RELEASE = %s\n", buf);
  sysinfo (SI_VERSION, buf, count);
  fprintf (stdout, "  VERSION = %s\n", buf);
  sysinfo (SI_MACHINE, buf, count);
  fprintf (stdout, "  MACHINE = %s\n", buf);
  sysinfo (SI_ARCHITECTURE, buf, count);
  fprintf (stdout, "  ARCHITECTURE = %s\n", buf);
  sysinfo (SI_ISALIST, buf, count);
  fprintf (stdout, "  ISALIST = %s\n", buf);
  sysinfo (SI_PLATFORM, buf, count);
  fprintf (stdout, "  PLATFORM = %s\n", buf);
  sysinfo (SI_HW_PROVIDER, buf, count);
  fprintf (stdout, "  HW_PROVIDER = %s\n", buf);
  sysinfo (SI_HW_SERIAL, buf, count);
  fprintf (stdout, "  HW_SERIAL = %s\n", buf);

  fprintf (stdout, "sysconf information:\n");
  fprintf (stdout, "  NUM_OF_PROCESSORS_CONF = %lu\n", sysconf(_SC_NPROCESSORS_CONF));
  fprintf (stdout, "  NUM_OF_PROCESSORS_ONLN = %lu\n", sysconf(_SC_NPROCESSORS_ONLN));
  fprintf (stdout, "  PAGE_SIZE = %ld bytes\n", sysconf(_SC_PAGE_SIZE));
  fprintf (stdout, "  PHYS_PAGES = %ld\n", sysconf(_SC_PHYS_PAGES));
  fprintf (stdout, "  MEMORY = %ld MiB\n", (sysconf(_SC_PAGE_SIZE) >> 10) * (sysconf(_SC_PHYS_PAGES) >> 10));

  return (0);
}

#elif defined(DJGPP)

int
main (int argc, char *argv[])
{
  fprintf (stdout, "sysconf information:\n");
  fprintf (stdout, "  CLK_TCK = %lu\n", sysconf(_SC_CLK_TCK));
  fprintf (stdout, "  OPEN_MAX = %lu\n", sysconf(_SC_OPEN_MAX));
  return (0);
}

#elif defined(__MINGW__)

int
main (int argc, char *argv[])
{
  fprintf (stdout, "No sysinfo information available for MINGW\n");
  return (0);
}

#else

int
main (int argc, char *argv[])
{
  fprintf (stdout, "No sysinfo information available for unknown system\n");
  return (0);
}

#endif
