/* filehash:  Print various hash digests and filesizes for specified files.

   Copyright (C) 2004-2020 by Brian Lindholm.
   This file is part of the littleutils utility set.

   The filehash utility is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 3, or (at your option) any later
   version.

   The filehash utility is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   the littleutils.  If not, see <https://www.gnu.org/licenses/>. */


#include <config.h>

#include <limits.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif

#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
# define OPTEND -1
#else
# define OPTEND EOF
#endif

#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif

#ifdef MSDOS
# include "io.h"
#endif

#include "md5.h"
#include "sha1.h"
#include "sha256.h"
#include "sha512.h"
#include "b2sum.h"

#ifdef HAVE_FSEEKO
# define fseek(a,b,c) fseeko((a),(off_t)(b),(c))
#endif

#ifdef __MINGW32__
extern int getopt (int argc, char * const *argv, const char *optstring);
extern char *optarg;
extern int optind;
#endif

#ifdef DJGPP
unsigned short _djstat_flags = 63;
#endif

#ifndef PATH_MAX
# define PATH_MAX 256
#endif

char *sizefmt = (sizeof (off_t) <= sizeof (long) ? "%lu" : "%llu");


static void
help (FILE *where)
{
  fprintf (where,
    "filehash " PACKAGE_VERSION "\n"
    "usage: filehash [ -1(MD5) ] [ -2(SHA1) ] [ -3(SHA224) ] [ -4(SHA256) ]\n"
    "         [ -5(SHA384) ] [ -6(SHA512) ] [ -7(BLAKE2B_256)] [ -8(BLAKE2B_256)]\n"
    "         [-c(lassic)] [ -f file_list ] [ -h(elp) ] [ -n byte_count ]\n"
    "         [ -p(ipe) ] [ -q(uiet) ] [ -s(ize) ] file...\n");
}


static void
print_filehash (char *filename, off_t read_bytes, int print_size, int run_md5,
  int run_sha1, int run_sha224, int run_sha256, int run_sha384, int run_sha512,
  int run_blake2b_256, int run_blake2b_512, int verbose, int classic)
{
  FILE *file;
  int i, not_first, rc;
  struct stat file_stats;
  unsigned char md5result[16], sha1result[20], sha224result[28],
                sha256result[32], sha384result[48], sha512result[64],
                blake2b_256result[32], blake2b_512result[64];

  if (stat (filename, &file_stats))
    fprintf (stderr, "filehash error: can't stat %s\n", filename);
  else if (((file_stats.st_mode & S_IFDIR) != S_IFDIR) &&
           ((file_stats.st_mode & S_IFREG) == S_IFREG))
    {
      if ((file = fopen (filename, "rb")) == NULL)
        fprintf (stderr, "filehash error: can't open %s\n", filename);
      else
        {
          not_first = 0;
          if (run_md5) {
            if (not_first)
              (void) fseek (file, (off_t) 0, 0);
            rc = md5_stream (file, md5result, read_bytes);
            if (rc)
              fprintf (stderr,
                "filehash error: md5_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha1) {
            if (not_first)
              (void) fseek (file, (off_t) 0, 0);
            rc = sha1_stream (file, sha1result, read_bytes);
            if (rc)
              fprintf (stderr,
                "filehash error: sha1_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha224) {
            if (not_first)
              (void) fseek (file, (off_t) 0, 0);
            rc = sha224_stream (file, sha224result, read_bytes);
            if (rc)
              fprintf (stderr,
                "filehash error: sha224_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha256) {
            if (not_first)
              (void) fseek (file, (off_t) 0, 0);
            rc = sha256_stream (file, sha256result, read_bytes);
            if (rc)
              fprintf (stderr,
                "filehash error: sha256_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha384) {
            if (not_first)
              (void) fseek (file, (off_t) 0, 0);
            rc = sha384_stream (file, sha384result, read_bytes);
            if (rc)
              fprintf (stderr,
                "filehash error: sha384_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha512) {
            if (not_first)
              (void) fseek (file, (off_t) 0, 0);
            rc = sha512_stream (file, sha512result, read_bytes);
            if (rc)
              fprintf (stderr,
                "filehash error: sha512_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_blake2b_256) {
            if (not_first)
              (void) fseek (file, (off_t) 0, 0);
            rc = blake2b_stream (file, blake2b_256result, 32, read_bytes);
            if (rc)
              fprintf (stderr,
                "filehash error: sha384_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_blake2b_512) {
            if (not_first)
              (void) fseek (file, (off_t) 0, 0);
            rc = blake2b_stream (file, blake2b_512result, 64, read_bytes);
            if (rc)
              fprintf (stderr,
                "filehash error: sha512_stream failed on %s\n", filename);
            not_first = 1;
          }
          (void) fclose (file);
          not_first = 0;
          if ((verbose == 1) && (classic == 0)) {
            fprintf (stdout, "%s", filename);
            not_first = 1;
          }
          if (print_size) {
            if (classic) {
              fprintf (stdout, sizefmt, file_stats.st_size);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              fprintf (stdout, sizefmt, file_stats.st_size);
              not_first = 1;
            }
          }
          if (run_md5) {
            if (classic) {
              for (i = 0; i < 16; ++i)
                fprintf (stdout, "%02x", (unsigned int) md5result[i]);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              for (i = 0; i < 16; ++i)
                fprintf (stdout, "%02x", (unsigned int) md5result[i]);
              not_first = 1;
            }
          }
          if (run_sha1) {
            if (classic) {
              for (i = 0; i < 20; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha1result[i]);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              for (i = 0; i < 20; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha1result[i]);
              not_first = 1;
            }
          }
          if (run_sha224) {
            if (classic) {
              for (i = 0; i < 28; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha224result[i]);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              for (i = 0; i < 28; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha224result[i]);
              not_first = 1;
            }
          }
          if (run_sha256) {
            if (classic) {
              for (i = 0; i < 32; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha256result[i]);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              for (i = 0; i < 32; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha256result[i]);
              not_first = 1;
            }
          }
          if (run_sha384) {
            if (classic) {
              for (i = 0; i < 48; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha384result[i]);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              for (i = 0; i < 48; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha384result[i]);
              not_first = 1;
            }
          }
          if (run_sha512) {
            if (classic) {
              for (i = 0; i < 64; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha512result[i]);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              for (i = 0; i < 64; ++i)
                fprintf (stdout, "%02x", (unsigned int) sha512result[i]);
            }
          }
          if (run_blake2b_256) {
            if (classic) {
              for (i = 0; i < 32; ++i)
                fprintf (stdout, "%02x", (unsigned int) blake2b_256result[i]);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              for (i = 0; i < 32; ++i)
                fprintf (stdout, "%02x", (unsigned int) blake2b_256result[i]);
            }
          }
          if (run_blake2b_512) {
            if (classic) {
              for (i = 0; i < 64; ++i)
                fprintf (stdout, "%02x", (unsigned int) blake2b_512result[i]);
              fprintf (stdout, "  %s\n", filename);
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              for (i = 0; i < 64; ++i)
                fprintf (stdout, "%02x", (unsigned int) blake2b_512result[i]);
            }
          }
          if (classic == 0)
            fprintf (stdout, "\n");
        }
    }
}


int
main (int argc, char **argv)
{
  FILE *infile;
  char filename[PATH_MAX], *listname, *newline, *rc;
  int argn, classic, opt, print_size, run_md5, run_sha1, run_sha224,
      run_sha256, run_sha384, run_sha512, run_blake2b_256, run_blake2b_512,
      use_file, use_pipe, verbose;
  off_t read_bytes, tmp_bytes;

  /* parse options */

  classic = 0;
  listname = "";
  print_size = 0;
  read_bytes = -1;
  run_md5 = 0;
  run_sha1 = 0;
  run_sha224 = 0;
  run_sha256 = 0;
  run_sha384 = 0;
  run_sha512 = 0;
  run_blake2b_256 = 0;
  run_blake2b_512 = 0;
  use_file = 0;
  use_pipe = 0;
  verbose = 0;
  while ((opt = getopt (argc, argv, "12345678cf:hpn:qsv")) != OPTEND)
    switch (opt)
      {
      case '1':
        run_md5 = 1;
        break;
      case '2':
        run_sha1 = 1;
        break;
      case '3':
        run_sha224 = 1;
        break;
      case '4':
        run_sha256 = 1;
        break;
      case '5':
        run_sha384 = 1;
        break;
      case '6':
        run_sha512 = 1;
        break;
      case '7':
        run_blake2b_256 = 1;
        break;
      case '8':
        run_blake2b_512 = 1;
        break;
      case 'c':
        classic = 1;
        break;
      case 'f':
        use_file = 1;
        listname = optarg;
        break;
      case 'h':
        help (stdout);
        return (0);
      case 'n':
        tmp_bytes = (sizeof (off_t) <= sizeof (long) ? atol(optarg) : atoll(optarg));
        if (tmp_bytes > 0)
          read_bytes = tmp_bytes;
        else
          read_bytes = 0;
        break;
      case 'p':
        use_pipe = 1;
        break;
      case 'q':
        verbose = -1;
        break;
      case 's':
        print_size = 1;
        break;
      case 'v':
        verbose = 1;
        break;
      case '?':
        help (stderr);
        return (1);
      }

  /* finalize options */

  if ((optind == argc) && (use_file == 0) && (use_pipe == 0))
    {
      help (stdout);
      return (0);
    }
  if (verbose == 0)
    {
      if (((argc - optind) != 1) || use_file || use_pipe)
        verbose = 1;
      else
        verbose = -1;
    }
  if ((run_md5 + run_sha1 + run_sha224 + run_sha256 + run_sha384 + run_sha512 +
    run_blake2b_256 + run_blake2b_512) == 0)
      run_sha256 = 1;

  /* process files in listed in file specified by -f option */

  if (use_file)
    {
      infile = fopen (listname, "r");
      if (infile == NULL)
        fprintf (stderr, "filehash error: can't open %s!\n", listname);
      else
        {
          while (!feof (infile))
            {
              rc = fgets (filename, PATH_MAX - 1, infile);
              if (rc != NULL)
                {
                  newline = strchr (filename, '\n');
                  if (newline != NULL)
                    *newline = '\0';
                  if (strlen (filename) != 0)
                    print_filehash (filename, read_bytes, print_size, run_md5,
                      run_sha1, run_sha224, run_sha256, run_sha384, run_sha512,
                      run_blake2b_256, run_blake2b_512, verbose, classic);
                }
            }
          (void) fclose (infile);
        }
    }

  /* process files listed on stdin (i.e., the -p option) */

  if (use_pipe)
    while (!feof (stdin))
      {
        rc = fgets (filename, PATH_MAX - 1, stdin);
        if (rc != NULL)
          {
            newline = strchr (filename, '\n');
            if (newline != NULL)
              *newline = '\0';
            if (strlen (filename) != 0)
              print_filehash (filename, read_bytes, print_size, run_md5,
                run_sha1, run_sha224, run_sha256, run_sha384, run_sha512,
                run_blake2b_256, run_blake2b_512, verbose, classic);
          }
      }

  /* process files given in the argument list */

  for (argn = optind; argn < argc; argn++)
    print_filehash (argv[argn], read_bytes, print_size, run_md5, run_sha1,
      run_sha224, run_sha256, run_sha384, run_sha512, run_blake2b_256,
      run_blake2b_512, verbose, classic);

  /* indicate successful finish */

  return (0);
}
