// pngrecolor.c - part of pngrecolor
// Copyright (C) 2010 by Jason Summers (from original pngrewrite.c)
// Modifications (C) 2013-2020 by Brian Lindholm
//   [1] Renamed from "pngrewrite" to "pngrecolor" at avoid confusion with
//       original program.
//   [2] Modified palette comparison routine to ensure a *unique* sorting.
//   [3] Added command-line options "-n" and "-q" for no-interlacing and quiet
//       output.
//   These modifications may be used without restriction.

#define PNGRW_PALSORT_BY_FREQUENCY

#if defined(_WIN32) && !defined(__GNUC__) && !defined(PNGRW_WINDOWS)
# define PNGRW_WINDOWS
#endif

#ifdef PNGRW_WINDOWS
# include <tchar.h>
#endif

#include <config.h>

#include <stdio.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#endif

#ifdef PNGRW_WINDOWS
# include <io.h>
# include <fcntl.h>
#endif

#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
# define OPTEND -1
#else
# define OPTEND EOF
#endif

#ifdef __MINGW32__
extern int getopt (int argc, char * const *argv, const char *optstring);
extern char *optarg;
extern int optind;
#endif

#include "libpngrecolor.h"

#ifndef PNGRW_WINDOWS
# define TCHAR char
# define _T
# define _ftprintf fprintf
# define _tcscmp   strcmp
# define _fileno   fileno
# define _tmain    main
# define _isatty   isatty
#endif

static void my_error_fn(struct pngrw_ctx *ctx, const PNGRW_CHAR *msg)
{
	_ftprintf(stderr,_T("%s\n"),msg);
}

static void my_print_fn(struct pngrw_ctx *ctx, const PNGRW_CHAR *msg)
{
	_ftprintf(stderr,_T("%s\n"),msg);
}

// For stdin/stdout, pass "-" as the filename.
static int run(const TCHAR *in_filename, const TCHAR *out_filename, const int no_interlace, const int verbose)
{
	struct pngrw_ctx *ctx = NULL;
	int retval=0;

	ctx = pngrw_create();
	if(!ctx) goto done;
	pngrw_set_print_fn(ctx,my_print_fn);
	pngrw_set_error_fn(ctx,my_error_fn);

#ifdef PNGRW_PALSORT_BY_FREQUENCY
	pngrw_set_sort_by_frequency(ctx,1);
#else
	pngrw_set_sort_by_frequency(ctx,0);
#endif

	// Prepare input file.
	if(!_tcscmp(in_filename,_T("-"))) {
#ifdef PNGRW_WINDOWS
		_setmode(_fileno(stdin), O_BINARY);
#endif
		if(!pngrw_read_stdio(ctx,stdin,verbose)) {
			goto done;
		}
	}
	else {
		if(!pngrw_read_filename(ctx,in_filename,verbose)) {
			goto done;
		}
	}

	if(!pngrw_optimize_image(ctx,verbose)) {
		goto done;
	}

	// Prepare output file.
	if(!_tcscmp(out_filename,_T("-"))) {
#ifdef PNGRW_WINDOWS
		_setmode(_fileno(stdout), O_BINARY);
#endif
		if(!pngrw_write_stdio(ctx,stdout,no_interlace)) {
			goto done;
		}
	}
	else {
		if(!pngrw_write_filename(ctx,out_filename,no_interlace)) {
			goto done;
		}
	}

	retval = 1;

done:
	if(ctx) pngrw_destroy(ctx);
	return retval;
}

static void help (FILE *where)
{
	fprintf (where, "pngrecolor " PACKAGE_VERSION "\n"
	"usage: pngrecolor [-h(elp)] [-n(o_interlace)] [-q(uiet)] infile outfile\n");
}

int _tmain(int argc, TCHAR **argv)
{
	TCHAR *out_filename;
	TCHAR *in_filename;
	int c, no_interlace, ret, verbose ;
	TCHAR specstamp[38]=_T("");

	/* Parse command-line options */
	verbose = 1;
	no_interlace = 0;
	while ((c = getopt (argc, argv, "hnq")) != OPTEND)
		switch (c) {
			case 'h':
				help (stdout);
				return (0);
			case 'n':
				no_interlace = 1;
				break;
			case 'q':
				verbose = 0;
				break;
			case '?':
				help (stderr);
				return (1);
		}

	/* Ensure that two files were given as input */
	if ((argc - optind) != 2) {
		_ftprintf (stderr, _T("pngrecolor error: missing input or output file%s\n"), specstamp);
		return 1;
	}
	else {
		in_filename = argv[optind];
		out_filename = argv[optind + 1];
	}

	ret = run(in_filename,out_filename,no_interlace,verbose);
	if(!ret) return 1;
	return 0;
}
