/* lrealpath:  Prints the real path of the specified files.  This is intended
   to resolve symbolic links on Unix machines.  Amazingly, it works on DOS
   machines too.

   Copyright (C) 2004-2013 by Brian Lindholm.
   This file is part of the littleutils utility set.

   The lrealpath utility is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option) any
   later version.

   The lrealpath utility is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   the littleutils.  If not, see <https://www.gnu.org/licenses/>. */


#include "../config.h"
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef DJGPP
#include <sys/stat.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#define OPTEND -1
#else
#define OPTEND EOF
#endif

#ifdef __MINGW32__
extern int getopt (int argc, char * const *argv, const char *optstring);
extern char *optarg;
extern int optind;
#endif

#ifndef PATH_MAX
#define PATH_MAX 256
#endif


/* help function */

static void
help (FILE *where)
{
  fprintf (where,
    "lrealpath " PACKAGE_VERSION "\n"
    "usage: lrealpath [-f file_list] [-h(elp)] [-p(ipe)] [-v(erbose)] file...\n");
}


/* print realpath function */

static void
printpath (char *filename, int verbose)
{
  char resolved[PATH_MAX], *rc;

#if defined(HAVE_REALPATH)
  rc = realpath (filename, resolved);
#elif defined(DJGPP)
  rc = _truename (filename, resolved);
#else
  strncpy (resolved, filename, PATH_MAX);
  rc = resolved;
#endif
  if (rc == NULL)
    fprintf (stderr, "lrealpath error: can't determine path of %s\n", filename);
  else if (verbose == 1)
    fprintf (stdout, "%s\t%s\n", filename, resolved);
  else
    fprintf (stdout, "%s\n", resolved);
}


/* main program */

int
main (int argc, char *argv[])
{
  FILE *infile;
  char filename[PATH_MAX], *listname, *newline, *rc;
  int argn, opt, use_file, use_pipe, verbose;

  /* parse options */

  listname = "";
  verbose = 0;
  use_file = 0;
  use_pipe = 0;
  while ((opt = getopt (argc, argv, "f:hpv")) != OPTEND)
    switch (opt)
      {
      case 'f':
        use_file = 1;
        listname = optarg;
        break;
      case 'h':
        help (stdout);
        return (0);
      case 'p':
        use_pipe = 1;
        break;
      case 'v':
        verbose = 1;
        break;
      case '?':
        help (stderr);
        return (1);
      }

  /* finalize options */

  if ((optind == argc) && (use_file == 0) && (use_pipe == 0))
    {
      help (stdout);
      return (0);
    }

  /* process files in listed in file specified by -f option */

  if (use_file)
    {
      infile = fopen (listname, "r");
      if (infile == NULL)
        fprintf (stderr, "lrealpath error: can't open %s!\n", listname);
      else
        {
          while (!feof (infile))
            {
              rc = fgets (filename, PATH_MAX - 1, infile);
              if (rc != NULL)
                {
                  newline = strchr (filename, '\n');
                  if (newline != NULL)
                    *newline = '\0';
                  if (strlen (filename) != 0)
                    printpath (filename, verbose);
                }
            }
          (void) fclose (infile);
        }
    }

  /* process files listed on stdin (i.e., the -p option) */

  if (use_pipe)
    while (!feof (stdin))
      {
        rc = fgets (filename, PATH_MAX - 1, stdin);
        if (rc != NULL)
          {
            newline = strchr (filename, '\n');
            if (newline != NULL)
              *newline = '\0';
            if (strlen (filename) != 0)
              printpath (filename, verbose);
          }
      }

  /* process files given in the argument list */

  for (argn = optind; argn < argc; argn++)
    printpath (argv[argn], verbose);

  /* indicate successful finish */

  return (0);
}
