#include <xdkwrl/scene.h>
#include "parser/vrml97lexer.hpp"
#include "parser/vrml97fastlexer.h"
#include "parser/vrml97parser.hpp"

#include <antlr/TokenStreamException.hpp>
#include <fstream>
#include <stdexcept>

using namespace std;
using namespace wrl;

//************************************************************
// Implementation of Scene
//************************************************************
/*! \class wrl::Scene
 * \ingroup basetypes
 * This class offers encapsulation of the parsing of a VRML file.  The
 * function load calls the parser and you can then get the parsed nodes and
 * parsed protos declarations as member of the scene.  Below is a typical
 * example of use.
 \code
  #include <xdkwrl/scene.h>
  #include <xdkwrl/tools/prettyprinter.h>
  #include <stdexcept>

  using namespace std;

  int
  main(int argc,char** argv)
  {
    try
    {
      wrl::Scene scene;
      scene.load(argv[1],true);
      wrl::PrettyPrinter p(cout);
      p<<scene;
    }
    catch (runtime_error e)
    {
      cerr <<e.what()<<endl;
    }    
  } 
 \endcode
 */
/*!
 * Constructs an empty scene.
 */
wrl::Scene::Scene()
{
}
/*! 
 * Load a scene from a file whose path is given by \p fileName. If the
 * verbose parameter \p v is \c true (default is false), then debugging
 * information is send on standard output. You would normally not use this
 * facility but it is present to allow debugging in case you are faced with
 * a file that the library does not parse.
 *
 * The parser and lexer are produced using ANTLR (a powerful successor to
 * lex and yacc) and the official VRML2 grammar.  Therefore, it parses every
 * file that meets the official grammar.  The price to get this confidence
 * is that the lexer is quite slow.  Hopefully, ANTLR has the wonderful
 * feature that you can extend the lexer and parser by deriving a new class.
 * A faster version of the lexer is proposed by the library.  This version
 * should be equivalent but it is harder to check that the optimisation
 * still matches exactly the standard.  If this optimized lexer fails on a
 * VRML file that seems valid to you, check if the "slow" lexer succeeds and
 * send a bug report.  If the \p fast parameter is set to true (default),
 * the optimized lexer is used.
 *
 * If the parsing fails, the function throws an \c std::runtime_error (defined
 * in \c stdexcept header). The \p what() member of this message indicates
 * where the error occured.
 * \todo use a custom error type that contains line and column.
 * \todo fix line and column number that seems bogus in error msg.
 */
void 
wrl::Scene::load(const char* fileName,bool v,bool fast)
{
  try
  {
    ifstream s(fileName);
    vrml97lexer* lexer = fast?new vrml97fastlexer(s):new vrml97lexer(s);    
    lexer->setFilename(fileName);
    vrml97parser parser(*lexer);
    if (fast)
    {
      vrml97fastlexer* fastLexer = dynamic_cast<vrml97fastlexer*>(lexer);
      fastLexer->setParserPointer(&parser);
    }
    parser.setFilename(fileName);
    parser.setVerbose(v);
    parser.parseScene();
    (void) parser.getParsedNodes(back_inserter(nodes));
    (void) parser.getParsedProtos(back_inserter(protos));
    (void) parser.getParsedExternProtos(back_inserter(externProtos));
    delete lexer;
  }
  catch (ANTLR_USE_NAMESPACE(antlr)RecognitionException& e)
  {
    throw runtime_error(e.toString());
  }
  catch (ANTLR_USE_NAMESPACE(antlr)TokenStreamException& e)
  {
    throw runtime_error(e.toString());
  }
}
/*!
 * \overload void  wrl::Scene::load(const char* fileName,bool v,bool fast)
 */
void 
wrl::Scene::load(const string& fileName,bool v,bool fast)
{
  load(fileName.c_str(),v,fast);
}
/*! \var wrl::Scene::nodes
 * Once the load function has been called, this member contains all root nodes
 * that has been parsed in the file.  If load() is called successively on
 * several files, this member is cleared each time so you cannot
 * "accumulate" files in a Scene.
 * \todo fix the deallocating of nodes that is completely unhandled at the
 * moment and causes memory leak.
 */
/*! \var wrl::Scene::protos
 *
 * Once the load function has been called, this member contains all proto
 * declarations that has been parsed in the file.  If load() is called
 * successively on several files, this member is cleared each time so you
 * cannot "accumulate" files in a Scene.
 *
 * This protos are just declarations. If they are instanciated in the file
 * then the nodes member contains somewhere in the hierarchy ProtoInstance
 * nodes. See \ref proto for more info on protos classes.xb
 */

// Local variables section.
// This is only used by emacs!
// Local Variables:
// ff-search-directories: ("." "../../include/xdkwrl/")
// End:
