#include "./vrml97fastlexer.h"
#include "./vrml97parser.hpp"
#include <antlr/NoViableAltForCharException.hpp>
#include <iomanip>
#include <cstdlib>

using namespace std;

static
inline bool
isValidIdRestChars(char c)
{
  return ((c <= 0x20) ||
	  (c == 0x22) ||
	  (c == 0x23) ||
	  (c == 0x27) ||
	  (c == 0x2c) ||
	  (c == 0x2e) ||
	  (c == 0x5b) ||
	  (c == 0x5c) ||
	  (c == 0x5d) ||
	  (c == 0x7b) ||
	  (c == 0x7d) ||
	  (c == 0x7f))
    ?false:true;
}
static
inline bool
isValidIdFirstChar(char c)
{
  return (((c >= 0x30) && (c <= 0x39)) ||
	  (c == 0x2b) ||
	  (c == 0x2d) ||
	  !isValidIdRestChars(c))
    ? false:true;
}
static
inline bool
isWhitespaceChar(char c)
{
  return ((c == 0x0d) ||     // carriage return
	  (c == 0x0a) ||     // linefeed
	  (c == 0x20) ||     // space
	  (c == 0x09) ||     // tab
	  (c == 0x2c))       // comma
    ?true:false;
}
static
inline bool
isNewlineChar(char c)
{
  return ((c == 0x0a) || (c == 0x0d));
}
static 
inline bool
isHexDigit(char c)
{
  return (isdigit(c)|| 
	  (c == 'A') || (c == 'a') || 
	  (c == 'B') || (c == 'b') || 
	  (c == 'C') || (c == 'c') || 
	  (c == 'D') || (c == 'd') || 
	  (c == 'E') || (c == 'e') || 
	  (c == 'F') || (c == 'f')) 
    ?true:false;
}
static
inline bool
maybePartOfNumber(char c)
{
  return ((c == 'E') ||     
	  (c == 'e') ||     
	  (c == '.') ||     
	  (c == '+') ||     
	  (c == '-') ||     
	  (c == 'X') ||     
	  (c == 'x') ||     
	  (isdigit(c)) ||        
	  (isHexDigit(c)))       
    ?true:false;
}
//************************************************************
// Implementation of vrml97fastlexer
//************************************************************
/*! \class vrml97fastlexer
 * \ingroup parser
 */
vrml97fastlexer::vrml97fastlexer(istream& s)
  : vrml97lexer(s),
    istm_(s),
    cWasSpit_(false),
    parser_(NULL)
{
}
ANTLR_USE_NAMESPACE(antlr)RefToken
vrml97fastlexer::nextToken()
{
  using antlr::RefToken;
  using antlr::CommonToken;

  string   tokenString;
  RefToken token(new CommonToken);
  token->setLine(line_);
  token->setColumn(col_);
  // Gets a char from the stream
  swallowOneChar();  
  // Skip white space and comments
  while (isWhitespaceChar(c_) || (c_ == '#'))
  {
    if (c_ == '#')
    {
      string comment;
      while (!(isNewlineChar(c_) || c_ == EOF))
      {
	comment += c_;
	swallowOneChar();
      }
      if (comment.substr(0,10) == "#VRML V2.0")
      {
	token->setType(HEADER);
	spitOneChar();
	return token;
      }
    }
    //
    // Increment the line count (and reset the column count to zero) if the
    // current character is a newline character EXCEPT if the current
    // character is a linefeed AND the previous character is a carriage
    // return.
    //  
    if (isNewlineChar(c_))
    {
      if (!((c_ == 0x0a) && (prevChar_ == 0x0d)))
      {
	++line_; // newline();
	col_ = 0;
      }
    }
    swallowOneChar();
  }
  ////////////////////////////////////////////////////////////
  // End of line
  ////////////////////////////////////////////////////////////  
  if (c_ == EOF)
  {
    token->setType(EOF_);
    return token;
  }
  ////////////////////////////////////////////////////////////
  // Id or Keyword
  ////////////////////////////////////////////////////////////  
  if (isValidIdFirstChar(c_))
  {
    while (isValidIdRestChars(c_))
    {
      tokenString += c_;
      swallowOneChar();
    }
    // Last read char (rejected by while test) has to be spit    
    spitOneChar();  
    token->setText(tokenString);
    // Check if it is not a keyword
    int t = Id;
    if      (tokenString == "DEF")          { t = LITERAL_DEF; }
    else if (tokenString == "eventIn")      { t = LITERAL_eventIn; }
    else if (tokenString == "eventOut")     { t = LITERAL_eventOut; }
    else if (tokenString == "exposedField") { t = LITERAL_exposedField; }
    else if (tokenString == "EXTERNPROTO")  { t = LITERAL_EXTERNPROTO; }
    else if (tokenString == "FALSE")        { t = LITERAL_FALSE; }
    else if (tokenString == "field")        { t = LITERAL_field; }
    else if (tokenString == "IS")           { t = LITERAL_IS; }
    else if (tokenString == "NULL")         { t = LITERAL_NULL; }
    else if (tokenString == "PROTO")        { t = LITERAL_PROTO; }
    else if (tokenString == "ROUTE")        { t = LITERAL_ROUTE; }
    else if (tokenString == "TO")           { t = LITERAL_TO; }
    else if (tokenString == "TRUE")         { t = LITERAL_TRUE; }
    else if (tokenString == "USE")          { t = LITERAL_USE; }
    else if (tokenString == "Script")       { t = LITERAL_Script; }
    else if (tokenString == "SFBool")     { t = LITERAL_SFBool; }
    else if (tokenString == "SFColor")    { t = LITERAL_SFColor; }
    else if (tokenString == "SFFloat")    { t = LITERAL_SFFloat; }
    else if (tokenString == "SFImage")    { t = LITERAL_SFImage; }
    else if (tokenString == "SFInt32")    { t = 36;/*LITERAL_SFInt32;*/ }
    else if (tokenString == "SFNode")     { t = LITERAL_SFNode; }
    else if (tokenString == "SFRotation") { t = LITERAL_SFRotation; }
    else if (tokenString == "SFString")   { t = LITERAL_SFString; }
    else if (tokenString == "SFTime")     { t = LITERAL_SFTime; }
    else if (tokenString == "SFVec2f")    { t = 41;/*LITERAL_SFVec2f;*/ }
    else if (tokenString == "SFVec3f")    { t = 42;/*LITERAL_SFVec3f;*/ }
    else if (tokenString == "MFColor")    { t = LITERAL_MFColor; }
    else if (tokenString == "MFFloat")    { t = LITERAL_MFFloat; }
    else if (tokenString == "MFInt32")    { t = 25;/*LITERAL_MFInt32;*/ }
    else if (tokenString == "MFNode")     { t = LITERAL_MFNode; }
    else if (tokenString == "MFRotation") { t = LITERAL_MFRotation; }
    else if (tokenString == "MFString")   { t = LITERAL_MFString; }
    else if (tokenString == "MFVec2f")    { t = 30;/*LITERAL_MFVec2f*/; }
    else if (tokenString == "MFVec3f")    { t = 31;/*LITERAL_MFVec3f*/; }
   
    token->setType(t);
    return token;
  }  
  ////////////////////////////////////////////////////////////
  // Numbers (the big part)
  // Note: this can also return a DOT
  ////////////////////////////////////////////////////////////
  if ((c_ == '.') || (c_ == '+') || (c_ == '-') || isdigit(c_))
  {
    // We are probably in a number. Except if the . is not followed by a
    // digit. We get rid of this special case first.
    string numberString;
    if (c_ == '.')
    {
      swallowOneChar();
      if (!isdigit(c_))
      {
	spitOneChar();
	token->setType(DOT);
	return token;
      }
      else
      {
	numberString.push_back('.');
      }
    }
    // Now we know we are in a number but we do not know yet if it is a
    // float or an int. We must also check that we do not have invalid
    // declaration (like 9.123.876 which should not be read as 2 floats!!!)
    // We read everything until the next white space and then we decide.
    while (maybePartOfNumber(c_))
    {
      numberString += c_;
      swallowOneChar();
    }
    spitOneChar();
    // We use already debugged function from C to test validity?!?
    // TODO: we could return the read values to avoid a second atoi/atof
    // call in parser.
    char* firstErrorChar;
    (void) strtol(numberString.c_str(),&firstErrorChar,0);
    if (*firstErrorChar == 0)
    {
      token->setText(numberString);
      token->setType(INT);
      return token;
    }
#ifndef WIN32
    (void) strtof(numberString.c_str(),&firstErrorChar);
    if (*firstErrorChar == 0)
    {
      token->setText(numberString);
      token->setType(FLOAT);
      return token;
    }
#else
	token->setText(numberString);
    token->setType(FLOAT);
    return token;
#endif
    throw ANTLR_USE_NAMESPACE(antlr)NoViableAltForCharException(*firstErrorChar,
								getFilename(),
								line_,
								col_);
  }  
  ////////////////////////////////////////////////////////////
  // Terminal symbols. 
  // Note: the DOT is filtered above by numbers
  ////////////////////////////////////////////////////////////  
  if (c_ == '[')  { token->setType(LBRACKET); return token; }
  if (c_ == ']')  { token->setType(RBRACKET); return token; }
  if (c_ == '{')  { token->setType(LBRACE); return token; }
  if (c_ == '}')  { token->setType(RBRACE); return token; }
  ////////////////////////////////////////////////////////////
  // Strings
  ////////////////////////////////////////////////////////////  
  if (c_ == '"')
  {
    token->setType(STRING);
    // Rq: we don't add the quotes in the token
    swallowOneChar();
    char p = '\0';
    while ((c_ != '"') || (p == '\\'))
    {
      if (isNewlineChar(c_))
      {
	if (!((c_ == 0x0a) && (prevChar_ == 0x0d)))
	{
	  ++line_; // newline();
	  col_ = 0;
	}
      }
      tokenString += c_;
      p = c_;
      swallowOneChar();
    }
    token->setText(tokenString);
    return token;
  }  
  exit(1);
}
void
vrml97fastlexer::swallowOneChar()
{
  if (!cWasSpit_)
  {
    prevChar_ = c_;
    c_ = istm_.get();
    ++col_; // Increment the column count;
  }
  else
  {
    cWasSpit_ = false;
  }    
}
void
vrml97fastlexer::spitOneChar()
{
  cWasSpit_ = true;
}
int
vrml97fastlexer::getLine() const
{
  return line_;
}
int
vrml97fastlexer::getColumn() const
{
  return col_;
}
