#ifndef XDKWRL_NODE_H
#define XDKWRL_NODE_H

#include <xdkwrl/config.h>
#include <xdkwrl/fieldtypes/sfbool.h>
#include <xdkwrl/fieldtypes/sfcolor.h>
#include <xdkwrl/fieldtypes/sffloat.h>
#include <xdkwrl/fieldtypes/sfimage.h>
#include <xdkwrl/fieldtypes/sfint32.h>
#include <xdkwrl/fieldtypes/sfnode.h>
#include <xdkwrl/fieldtypes/sfrotation.h>
#include <xdkwrl/fieldtypes/sfstring.h>
#include <xdkwrl/fieldtypes/sftime.h>
#include <xdkwrl/fieldtypes/sfvec2f.h>
#include <xdkwrl/fieldtypes/sfvec3f.h>
#include <xdkwrl/fieldtypes/mfcolor.h>
#include <xdkwrl/fieldtypes/mffloat.h>
#include <xdkwrl/fieldtypes/mfint32.h>
#include <xdkwrl/fieldtypes/mfnode.h>
#include <xdkwrl/fieldtypes/mfrotation.h>
#include <xdkwrl/fieldtypes/mfstring.h>
#include <xdkwrl/fieldtypes/mftime.h>
#include <xdkwrl/fieldtypes/mfvec2f.h>
#include <xdkwrl/fieldtypes/mfvec3f.h>  
#include <xdkwrl/event.h>

#include <string>
#include <map>

namespace wrl
{
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Interface of Node
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++  
  class XDKWRL_API Node
  {
  public:
    class FieldHandle;
    class EventInHandle;
    class EventOutHandle;
    Node(const char* name="");
    virtual Node* duplicate() const = 0;    
    virtual Node* duplicate(std::map<const Node*,Node*>&) const = 0;    
    virtual inline ~Node();
    /*! @name Field Access
     * 
     */
    //!@{
    virtual FieldHandle field(const std::string& n) = 0;
    virtual unsigned int nbFields() const = 0;
    virtual FieldHandle field(unsigned int i) = 0;
    virtual bool isSetToDefaultValue(unsigned int i) const = 0;
    //!@}
    /*! @name Field Modification
     * Setting a field from a handle and a value.
     */
    //!@{
    virtual void setField(const FieldHandle& f,const SFBool&);
    virtual void setField(const FieldHandle& f,const SFColor&);
    virtual void setField(const FieldHandle& f,const SFFloat&);
    virtual void setField(const FieldHandle& f,const SFImage&);
    virtual void setField(const FieldHandle& f,const SFInt32&);
    virtual void setField(const FieldHandle& f,const SFNode&);
    virtual void setField(const FieldHandle& f,const SFRotation&);
    virtual void setField(const FieldHandle& f,const SFString&);
    virtual void setField(const FieldHandle& f,const SFTime&);
    virtual void setField(const FieldHandle& f,const SFVec2f&);
    virtual void setField(const FieldHandle& f,const SFVec3f&);
    virtual void setField(const FieldHandle& f,const MFColor&);
    virtual void setField(const FieldHandle& f,const MFFloat&);
    virtual void setField(const FieldHandle& f,const MFInt32&);
    virtual void setField(const FieldHandle& f,const MFNode&);
    virtual void setField(const FieldHandle& f,const MFRotation&);
    virtual void setField(const FieldHandle& f,const MFString&);
    virtual void setField(const FieldHandle& f,const MFTime&);
    virtual void setField(const FieldHandle& f,const MFVec2f&);
    virtual void setField(const FieldHandle& f,const MFVec3f&);
    //!@}
    /*! @name Optimized Field Modification
     * Alternative to setField() functions that use swaps to be faster. 
     */
    //!@{
    void swapField(const FieldHandle& f,MFColor&);
    void swapField(const FieldHandle& f,MFFloat&);
    void swapField(const FieldHandle& f,MFInt32&);
    void swapField(const FieldHandle& f,MFNode&);
    void swapField(const FieldHandle& f,MFRotation&);
    void swapField(const FieldHandle& f,MFString&);
    void swapField(const FieldHandle& f,MFTime&);
    void swapField(const FieldHandle& f,MFVec2f&);
    void swapField(const FieldHandle& f,MFVec3f&);
    //!@}
    /*! @name Events Access
     * 
     */
    //!@{
    virtual unsigned int nbEventsIn() const = 0;
    virtual EventInHandle eventIn(const std::string& n) = 0;
    virtual EventInHandle eventIn(unsigned int i) = 0;
    virtual unsigned int nbEventsOut() const = 0;
    virtual EventOutHandle eventOut(const std::string& n) = 0;
    virtual EventOutHandle eventOut(unsigned int i) = 0;
    //!@}
    inline bool hasName() const;
    inline const char* name() const;
    void setName(const char*);
    
    virtual const char* typeName() const = 0;
  protected:
    FieldHandle field(const std::string& n,
		      const FieldTypeId t,void* f);
    EventInHandle eventIn(const std::string& n,
			  const FieldTypeId t,void* f);
    EventOutHandle eventOut(const std::string& n,
			    const FieldTypeId t,void* f);
  private:
    std::string name_;
  };
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Interface of DeclaredNode
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++  
  class XDKWRL_API DeclaredNode : public Node
  {
  public:
    DeclaredNode(const char* name="");    
    virtual unsigned int nbFields() const;
    virtual FieldHandle field(const std::string& n);
    virtual FieldHandle field(unsigned int i);

    virtual unsigned int nbEventsIn() const;
    virtual EventInHandle eventIn(const std::string& n);
    virtual EventInHandle eventIn(unsigned int i);
    virtual unsigned int nbEventsOut() const;
    virtual EventOutHandle eventOut(const std::string& n);
    virtual EventOutHandle eventOut(unsigned int i);

    /*! @name Declaration
     * Function used to declare dynamically extra fields or events to the
     * node type.
     */
    //! @{
    EventInHandle  declareEventIn(const FieldTypeId t,const std::string& n);
    EventOutHandle declareEventOut(const FieldTypeId t,const std::string& n);
    FieldHandle declareField(const FieldTypeId t,const std::string& n);
    FieldHandle declareExposedField(const FieldTypeId t,const std::string& n);
    virtual bool isDeclaredField(const std::string& n) const;
    virtual bool isDeclaredField(unsigned int i) const;
    virtual bool isDeclaredEventIn(const std::string& n) const;
    virtual bool isDeclaredEventIn(unsigned int i) const;
    virtual bool isDeclaredEventOut(const std::string& n) const;
    virtual bool isDeclaredEventOut(unsigned int i) const;
    //! @}
  protected:
    std::map<std::string,std::pair<SFBool,bool> >     sfBoolMap_;
    std::map<std::string,std::pair<SFColor,bool> >    sfColorMap_;
    std::map<std::string,std::pair<SFFloat,bool> >    sfFloatMap_;
    std::map<std::string,std::pair<SFImage,bool> >    sfImageMap_;
    std::map<std::string,std::pair<SFInt32,bool> >    sfInt32Map_;
    std::map<std::string,std::pair<SFNode,bool> >     sfNodeMap_;
    std::map<std::string,std::pair<SFRotation,bool> > sfRotationMap_;
    std::map<std::string,std::pair<SFString,bool> >   sfStringMap_;
    std::map<std::string,std::pair<SFTime,bool> >     sfTimeMap_;
    std::map<std::string,std::pair<SFVec2f,bool> >    sfVec2fMap_;
    std::map<std::string,std::pair<SFVec3f,bool> >    sfVec3fMap_;
    std::map<std::string,std::pair<MFColor,bool> >    mfColorMap_;
    std::map<std::string,std::pair<MFFloat,bool> >    mfFloatMap_;
    std::map<std::string,std::pair<MFInt32,bool> >    mfInt32Map_;
    std::map<std::string,std::pair<MFNode,bool> >     mfNodeMap_;
    std::map<std::string,std::pair<MFRotation,bool> > mfRotationMap_;
    std::map<std::string,std::pair<MFString,bool> >   mfStringMap_;
    std::map<std::string,std::pair<MFTime,bool> >     mfTimeMap_;
    std::map<std::string,std::pair<MFVec2f,bool> >    mfVec2fMap_;
    std::map<std::string,std::pair<MFVec3f,bool> >    mfVec3fMap_;    

    std::map<std::string,EventIn<SFBool> >     sfBoolInMap_;
    std::map<std::string,EventIn<SFColor> >    sfColorInMap_;
    std::map<std::string,EventIn<SFFloat> >    sfFloatInMap_;
    std::map<std::string,EventIn<SFImage> >    sfImageInMap_;
    std::map<std::string,EventIn<SFInt32> >    sfInt32InMap_;
    std::map<std::string,EventIn<SFNode> >     sfNodeInMap_;
    std::map<std::string,EventIn<SFRotation> > sfRotationInMap_;
    std::map<std::string,EventIn<SFString> >   sfStringInMap_;
    std::map<std::string,EventIn<SFTime> >     sfTimeInMap_;
    std::map<std::string,EventIn<SFVec2f> >    sfVec2fInMap_;
    std::map<std::string,EventIn<SFVec3f> >    sfVec3fInMap_;
    std::map<std::string,EventIn<MFColor> >    mfColorInMap_;
    std::map<std::string,EventIn<MFFloat> >    mfFloatInMap_;
    std::map<std::string,EventIn<MFInt32> >    mfInt32InMap_;
    std::map<std::string,EventIn<MFNode> >     mfNodeInMap_;
    std::map<std::string,EventIn<MFRotation> > mfRotationInMap_;
    std::map<std::string,EventIn<MFString> >   mfStringInMap_;
    std::map<std::string,EventIn<MFTime> >     mfTimeInMap_;
    std::map<std::string,EventIn<MFVec2f> >    mfVec2fInMap_;
    std::map<std::string,EventIn<MFVec3f> >    mfVec3fInMap_;

    std::map<std::string,EventOut<SFBool> >     sfBoolOutMap_;
    std::map<std::string,EventOut<SFColor> >    sfColorOutMap_;
    std::map<std::string,EventOut<SFFloat> >    sfFloatOutMap_;
    std::map<std::string,EventOut<SFImage> >    sfImageOutMap_;
    std::map<std::string,EventOut<SFInt32> >    sfInt32OutMap_;
    std::map<std::string,EventOut<SFNode> >     sfNodeOutMap_;
    std::map<std::string,EventOut<SFRotation> > sfRotationOutMap_;
    std::map<std::string,EventOut<SFString> >   sfStringOutMap_;
    std::map<std::string,EventOut<SFTime> >     sfTimeOutMap_;
    std::map<std::string,EventOut<SFVec2f> >    sfVec2fOutMap_;
    std::map<std::string,EventOut<SFVec3f> >    sfVec3fOutMap_;
    std::map<std::string,EventOut<MFColor> >    mfColorOutMap_;
    std::map<std::string,EventOut<MFFloat> >    mfFloatOutMap_;
    std::map<std::string,EventOut<MFInt32> >    mfInt32OutMap_;
    std::map<std::string,EventOut<MFNode> >     mfNodeOutMap_;
    std::map<std::string,EventOut<MFRotation> > mfRotationOutMap_;
    std::map<std::string,EventOut<MFString> >   mfStringOutMap_;
    std::map<std::string,EventOut<MFTime> >     mfTimeOutMap_;
    std::map<std::string,EventOut<MFVec2f> >    mfVec2fOutMap_;
    std::map<std::string,EventOut<MFVec3f> >    mfVec3fOutMap_;
  };
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Interface of Node::FieldHandle
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  class XDKWRL_API Node::FieldHandle
  {
  public:
    inline FieldHandle();
    inline FieldTypeId fieldTypeId() const;
    inline Node* node() const;
    inline const char* fieldName() const;
    const char* fieldTypeName() const;
    inline bool isExposed() const;
    inline SFBool*	sfBoolValue() const;
    inline SFColor*	sfColorValue() const;
    inline SFFloat*	sfFloatValue() const;
    inline SFImage*	sfImageValue() const;
    inline SFInt32*	sfInt32Value() const;
    inline SFNode*	sfNodeValue() const;
    inline SFRotation*	sfRotationValue() const;
    inline SFString*	sfStringValue() const;
    inline SFTime*	sfTimeValue() const;
    inline SFVec2f*	sfVec2fValue() const;
    inline SFVec3f*	sfVec3fValue() const;
    inline MFColor*	mfColorValue() const;
    inline MFFloat*	mfFloatValue() const;
    inline MFInt32*	mfInt32Value() const;
    inline MFNode*	mfNodeValue() const;
    inline MFRotation*	mfRotationValue() const;
    inline MFString*	mfStringValue() const;
    inline MFTime*	mfTimeValue() const;
    inline MFVec2f*	mfVec2fValue() const;
    inline MFVec3f*	mfVec3fValue() const;
    bool operator==(const FieldHandle& h) const;
    bool operator!=(const FieldHandle& h) const;
    static void copy(const FieldHandle& to,const FieldHandle& from);
  protected:
    friend class Node;
    FieldHandle(Node* node,
		const std::string& n,
		const FieldTypeId t,void* f);
  private:
    Node*       node_;
    std::string fieldName_;          
    FieldTypeId fieldTypeId_;
    bool        isExposed_;
    SFBool*	sfBoolField_;
    SFColor*	sfColorField_;
    SFFloat*	sfFloatField_;
    SFImage*	sfImageField_;
    SFInt32*	sfInt32Field_;
    SFNode*	sfNodeField_;
    SFRotation*	sfRotationField_;
    SFString*	sfStringField_;
    SFTime*	sfTimeField_;
    SFVec2f*	sfVec2fField_;
    SFVec3f*	sfVec3fField_;
    MFColor*	mfColorField_;
    MFFloat*	mfFloatField_;
    MFInt32*	mfInt32Field_;
    MFNode*	mfNodeField_;
    MFRotation*	mfRotationField_;
    MFString*	mfStringField_;
    MFTime*	mfTimeField_;
    MFVec2f*	mfVec2fField_;
    MFVec3f*	mfVec3fField_;  
  };
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Interface of Node::EventInHandle
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  class XDKWRL_API Node::EventInHandle
  {
  public:
    inline EventInHandle();
    inline FieldTypeId fieldTypeId() const;
    inline Node* node() const;
    inline const char* fieldName() const;
    const char* fieldTypeName() const;
    inline EventIn<SFBool>*	sfBoolValue() const;
    inline EventIn<SFColor>*	sfColorValue() const;
    inline EventIn<SFFloat>*	sfFloatValue() const;
    inline EventIn<SFImage>*	sfImageValue() const;
    inline EventIn<SFInt32>*	sfInt32Value() const;
    inline EventIn<SFNode>*	sfNodeValue() const;
    inline EventIn<SFRotation>*	sfRotationValue() const;
    inline EventIn<SFString>*	sfStringValue() const;
    inline EventIn<SFTime>*	sfTimeValue() const;
    inline EventIn<SFVec2f>*	sfVec2fValue() const;
    inline EventIn<SFVec3f>*	sfVec3fValue() const;
    inline EventIn<MFColor>*	mfColorValue() const;
    inline EventIn<MFFloat>*	mfFloatValue() const;
    inline EventIn<MFInt32>*	mfInt32Value() const;
    inline EventIn<MFNode>*	mfNodeValue() const;
    inline EventIn<MFRotation>*	mfRotationValue() const;
    inline EventIn<MFString>*	mfStringValue() const;
    inline EventIn<MFTime>*	mfTimeValue() const;
    inline EventIn<MFVec2f>*	mfVec2fValue() const;
    inline EventIn<MFVec3f>*	mfVec3fValue() const;
    bool operator==(const EventInHandle& h) const;
    bool operator!=(const EventInHandle& h) const;
  protected:
    friend class Node;
    EventInHandle(Node* node,
		  const std::string& n,
		  const FieldTypeId t,void* f);
  private:
    Node*                node_;
    std::string          fieldName_;          
    FieldTypeId          fieldTypeId_;
    EventIn<SFBool>*	 sfBoolField_;
    EventIn<SFColor>*	 sfColorField_;
    EventIn<SFFloat>*	 sfFloatField_;
    EventIn<SFImage>*	 sfImageField_;
    EventIn<SFInt32>*	 sfInt32Field_;
    EventIn<SFNode>*	 sfNodeField_;
    EventIn<SFRotation>* sfRotationField_;
    EventIn<SFString>*	 sfStringField_;
    EventIn<SFTime>*	 sfTimeField_;
    EventIn<SFVec2f>*	 sfVec2fField_;
    EventIn<SFVec3f>*	 sfVec3fField_;
    EventIn<MFColor>*	 mfColorField_;
    EventIn<MFFloat>*	 mfFloatField_;
    EventIn<MFInt32>*	 mfInt32Field_;
    EventIn<MFNode>*	 mfNodeField_;
    EventIn<MFRotation>* mfRotationField_;
    EventIn<MFString>*	 mfStringField_;
    EventIn<MFTime>*	 mfTimeField_;
    EventIn<MFVec2f>*	 mfVec2fField_;
    EventIn<MFVec3f>*	 mfVec3fField_;  
  };
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Interface of Node::EventOutHandle
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  class XDKWRL_API Node::EventOutHandle
  {
  public:
    inline EventOutHandle();
    inline FieldTypeId fieldTypeId() const;
    inline Node* node() const;
    inline const char* fieldName() const;
    const char* fieldTypeName() const;
    inline EventOut<SFBool>*	sfBoolValue() const;
    inline EventOut<SFColor>*	sfColorValue() const;
    inline EventOut<SFFloat>*	sfFloatValue() const;
    inline EventOut<SFImage>*	sfImageValue() const;
    inline EventOut<SFInt32>*	sfInt32Value() const;
    inline EventOut<SFNode>*	sfNodeValue() const;
    inline EventOut<SFRotation>*	sfRotationValue() const;
    inline EventOut<SFString>*	sfStringValue() const;
    inline EventOut<SFTime>*	sfTimeValue() const;
    inline EventOut<SFVec2f>*	sfVec2fValue() const;
    inline EventOut<SFVec3f>*	sfVec3fValue() const;
    inline EventOut<MFColor>*	mfColorValue() const;
    inline EventOut<MFFloat>*	mfFloatValue() const;
    inline EventOut<MFInt32>*	mfInt32Value() const;
    inline EventOut<MFNode>*	mfNodeValue() const;
    inline EventOut<MFRotation>*	mfRotationValue() const;
    inline EventOut<MFString>*	mfStringValue() const;
    inline EventOut<MFTime>*	mfTimeValue() const;
    inline EventOut<MFVec2f>*	mfVec2fValue() const;
    inline EventOut<MFVec3f>*	mfVec3fValue() const;
    bool operator==(const EventOutHandle& h) const;
    bool operator!=(const EventOutHandle& h) const;
  protected:
    friend class Node;
    EventOutHandle(Node* node,
		   const std::string& n,
		   const FieldTypeId t,void* f);
  private:
    Node*                node_;
    std::string          fieldName_;          
    FieldTypeId          fieldTypeId_;
    EventOut<SFBool>*	 sfBoolField_;
    EventOut<SFColor>*	 sfColorField_;
    EventOut<SFFloat>*	 sfFloatField_;
    EventOut<SFImage>*	 sfImageField_;
    EventOut<SFInt32>*	 sfInt32Field_;
    EventOut<SFNode>*	 sfNodeField_;
    EventOut<SFRotation>* sfRotationField_;
    EventOut<SFString>*	 sfStringField_;
    EventOut<SFTime>*	 sfTimeField_;
    EventOut<SFVec2f>*	 sfVec2fField_;
    EventOut<SFVec3f>*	 sfVec3fField_;
    EventOut<MFColor>*	 mfColorField_;
    EventOut<MFFloat>*	 mfFloatField_;
    EventOut<MFInt32>*	 mfInt32Field_;
    EventOut<MFNode>*	 mfNodeField_;
    EventOut<MFRotation>* mfRotationField_;
    EventOut<MFString>*	 mfStringField_;
    EventOut<MFTime>*	 mfTimeField_;
    EventOut<MFVec2f>*	 mfVec2fField_;
    EventOut<MFVec3f>*	 mfVec3fField_;  
  };
  //************************************************************
  // Implementation of Node
  //************************************************************
  inline
  Node::~Node()
  {
  }
  /*!
   * Return true if the node has a name, defined thru a DEF directive in a
   * VRML file of with the setName command.
   */
  inline bool
  Node::hasName() const
  {
    return !name_.empty();
  }
  /*!
   * Return the name of the node, defined thru a DEF directive in a
   * VRML file of with the setName command.
   */
  inline const char*
  Node::name() const
  {
    return name_.c_str();
  }
  //************************************************************
  // Implementation of Node::FieldHandle
  //************************************************************
  inline
  Node::FieldHandle::FieldHandle()
    : node_(NULL),fieldTypeId_(unknown)
  {
  }
  inline FieldTypeId
  Node::FieldHandle::fieldTypeId() const
  {
    return fieldTypeId_;
  }
  inline Node*
  Node::FieldHandle::node() const
  {
    return node_;
  }
  inline const char*
  Node::FieldHandle::fieldName() const
  {
    return fieldName_.c_str();
  }
  inline bool
  Node::FieldHandle::isExposed() const
  {
    return isExposed_;
  }
  inline SFBool*
  Node::FieldHandle::sfBoolValue() const
  {
    return sfBoolField_;
  }
  inline SFColor*
  Node::FieldHandle::sfColorValue() const
  {
    return sfColorField_;
  }
  inline SFFloat*
  Node::FieldHandle::sfFloatValue() const
  {
    return sfFloatField_;
  }
  inline SFImage*
  Node::FieldHandle::sfImageValue() const
  {
    return sfImageField_;
  }
  inline SFInt32*
  Node::FieldHandle::sfInt32Value() const
  {
    return sfInt32Field_;
  }
  inline SFNode*
  Node::FieldHandle::sfNodeValue() const
  {
    return sfNodeField_;
  }
  inline SFRotation*
  Node::FieldHandle::sfRotationValue() const
  {
    return sfRotationField_;
  }
  inline SFString*
  Node::FieldHandle::sfStringValue() const
  {
    return sfStringField_;
  }
  inline SFTime*
  Node::FieldHandle::sfTimeValue() const
  {
    return sfTimeField_;
  }
  inline SFVec2f*
  Node::FieldHandle::sfVec2fValue() const
  {
    return sfVec2fField_;
  }
  inline SFVec3f*
  Node::FieldHandle::sfVec3fValue() const
  {
    return sfVec3fField_;
  }
  inline MFColor*
  Node::FieldHandle::mfColorValue() const
  {
    return mfColorField_;
  }
  inline MFFloat*
  Node::FieldHandle::mfFloatValue() const
  {
    return mfFloatField_;
  }
  inline MFInt32*
  Node::FieldHandle::mfInt32Value() const
  {
    return mfInt32Field_;
  }
  inline MFNode*
  Node::FieldHandle::mfNodeValue() const
  {
    return mfNodeField_;
  }
  inline MFRotation*
  Node::FieldHandle::mfRotationValue() const
  {
    return mfRotationField_;
  }
  inline MFString*
  Node::FieldHandle::mfStringValue() const
  {
    return mfStringField_;
  }
  inline MFTime*
  Node::FieldHandle::mfTimeValue() const
  {
    return mfTimeField_;
  }
  inline MFVec2f*
  Node::FieldHandle::mfVec2fValue() const
  {
    return mfVec2fField_;
  }
  inline MFVec3f*
  Node::FieldHandle::mfVec3fValue() const
  {
    return mfVec3fField_;
  }
  //************************************************************
  // Implementation of Node::EventInHandle
  //************************************************************
  inline
  Node::EventInHandle::EventInHandle()
    : node_(NULL),fieldTypeId_(unknown)
  {
  }
  inline FieldTypeId
  Node::EventInHandle::fieldTypeId() const
  {
    return fieldTypeId_;
  }
  inline Node*
  Node::EventInHandle::node() const
  {
    return node_;
  }
  inline const char*
  Node::EventInHandle::fieldName() const
  {
    return fieldName_.c_str();
  }
  inline EventIn<SFBool>*
  Node::EventInHandle::sfBoolValue() const
  {
    return sfBoolField_;
  }
  inline EventIn<SFColor>*
  Node::EventInHandle::sfColorValue() const
  {
    return sfColorField_;
  }
  inline EventIn<SFFloat>*
  Node::EventInHandle::sfFloatValue() const
  {
    return sfFloatField_;
  }
  inline EventIn<SFImage>*
  Node::EventInHandle::sfImageValue() const
  {
    return sfImageField_;
  }
  inline EventIn<SFInt32>*
  Node::EventInHandle::sfInt32Value() const
  {
    return sfInt32Field_;
  }
  inline EventIn<SFNode>*
  Node::EventInHandle::sfNodeValue() const
  {
    return sfNodeField_;
  }
  inline EventIn<SFRotation>*
  Node::EventInHandle::sfRotationValue() const
  {
    return sfRotationField_;
  }
  inline EventIn<SFString>*
  Node::EventInHandle::sfStringValue() const
  {
    return sfStringField_;
  }
  inline EventIn<SFTime>*
  Node::EventInHandle::sfTimeValue() const
  {
    return sfTimeField_;
  }
  inline EventIn<SFVec2f>*
  Node::EventInHandle::sfVec2fValue() const
  {
    return sfVec2fField_;
  }
  inline EventIn<SFVec3f>*
  Node::EventInHandle::sfVec3fValue() const
  {
    return sfVec3fField_;
  }
  inline EventIn<MFColor>*
  Node::EventInHandle::mfColorValue() const
  {
    return mfColorField_;
  }
  inline EventIn<MFFloat>*
  Node::EventInHandle::mfFloatValue() const
  {
    return mfFloatField_;
  }
  inline EventIn<MFInt32>*
  Node::EventInHandle::mfInt32Value() const
  {
    return mfInt32Field_;
  }
  inline EventIn<MFNode>*
  Node::EventInHandle::mfNodeValue() const
  {
    return mfNodeField_;
  }
  inline EventIn<MFRotation>*
  Node::EventInHandle::mfRotationValue() const
  {
    return mfRotationField_;
  }
  inline EventIn<MFString>*
  Node::EventInHandle::mfStringValue() const
  {
    return mfStringField_;
  }
  inline EventIn<MFTime>*
  Node::EventInHandle::mfTimeValue() const
  {
    return mfTimeField_;
  }
  inline EventIn<MFVec2f>*
  Node::EventInHandle::mfVec2fValue() const
  {
    return mfVec2fField_;
  }
  inline EventIn<MFVec3f>*
  Node::EventInHandle::mfVec3fValue() const
  {
    return mfVec3fField_;
  }
  //************************************************************
  // Implementation of Node::EventOutHandle
  //************************************************************
  inline
  Node::EventOutHandle::EventOutHandle()
    : node_(NULL),fieldTypeId_(unknown)
  {
  }
  inline FieldTypeId
  Node::EventOutHandle::fieldTypeId() const
  {
    return fieldTypeId_;
  }
  inline Node*
  Node::EventOutHandle::node() const
  {
    return node_;
  }
  inline const char*
  Node::EventOutHandle::fieldName() const
  {
    return fieldName_.c_str();
  }
  inline EventOut<SFBool>*
  Node::EventOutHandle::sfBoolValue() const
  {
    return sfBoolField_;
  }
  inline EventOut<SFColor>*
  Node::EventOutHandle::sfColorValue() const
  {
    return sfColorField_;
  }
  inline EventOut<SFFloat>*
  Node::EventOutHandle::sfFloatValue() const
  {
    return sfFloatField_;
  }
  inline EventOut<SFImage>*
  Node::EventOutHandle::sfImageValue() const
  {
    return sfImageField_;
  }
  inline EventOut<SFInt32>*
  Node::EventOutHandle::sfInt32Value() const
  {
    return sfInt32Field_;
  }
  inline EventOut<SFNode>*
  Node::EventOutHandle::sfNodeValue() const
  {
    return sfNodeField_;
  }
  inline EventOut<SFRotation>*
  Node::EventOutHandle::sfRotationValue() const
  {
    return sfRotationField_;
  }
  inline EventOut<SFString>*
  Node::EventOutHandle::sfStringValue() const
  {
    return sfStringField_;
  }
  inline EventOut<SFTime>*
  Node::EventOutHandle::sfTimeValue() const
  {
    return sfTimeField_;
  }
  inline EventOut<SFVec2f>*
  Node::EventOutHandle::sfVec2fValue() const
  {
    return sfVec2fField_;
  }
  inline EventOut<SFVec3f>*
  Node::EventOutHandle::sfVec3fValue() const
  {
    return sfVec3fField_;
  }
  inline EventOut<MFColor>*
  Node::EventOutHandle::mfColorValue() const
  {
    return mfColorField_;
  }
  inline EventOut<MFFloat>*
  Node::EventOutHandle::mfFloatValue() const
  {
    return mfFloatField_;
  }
  inline EventOut<MFInt32>*
  Node::EventOutHandle::mfInt32Value() const
  {
    return mfInt32Field_;
  }
  inline EventOut<MFNode>*
  Node::EventOutHandle::mfNodeValue() const
  {
    return mfNodeField_;
  }
  inline EventOut<MFRotation>*
  Node::EventOutHandle::mfRotationValue() const
  {
    return mfRotationField_;
  }
  inline EventOut<MFString>*
  Node::EventOutHandle::mfStringValue() const
  {
    return mfStringField_;
  }
  inline EventOut<MFTime>*
  Node::EventOutHandle::mfTimeValue() const
  {
    return mfTimeField_;
  }
  inline EventOut<MFVec2f>*
  Node::EventOutHandle::mfVec2fValue() const
  {
    return mfVec2fField_;
  }
  inline EventOut<MFVec3f>*
  Node::EventOutHandle::mfVec3fValue() const
  {
    return mfVec3fField_;
  }
};

#endif // XDKWRL_NODE_H

// Local variables section.
// This is only used by emacs!
// Local Variables:
// ff-search-directories: ("." "../../src/xdkwrl/")
// End:
