/*******************************************************************************
* upld.c                                                                       *
* A test program. Takes hostname, port, and filename as three args to the upld *
* shl command. Sends the raw file to the host/port.                            *
* Copyright Troy Hanson, 21 July 2005                                          *
*******************************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <string.h>
#include "libut/ut.h"

int upld_done_cb(char *name, void *data, int flags, UT_iob *io[2]) {

    if (flags & UTFD_IS_REQFAILURE) {
        UT_LOG(Error, "upld failed");
        return;
    }

    /* can assume REQSUCCESS since we did not have REQFALURE */
    if (flags & UTFD_IS_REQSUCCESS) UT_LOG(Info, "upld finished");

    /* the iobs are our responsibility to free */
    UT_iob_free(io[0]);
}

UT_iob *file_to_iob(char *filename) {
    struct stat stat_buf;
    char *text;
    UT_iob *iob;
    int fd;

    UT_LOG(Debug, "reading file %s", filename);

    if ( (fd = open(filename, O_RDONLY)) == -1 ) {
        UT_LOG(Error,"Couldn't open file %s: %s", filename, strerror(errno));
        return NULL;
    }

    if ( fstat(fd, &stat_buf) == -1) {
        close(fd);
        UT_LOG(Error,"Couldn't stat file %s: %s", filename, strerror(errno));
        return NULL;
    }

    /* map the file into memory */
    text = (char*)mmap(0, stat_buf.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
    if (text == MAP_FAILED) {
        close(fd);
        UT_LOG(Error,"Failed to mmap %s: %s", filename, strerror(errno));
        return NULL;
    }

    iob = UT_iob_create();
    UT_iob_append(iob, text, stat_buf.st_size);
    if ( munmap( text, stat_buf.st_size ) == -1 ) {
        UT_LOG(Error,"Failed to munmap %s: %s", filename, strerror(errno));
    }
    close(fd);
    return iob;
}

int upld_test_cmd(int argc, char *argv[] ) {
    int rc,port;
    UT_iob *io[2];
    char *dst_ipport;

    if (argc != 3) {
        UT_shle("usage: upld <file> <host:port>\n");
        return SHL_ERROR;
    }

    if ( (dst_ipport = UT_net_resolve( argv[2])) == NULL ) {
        UT_shle("name resolution of %s failed\n", argv[2]);
        return SHL_ERROR;
    }

    io[0] = file_to_iob(argv[1]);
    if (io[0] == NULL) {
        UT_shle("upld failed\n");
        return SHL_ERROR;
    }
    io[1] = NULL;  /* don't gather reply */
    rc = UT_net_request("upld", upld_done_cb, NULL, io, UT_CONNECT_TO_IPPORT,
            dst_ipport); 
    if (rc < 0) {
        UT_shle("request error");
        return SHL_ERROR;
    }
    return SHL_OK;
}

int dns_test_cmd(int argc, char *argv[] ) {
    char *s;

    if (argc != 2) {
        UT_shle("usage: dns host:port");
        return SHL_ERROR;
    }

    s = UT_net_resolve(argv[1]);
    UT_shlf("%s converts to %s\n", argv[1], (s ? s : "<invalid>"));
    return SHL_OK;
}

int main(int argc, char **argv) {
	UT_init(INIT_END);
    UT_shl_cmd_create("upld", "upload <file> to <host> <port>",upld_test_cmd,NULL);
    UT_shl_cmd_create("dns", "translate host:port to ip:port", dns_test_cmd, NULL);
	UT_event_loop();
}
