/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
 * Command port shell support                                                  *
 ******************************************************************************/
static const char id[]="$Id: shl.c,v 1.29 2005/11/09 05:43:31 thanson Exp $";

#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "libut/ut_internal.h"

extern UT_loop_global_type UT_loop_global;

/* boot_tmp is a init hack to support initialization of UT subsystems (such as 
 * mem) which need to register shl cmds prior to shl's own initialization.  */
static UT_shl_cmd boot_tmp_cmds[NUM_BOOT_TMP_CMDS];
static boot_tmp_idx=0;

static UT_shl_lang_glue glues[] = {GLUES};

UT_shl_global_type UT_shl_global = { 
    .inited  = 0,
    .listen_fd  = -1,
    .sessions  = NULL,
    .active_session  = NULL,
    .cmds  = NULL,
    .glue  = glues
};

void UT_shl_init(char *ipport) {
    UT_shl_cmd *cmd;
    int i;
    char *cwd;

    UT_mem_pool_create( SHL_SESSION_POOL, sizeof(UT_shl_session), 1);
    UT_mem_pool_create( SHL_CMD_POOL, sizeof(UT_shl_cmd), 10);
    UT_var_create(SHL_IPPORT_VARNAME, "control port IP/port", UT_var_string, ipport);
    UT_var_reg_cb(SHL_IPPORT_VARNAME, (UT_var_upd_cb*)UT_shl_upd_port, NULL);

    /* shl commands that deal with relative files should resolve their path
     * relative to the shl base directory rather than process working directory
     * since the latter may be subject to application whim */
    cwd = getcwd( NULL, PATH_MAX);
    UT_var_create( SHL_BASEDIR_VARNAME, "shl base directory", UT_var_string, cwd);
    free( cwd );

    /* set the init flag, to prevent further command defs from being deferred */
    /* and define those commands whose definition was deferred until now.     */
    UT_shl_global.inited = 1; 
    for(i=0;i<boot_tmp_idx;i++) {
        cmd=&boot_tmp_cmds[i];
        UT_shl_cmd_create(cmd->name,cmd->desc,cmd->cb,cmd->helpcb);
    }

    UT_shl_cmd_create("help", "view command help", UT_shl_help_cmd, NULL);
    UT_shl_cmd_create("exit", "close connection", UT_shl_exit_cmd, NULL);

    /* the shl listener port is opened during final init (after reading the 
     * config file) to allow the shl port to be altered in the config file */
}

/*******************************************************************************
* UT_shl_cmd_create()                                                          *
* This function is how applications (and UT) register commands into the shl.   *
*******************************************************************************/
UT_API int UT_shl_cmd_create( char *name, char *desc, UT_shlcmd *proc, UT_helpcb
        *help) {
    UT_shl_cmd *cmd,*tmp;

    /* Prior to shl init, early UT subsystem init (e.g. mem) needs to define 
     * shl cmds. These are "buffered up" in boot_tmp_cmds until shl_init. */
    if (UT_shl_global.inited == 0) {
        if (boot_tmp_idx < NUM_BOOT_TMP_CMDS-1) {
            cmd = &boot_tmp_cmds[boot_tmp_idx++];
            UT_strncpy(cmd->name,name,SHLCMD_MAXLEN);
            UT_strncpy(cmd->desc,desc,SHLCMD_DESC_MAXLEN);
            cmd->cb= proc;
            cmd->helpcb= help;
            cmd->next= NULL;
        } else UT_LOG(Fatal,"insufficient boot_tmp space");
        return 0;
    }

    cmd = (UT_shl_cmd *)UT_mem_alloc( SHL_CMD_POOL, 1);
    UT_strncpy(cmd->name, name, SHLCMD_MAXLEN);
    UT_strncpy(cmd->desc, desc, SHLCMD_DESC_MAXLEN);
    cmd->cb = proc;
    cmd->helpcb = help;
    cmd->next = NULL;
    LL_ADD( UT_shl_global.cmds, tmp, cmd);

    return (UT_shl_global.glue->addcmd) ? UT_shl_global.glue->addcmd(cmd) : 0;
}

/*******************************************************************************
* UT_shl_create_session()                                                      *
* Establish a new session, i.e. interactive shl port connection.               *
*******************************************************************************/
UT_shl_session* UT_shl_create_session(int fd) {
    UT_shl_session *ses,*tmp;

    ses = (UT_shl_session*)UT_mem_alloc( SHL_SESSION_POOL, 1);
    ses->fd = fd;
    ses->in = UT_iob_create();
    ses->out = UT_iob_create();
    ses->err = UT_iob_create();
    ses->next = NULL;

    LL_ADD(UT_shl_global.sessions, tmp, ses);
    return ses;
}

/*******************************************************************************
* UT_shl_close_session()                                                       *
* Close the given session. If NULL, close all sessions.                        *
*******************************************************************************/
int UT_shl_close_session(UT_shl_session *ses) {
    UT_shl_session *s,*tmp;

    for( s = (ses ? ses : UT_shl_global.sessions); s; s= (ses ? NULL : s->next)) {
        LL_DEL(UT_shl_global.sessions,tmp,s);

        if (UT_shl_global.active_session == s) 
            UT_shl_global.active_session = NULL;

        if (s->fd > 0) {
            close(s->fd);
            UT_fd_unreg(s->fd);
        }

        UT_iob_free(s->in);
        UT_iob_free(s->out);
        UT_mem_free(SHL_SESSION_POOL, s, 1);
    }

    return 0;
}

/*******************************************************************************
*  UT_shl_build_argv()                                                         *
*  Primitive parsing of a space-delimited argument list into malloc'd argv[].  *
*******************************************************************************/
char **UT_shl_build_argv(char *in, int *argc_ptr) {
   char *c, *arg_start, **argi, **argv = NULL;
   int arg_count, in_arg;
 
   /* loop over input string doing initial arg count */
   for(c=in,arg_count=0,in_arg=0; *c != '\0'; c++) {
     if (in_arg && *c == ' ') { arg_count++; in_arg = 0; }
     else if (!in_arg && *c != ' ') { in_arg = 1; }
   }
   if (in_arg) arg_count++;
   *argc_ptr = arg_count;
 
   if (arg_count == 0) return NULL;
 
   argv = (char**)malloc(*argc_ptr * sizeof(char*));
   if (!argv) UT_LOG(Fatal,"malloc failure");
 
   for(c=in,arg_start=in,argi=argv,in_arg=0; *c != '\0'; c++) {
     if (in_arg && *c == ' ') { *argi++ = arg_start; *c='\0'; in_arg = 0; }
     else if (!in_arg && *c != ' ') { arg_start = c; in_arg = 1; }
   }
   if (in_arg) *argi = arg_start;

   return argv;
}

/*******************************************************************************
* UT_dsprintf()                                                                *
* This function has the interface of sprintf, but it dynamically allocates a   *
* buffer of sufficient length for the formatted result. The code is modeled    *
* closely after the example code in the Linux "vsnprintf" man page.            *
*******************************************************************************/
char *UT_dsprintf(char *fmt, va_list ap) {
    int n, size = 100;  /* initial guess */
    char *p, *np;
    va_list ac;

    if ((p = malloc (size)) == NULL) return NULL;

    while (1) {
            va_copy(ac, ap);                   /* preserve ap for re-use */
            n = vsnprintf(p, size, fmt, ac);   /* try to print in space */
            va_end(ac);
            if (n > -1 && n < size) return p;  /* it fit */

            /* it didn't fit. more space is needed. */

            if (n > -1)  size = n+1; /* exact size needed (glibc 2.1) */
            else         size *= 2;  /* double the guess  (glibc 2.0) */
            if ((np = realloc (p, size)) == NULL) {
                    free(p);
                    return NULL;
            } else {
                p = np;
            }
    }
}

/*******************************************************************
 * A utility function that checks if a string is present in a NULL-*
 * terminated array of strings.                                    *
 * Returns: index of matching string in list, or -1 (no match)     *
 *******************************************************************/
UT_API int UT_stridx( char *str, char *list[] ) {
    int i;

    for (i=0; list[i]; i++) if (!strcmp( str, list[i] )) return i;
    return -1;
}

/*******************************************************************************
* UT_shlf()                                                                    *
* This is how shl cmd implementations "printf" to the shl result buffer.       *
*******************************************************************************/
int UT_shlf(char*fmt, ...) {
  va_list ap;
  char *s;
  int i;

  va_start(ap,fmt);
  s = UT_dsprintf(fmt,ap);
  va_end(ap);

  if (!s) UT_LOG(Fatal,"Out of memory");

  if (UT_shl_global.active_session) {
    UT_iob_append(UT_shl_global.active_session->out, s, strlen(s));
  } else {
      /* config files emit output to no active session; echo to log */
      for(i=0;i<strlen(s);i++) if (s[i] == '\n') s[i] = '\\'; /* no newlines */
      UT_LOG(Info,"shl output [%s]", s);
  }

  free(s);
  return 0;
}

/*******************************************************************************
* UT_shle()                                                                    *
* This is how shl cmd implementations "printf" to the shl error  buffer.       *
*******************************************************************************/
int UT_shle(char*fmt, ...) {
  va_list ap;
  char *s;
  int i;

  va_start(ap,fmt);
  s = UT_dsprintf(fmt,ap);
  va_end(ap);

  if (!s) UT_LOG(Fatal,"Out of memory");

  if (UT_shl_global.active_session) {
    UT_iob_append(UT_shl_global.active_session->err, s, strlen(s));
  } else {
      /* config files emit output to no active session; echo to log */
      for(i=0;i<strlen(s);i++) if (s[i] == '\n') s[i] = '\\'; /* no newlines */
      UT_LOG(Info,"shl error [%s]", s);
  }

  free(s);
  return 0;
}

/* NOTE: tcl wrapper around all shl cmds should
 * TMP save active_session->out
 * CREATE new iob ""
 * evaluate the command
 * transfer ithe iob "" to the tcl result
 * RESTORE the original out iob
 */

/*******************************************************************************
* UT_shl_eval()                                                                *
* Evaluate a single command in the shl.                                        *
*******************************************************************************/
int UT_shl_eval(char*in) {
    int argc, rc, i,j;
    char **argv;
    UT_shl_cmd *cmd;

    if (UT_shl_global.glue->eval) return UT_shl_global.glue->eval(in);

    argv = UT_shl_build_argv(in, &argc);
    if (argc <= 0) return SHL_OK;  /* empty lines ok */

    LL_FIND(UT_shl_global.cmds, cmd, argv[0]);
    if (cmd) { 
        rc = cmd->cb(argc,argv); 
        if (rc == SHL_ERROR) {
            /* de-argv'ify the erroneous shl input so we can log it */
            for (i=0,j=0;i<argc-1;i++,j=0) {
                while(argv[i][j] != '\0') j++; 
                argv[i][j] = ' ';
            }
            UT_LOG(Error,"shl error evaluating [%s]", in);
        }
    } else {
        UT_shlf("unknown command");
        rc = SHL_ERROR;
    }

    free(argv);
    return rc;
}

/*******************************************************************************
* UT_shl_eval_file()                                                           *
* Evaluates a config file, normally used at program startup or via 'source'.   *
*******************************************************************************/
int UT_shl_eval_file(char *filename) {
    struct stat stat_buf;
    char *text, *start, *end;
    int fd, end_of_file;

    UT_LOG(Debug, "Evaluating shl file %s", filename);

    if (UT_shl_global.glue->feval) return UT_shl_global.glue->feval(filename);

    if ( (fd = open(filename, O_RDONLY)) == -1 ) {
        UT_LOG(Error,"Couldn't open file %s: %s", filename, strerror(errno));
        return -1;
    }

    if ( fstat(fd, &stat_buf) == -1) {
        close(fd);
        UT_LOG(Error,"Couldn't stat file %s: %s", filename, strerror(errno));
        return -1;
    }

    /* map the config file into memory; writes to memory only */
    text = (char*)mmap(0, stat_buf.st_size + 1, PROT_READ|PROT_WRITE, MAP_PRIVATE, fd, 0);
    if (text == MAP_FAILED) {
        close(fd);
        UT_LOG(Error,"Failed to mmap %s: %s", filename, strerror(errno));
        return -1;
    }

    /* evaluate lines, finding their start/end as a prerequisite */
    start = text;
    end = text;
    end_of_file = 0;
    while(!end_of_file) {
        if (end < text + stat_buf.st_size) {
            if (*end != '\r' && *end != '\n') end++;
            else {
                *end = '\0';
                if (UT_shl_eval(start) == SHL_ERROR) {
                    end_of_file = 1;
                    UT_LOG(Error,"stopping evaluation of %s", filename);
                }
                start = end+1;
                end = start;
            }
        } else {
            /* handle non-newline terminated last line. use extra mmap'd byte */
            end = text + stat_buf.st_size;   
            *end = '\0';                    
            UT_shl_eval(start);
            end_of_file = 1;
        }
    }

    if ( munmap( text, stat_buf.st_size ) == -1 ) {
        UT_LOG(Error,"Failed to munmap %s: %s", filename, strerror(errno));
    }
    close(fd);
    return 0;
}

/*******************************************************************************
* UT_shl_get_result()                                                          *
* Used internally to transfer the result out of the "result buffer" and into a *
* UT_iob.  In the case of native shl, the output buffer is already a UT_iob.   *
*******************************************************************************/
UT_iob *UT_shl_get_result(void) {
    UT_iob *out;

    out = UT_shl_global.active_session->out;
    if (UT_shl_global.glue->toiob) UT_shl_global.glue->toiob(out);  /* outbuf-->out */
    return out;
}

/*******************************************************************************
* UT_shl_get_error()                                                           *
* Used internally to transfer the result out of the "error  buffer" and into a *
* UT_iob.  In the case of native shl, the error  buffer is already a UT_iob.   *
*******************************************************************************/
UT_iob *UT_shl_get_error(void) {
    UT_iob *err;

    err = UT_shl_global.active_session->err;
    if (UT_shl_global.glue->errtoiob) UT_shl_global.glue->errtoiob(err); /* get err */
    return err;
}

/*******************************************************************************
* UT_shl_port_cb()                                                             *
* Accumulate and evaluate user-entered shl commands                            *
*******************************************************************************/
int UT_shl_port_cb(int fd, char *name, int flags, void *unused) {
    const char *result, *prompt="\r\nOk\r\n", *error="\r\nError\r\n",
               *greeting="libut control port interpreter\r\n"
                         "Type 'help' for command list.\r\n\r\n", *errsts=NULL;
    char input[SHL_PORT_MAX_INPUT], *cmdtxt=NULL;
    size_t ioblen;
    int rc,i,found_eol,fg;
    UT_shl_session *session;
    UT_iob *in,*out,*err;

    if (flags & UTFD_IS_NEWACCEPT) {
        UT_fd_write(fd, (void*)greeting, strlen(greeting));
        UT_LOG( Info, "accepted shl port connection (fd %d)", fd);
        UT_shl_create_session(fd);

        /* set close-on-exec flag so any exec'd children close shl conns  */
        if ( (fg = fcntl( fd, F_GETFD, 0)) >= 0) {
            fg |= FD_CLOEXEC;
            if (fcntl(fd,F_SETFD,fg) == -1) UT_LOG(Error,"fcntl setfd failed");
        } else UT_LOG(Error, "fcntl getfd failed");

        return;
    }

    LL_FIND_BY_MEMBER(UT_shl_global.sessions,session,fd,fd);
    if (!session) UT_LOG(Fatal,"failed to find session for fd %d", fd);
    UT_shl_global.active_session = session;
        

    if ( (rc=recv(fd, input, SHL_PORT_MAX_INPUT, 0)) > 0) {
        for(i=0,found_eol=0; i<rc && !found_eol; i++) {
            if (input[i]=='\r' || input[i]=='\n') {
                input[i] = '\0';
                found_eol=1;
            }
        }
    }

    in = session->in;

    if (rc == -1 && errno == EINTR) return;
    else if (rc == -1) errsts = strerror(errno);
    else if (rc == 0) errsts = "remote close";
    else if (UT_iob_len(in) + rc >= SHL_PORT_MAX_INPUT) errsts = "flooded";

    if (errsts) {
        UT_LOG(Info,"shl port connection closed (%s)", errsts);
        UT_shl_close_session(session);
        return;
    }

    if (!found_eol) {
        UT_iob_append(in,input,rc);
        return;
    }

    /* Concatenate buffer with current input, if needed. */
    if (UT_iob_len(in) > 0) {
        UT_iob_append(in,input,rc);
        cmdtxt= UT_iob_flatten(in,&ioblen);
        /* reset input buffer */
        UT_iob_free(in);
        session->in = UT_iob_create();
    }

    rc = UT_shl_eval(cmdtxt ? cmdtxt : input);

    /* Go on if the 'exit' command did not close the active session */
    if (UT_shl_global.active_session) {
        if (rc == SHL_OK) {
            out = UT_shl_get_result();
            UT_iob_writefd(out,fd);
            UT_fd_write(fd, (void*)prompt, strlen(prompt));
        } else {
            err = UT_shl_get_error();
            UT_iob_writefd(err,fd);
            UT_fd_write(fd, (void*)error, strlen(error));
            /* reset error iob */
            UT_iob_free(session->err);
            session->err = UT_iob_create();
        }

        /* reset outbut buffer */
        UT_iob_free(session->out);
        session->out = UT_iob_create();
    } 

    if (cmdtxt) free(cmdtxt);
}

/*******************************************************************************
* UT_shl_close_port()                                                          *
*******************************************************************************/
int UT_shl_close_port(void) {
    if (UT_shl_global.listen_fd != -1) {
        UT_net_listen_close(UT_shl_global.listen_fd);
        UT_shl_global.listen_fd = -1;
    }
    return 0;
}

/*******************************************************************************
* UT_shl_open_port()                                                           *
*******************************************************************************/
int UT_shl_open_port(char *ipport) {
    int rc;

    /* sanity check */
    if (UT_shl_global.listen_fd != -1) {
        UT_LOG(Warning, "control port listener already exists (fd %d)", 
                UT_shl_global.listen_fd);
        return -1;
    }

    /* listen on the requested ip/port */
    rc = UT_net_listen("control_port", ipport, (UT_fd_cb*)UT_shl_port_cb, NULL);
    if (rc == -1) UT_LOG(Error, "Failed to open control port on %s", ipport);

    UT_shl_global.listen_fd = rc;
    return rc;
}

/*******************************************************************************
* UT_shl_upd_port()                                                            *
* Callback invoked whenever ut_shl_port or ut_shl_ipaddr is updated.           *
*******************************************************************************/
int UT_shl_upd_port(char *name,void *data) {
    char *ipport;
    in_addr_t ip;
    int port, rc, oldfd;
    UT_fd *fdx;

    UT_var_get(SHL_IPPORT_VARNAME, &ipport);
    if (UT_ipport_parse(ipport, &ip, &port) < 0) {
        UT_LOG(Error, "Invalid IP:port [%s]; reverting.", ipport);
        return -1;
    }

    /* find the UT_fd record for the active shl listener, if there is one. */
    fdx = NULL;
    if (UT_shl_global.listen_fd != -1) {
        HASH_FIND_INT(UT_loop_global.fds, fdx, fd, &UT_shl_global.listen_fd);
    }

    /* See if the IP/port specified in the var already match the listener's */
    if (fdx) {
        if (!strcmp(ipport,fdx->aux.socket.local_ipport)) return 0;
    }

    /* If port is 0, close pre-existing listener (if any), don't open one. */
    if (port == 0 && fdx) {
        UT_shl_close_port();  
        return 0;
    }

    /* We're going to open the new port, and if its successful, we'll close the
     * old port afterward (if it was open). */
    oldfd = UT_shl_global.listen_fd;
    UT_shl_global.listen_fd = -1;

    if (UT_shl_open_port(ipport) < 0) {
        UT_LOG(Warning, "Control port reverting to former IP/port");
        UT_shl_global.listen_fd = oldfd;
        return -1;
    }

    if (oldfd != -1) UT_net_listen_close(oldfd);
    return 0;
}

/*******************************************************************************
* UT_shl_hdr()                                                                 *
* Applications (and UT) use this to write a header line to the shl result      *
* buffer. The header fields will be printed with hyphen-underlining. cols is a *
* NULL-terminated array of strings.                                            *
*******************************************************************************/
UT_API int UT_shl_hdr( char *cols[] ) {
    int i,j;

    for (i=0; cols[i]; i++) UT_shlf("%s ",cols[i]); 
    UT_shlf("\n");

    /* Print a nice hyphenated header underbar.   */
    for (i=0; cols[i]; i++) {
        j = strlen(cols[i]);
        while (j--) UT_shlf("-");
        UT_shlf(" ");
    }
    UT_shlf("\n");
    return 0;
}

/*****************************************************************************
 * UT_dispatch_subcmd                                                        *
 * This utility function takes a name (string) and a function table          *
 * (string->function mapping) and returns the appropriate function.          *
 * It returns NULL if there is no appropriate function.                      *
 ****************************************************************************/
UT_API UT_clproc *UT_dispatch_subcmd( char *name, UT_cmdtable f[]) {
    if (!name || !f) return NULL;

    while (f) {
        /* if the name is NULL then its the end of the functable */
        if (!(f->name)) break;

        /* if this is the element of the functable we want, return it */
        if (!strcmp(f->name, name)) return f->proc;

        /* look at the next entry in the functable */
        f++;
    }

    /* didn't find it */
    return NULL;  
}

/*******************************************************************************
* UT_shl_help_cmd()                                                            *
* Display the list of commands and their descriptions, or detailed help for a  *
* given command.                                                               *
*******************************************************************************/
int UT_shl_help_cmd( int argc, char *argv[] ) {
    UT_shl_cmd *cmd;
    char *cols[] = {"command          ", 
                    "description                          ", NULL};

    switch (argc) {
        case 1:
            UT_shl_hdr(cols);
            for(cmd = UT_shl_global.cmds; cmd; cmd=cmd->next) 
                UT_shlf("%c %-15s - %s\n", cmd->helpcb ? '*' : ' ', cmd->name, 
                        cmd->desc);
            UT_shlf("\nCommands preceded by * have detailed help. "
                    "Use help <command>.\n");
            break;
        case 2:
            LL_FIND(UT_shl_global.cmds, cmd, argv[1]);
            if (!cmd) UT_shlf("unknown command %s\n");
            else if (cmd->helpcb) cmd->helpcb(argc-1,&argv[1]);
            else UT_shlf("no detailed help is available on this command\n");
            break;
        default:
            UT_shlf("usage: help [command]\n");
    }

    return SHL_OK;
}

/*******************************************************************************
* UT_exit_shl_cmd()                                                            *
* Close the connection to the shl port.                                        *
*******************************************************************************/
int UT_shl_exit_cmd( int argc, char *argv[] ) {

    UT_shl_close_session( UT_shl_global.active_session );
    return SHL_OK;
}

/*******************************************************************************
* UT_strncpy()                                                                 *
* strncpy with guaranteed null termination                                     *
*******************************************************************************/
UT_API char *UT_strncpy(char *dst,char *src,size_t n) {
    if (dst != src) strncpy(dst,src,n-1); 
    dst[n-1] = '\0';
    return dst;
}

