/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
* request.c                                                                    *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/
static const char id[]="$Id: request.c,v 1.14 2005/10/24 01:40:32 thanson Exp $";

#include <errno.h>
#include "libut/ut_internal.h"

#define READ_BUFLEN 1024

/******************************************************************************
 * UT_net_request_cb()                                                        *
 * Invoked after a connect() request completes or fails. Our job is to send   *
 * the request buffer, and optionally wait for a response, then at last to    *
 * invoke the i/o callback to notify application that UT_net_request is done. *
 *****************************************************************************/
int UT_net_request_post_cb(int fd, char *name, int flags, UT_net_request_type
        *rqst) { char readbuf[READ_BUFLEN];
    int n,f,reply_wanted;

    /* If the connect() failed, inform the app callback. */
    if (flags & UTFD_IS_CONFAILURE) {
        (rqst->cb)(name, rqst->data, (flags|UTFD_IS_REQFAILURE), rqst->io);
        UT_mem_free( UTRQST, rqst, 1);
        return;
    } 

    reply_wanted = (rqst->io[1] != NULL) ? 1 : 0;

    /* The connect succeeded. If I/O hasn't started yet, send the request. */
    if (flags & UTFD_IS_CONSUCCESS) {
        UT_LOG(Debugk,"Sending request body");
        /* if reply not wanted, arrange for fd to be closed after write. 
         * we don't do this with a simple close() after writing because
         * the write may not finish until later (cb writes as fd is writable) */
        if (!reply_wanted) {
            UT_fd_cntl(fd, UTFD_GET_FLAGS, &f);
            f |= UTFD_CLOSE_AFTER_WRITE;
            UT_fd_cntl(fd, UTFD_SET_FLAGS, f);
        }
        UT_iob_writefd( rqst->io[0], fd);    /* send out the request */
        if (reply_wanted) return;            /* reply wanted, wait for it*/
        /* Here if requester doesn't want the reply. Finish up below. */
    } else {
        /* Socket is readable, meaning the remote host has started to reply. */
        /* Gather response into the result iob until socket closes.          */
        while ( (n = read(fd, readbuf, READ_BUFLEN)) > 0 ) 
            UT_iob_append( rqst->io[1], readbuf, n);

        /* Either n=0 (socket closed) or n=-1 (read error or EWOULDBLOCK).   */
        if (n == -1)  {
            if (errno == EAGAIN) return; /* not an error; get more later*/
            else UT_LOG(Error, "read error %s", strerror(errno));
        }
    }

    /* Reply is fully gathered. Invoke the application calback and clean up. */
    (rqst->cb)(name, rqst->data, flags|UTFD_IS_REQSUCCESS, rqst->io);
    UT_mem_free( UTRQST, rqst, 1);
    /* close the fd only if we gathered the reply, o.w. we closed-after-write */
    if (reply_wanted) {
        UT_fd_unreg(fd);
        close(fd);
    }
}

/******************************************************************************
 * UT_net_request()                                                           *
 * Initiate a request-response network communication over a transient socket. *
 * HTTP is an example of a request-response protocol. This function takes an  *
 * iob to send, and an iob to receive the response. The latter can be NULL    *
 * in which case the socket is closed after sending (no response is gathered).*
 *****************************************************************************/
UT_API int UT_net_request( char *name, UT_net_request_cb*cb, void *data, UT_iob
        *io[2], int flags, ...) { 
    UT_net_request_type *rqst;
    va_list ap;
    int rc;

    va_start(ap, flags);
    rqst = (UT_net_request_type*)UT_mem_alloc(UTRQST, 1);
    memcpy(rqst->io, io, 2*sizeof(UT_iob*));
    rqst->cb = cb;
    rqst->data = data;
    UT_LOG(Debugk,"Initiating request %s", name);
    rc = UT_net_connect2( name, (UT_fd_cb*)UT_net_request_post_cb, rqst, flags,
            ap); 
    va_end(ap);

    return rc;
}

UT_net_request_init() {
    UT_mem_pool_create( UTRQST, sizeof(UT_net_request_type), 10 );
}
