/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <string.h>
#include "libut/ut_internal.h"

#define PCT_MAXLEN 5

/*******************************************************************************
* mem_shl.c                                                                    *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/
extern UT_mem_global_type UT_mem_global;


int UT_mem_shlcmd(   int argc, char *argv[]) {
    UT_mem_pool *pool;
    UT_mem_pool_row *row;
    unsigned bufs_used, bufs_total, i, j, rows_total, rows_nonstd,bkt;
    double kb_used, kb_total;
    double grand_kb_used=0, grand_kb_total=0;
    char *bits;

    switch (argc) {
    case 1:
            UT_shlf("pool                      bufs_used     bufs_total     kb_used     kb_total\n");
            UT_shlf("---------------------------------------------------------------------------\n");
            pool = UT_mem_global.UT_mem_pools_all;
            while (pool) {
                    /* iterate over this pool's rows to collect buffer stats. */
                    bufs_used = 0;
                    bufs_total = 0;
                    for(bkt=0; bkt < UT_MEM_ROW_NUM_BKTS; bkt++) {
                        row = pool->rows[bkt];
                        while (row) {
                                bufs_total += row->num_bufs;
                                for (i=0; i<row->num_bufs;i++)  {
                                        if (BUF_USED(row->usage_bitmap,i)) bufs_used++;
                                }
                                row=row->next;
                        }
                    }

                    /* write out the summary into the shl result */
                    kb_used = bufs_used * pool->buf_size / 1024.0;
                    kb_total = bufs_total * pool->buf_size / 1024.0;
                    grand_kb_used += kb_used;
                    grand_kb_total += kb_total;

                    UT_shlf( "%-26.25s%-14u%-15u%-12.1f%-8.1f\n",
                                    pool->name,
                                    bufs_used,
                                    bufs_total,
                                    kb_used,
                                    kb_total );
                    pool = pool->next;
            }
            UT_shlf( "\n%-55.54s%-12.1f%-8.1f\n", "Total(KB)", grand_kb_used, grand_kb_total );

            if (grand_kb_total >= 1024) {
                UT_shlf("%-55.54s%-12.1f%-8.1f\n", "Total(MB)", 
                        grand_kb_used/1024.0, grand_kb_total/1024.0 );
            }

            return SHL_OK;
            break;
    case 2:
            if (!strcmp(argv[1],"-H")) return UT_hash_dump();

            /* mem -h displays summary information for the hashes */
            if (strcmp(argv[1],"-h") == 0) {
                    UT_shlf("Each hashed list has a number of buckets, each of\n"
                            "which has a number of items in it.  The number of\n"
                            "buckets is automatically increased as needed to try to\n"
                            "keep all buckets from having more than 10 items.\n\n"
                            "The hash function attempts to distribute the items\n"
                            "equally among the buckets, but different data sets\n"
                            "may hash less evenly than this ideal. The hq (hash\n"
                            "quality: 0=best,1=worst) measures how evenly the  \n"
                            "items are distributed among the buckets.        \n\n");
                    UT_hash_shlcmd(argc-1,&argv[1]);
                    return SHL_OK;
            }

            LL_FIND(UT_mem_global.UT_mem_pools_all, pool, argv[1])

            if (!pool) {
                    UT_shlf( "Couldn't find pool %s\n", argv[1]);
                    return SHL_OK;
            }

            /* iterate over this pool's rows to collect buffer stats. */
            bufs_used = 0;
            bufs_total = 0;
            rows_total = 0;
            rows_nonstd = 0;
            for(bkt=0; bkt < UT_MEM_ROW_NUM_BKTS; bkt++) {
                row = pool->rows[bkt];
                while (row) {
                        bufs_total += row->num_bufs;
                        for (i=0; i<row->num_bufs;i++) {
                                if (BUF_USED(row->usage_bitmap,i)) bufs_used++;
                        }
                        rows_total++;
                        if (row->num_bufs != pool->bufs_per_row) rows_nonstd++;
                        row=row->next;
                }
            }

            kb_used = bufs_used * pool->buf_size / 1024.0;
            kb_total = bufs_total * pool->buf_size / 1024.0;

            /* write out the summary into the shl result */
            UT_shlf("This pool has %u buffer row%s\n", rows_total, rows_total > 1 ? "s" : "");
            UT_shlf("Each row has %u buffers (%u exceptions)\n", pool->bufs_per_row, rows_nonstd);
            UT_shlf("Each buffer is %u bytes\n", pool->buf_size);
            UT_shlf("Of %u total buffers (occupying %.1f kb), %u are in use\n", bufs_total, kb_total, bufs_used);
            UT_shlf("In this pool's lifetime there have been %u buffers requests, %u frees\n", pool->allocd_buf_stats, pool->freed_buf_stats );

            return SHL_OK;
            break;
    case 3:
            
            if (strcmp(argv[1],"-d") != 0) {
                    UT_mem_usage(NULL);
                    return SHL_OK;
            }

            LL_FIND(UT_mem_global.UT_mem_pools_all, pool, argv[2])

            if (!pool) {
                UT_shlf( "Couldn't find pool %s\n", argv[1]);
                return SHL_OK;
            }

            for(bkt=0;bkt < UT_MEM_ROW_NUM_BKTS; bkt++) {
                row = pool->rows[bkt];
                while (row) {
                    UT_shlf("Bkt %u Max Extent: %d Row size: %d Usage map: ",
                       bkt,row->max_extent,row->num_bufs);
                    bits = (char*)malloc((row->num_bufs +1) * sizeof(char));
                    if (!bits) UT_LOG(Fatal, "malloc error");
                    for (i=0; i < row->num_bufs; i++) 
                        bits[i] = BUF_USED(row->usage_bitmap,i) ? '1' : '0';
                    bits[i] = '\0';
                    UT_shlf("%s\n", bits);
                    free(bits);
                    row=row->next;
                }
            }
            return SHL_OK;
            break;
    default:
            UT_mem_usage(0,NULL);  /* show usage */
            return SHL_OK;
            break;
    }
}

int UT_mem_usage(int argc,char*argv[]) {
    UT_shlf("Usage:\n");
    UT_shlf("mem              --   memory pool usage summary\n");
    UT_shlf("mem poolname     --   info on a pool\n");
    UT_shlf("mem -d poolname  --   detailed info on a pool\n");
    UT_shlf("mem -h           --   display #items/bucket histogram for each hash\n");
    UT_shlf("mem -H           --   ascii/hex dump of each hash by bucket\n");
}

/*******************************************************************************
 * Traverse the hash tables by getting their addresses from their memory pool, *
 * printing information about the number of buckets having 'm-n' items per bkt.*
 ******************************************************************************/
int UT_hash_shlcmd(   int argc, char *argv[]) {
        UT_hash_table**hashes,**h;
        unsigned bkt_top[] = {10, 50, 100, 500, 1000};
        unsigned bkt_top_sz = 5;
        unsigned bkt_hst[6]; /* one extra */
        unsigned i,sum,cnt,j,found;
        char *cols[] = {" hash rep ptr   ", "hq ", "#buckets", "0-9 ", "10-49", "50-99", "100-499", "500-999", ">999", NULL };
        char **col;
        char str[PCT_MAXLEN];
        int wid;
        double pct;
        UT_hash_handle*hh;
        void *rep; 
        double delta,sum_of_deltas,hash_quality;
        unsigned total_count;

        UT_shl_hdr( cols );
        hashes = (UT_hash_table**)UT_mem_get_buf_map( UTHASHPOOL );
        h=hashes;
        while(*h) {
            /* Iterate over each bucket within this hash, incorp. the bucket sz
             * into the histogram of bucket sizes for this hash. */
            memset(bkt_hst, 0, 6*sizeof(unsigned));
            for(i=0, sum=0; i < (*h)->num_buckets; i++) {
                cnt=0;  /* count items in this bucket */
                hh=(*h)->buckets[i].hh_head;
                while(hh) {cnt++; hh=hh->hh_next;}
                j=0; found=0;
                while (j < bkt_top_sz && !found) {
                        if (cnt < bkt_top[j]) found=1;
                        else j++;
                }
                bkt_hst[j] += cnt;
                sum += cnt;
            }

            /* Scan for the first name we come upon in the hashed list, and use
             * that as a "representative" name for the hash when printing. */
            for(i=0, rep=NULL; i < (*h)->num_buckets && !rep; i++) {
               if ((*h)->buckets[i].hh_head) 
                        rep=(*h)->buckets[i].hh_head->key;
            }

            /* determine the hash quality */
            total_count = 0;
            sum_of_deltas = 0;
            for(i=0; i < (*h)->num_buckets; i++) total_count += (*h)->buckets[i].count;
            for(i=0; i < (*h)->num_buckets; i++) {
                delta = (*h)->buckets[i].count - (total_count * 1.0 / (*h)->num_buckets);
                if (delta < 0) delta *= -1.0; 
                sum_of_deltas += delta;
            }
            hash_quality = sum_of_deltas / (2.0 * total_count);  /* 0=best,1=worst */

            col = cols;
            UT_shlf("%c%-15p ", ((*h)->inhibit_expansion ? '*' : ' '), rep);
            col++;
            UT_shlf("%1.1f ", hash_quality);
            col++;
            UT_shlf("%*u ", strlen(*(col++)), (*h)->num_buckets);

            if (sum > 0) {
                for( j = 0; j < 6; j++) {
                   wid = strlen(*(col++));
                   pct = bkt_hst[j] * 100.0 / sum;

                   if (bkt_hst[j] == 0) str[0] = '\0';
                   else snprintf(str, PCT_MAXLEN, "%.0f%%", pct);

                   UT_shlf("%*s ", wid, str);
                }
            }
            UT_shlf("\n");
            h++;
        }
        UT_shlf("\nHashes prefixed with '*' are expansion-inhibited. (Poor hq w/large # buckets.)\n");
        free(hashes);
        return SHL_OK;
}


int UT_hash_dump() {
        UT_hash_table**hashes,**h;
        UT_hash_handle*hh;
        int i,j,k;
        unsigned char *c;

        hashes = (UT_hash_table**)UT_mem_get_buf_map( UTHASHPOOL );
        h=hashes;
        j=0;
        while(*h) {
            /* Iterate over each bucket within this mod, *
             * and over each item therein, dumping it's id */
            for(i=0; i < (*h)->num_buckets; i++) {
                hh=(*h)->buckets[i].hh_head;
                UT_shlf("[%d][%d]: ",j, i);
                while(hh) {
                    c = (unsigned char*)hh->key;
                    for(k=0;k<hh->keylen;k++) {
                        /* show printable ascii literally, else two-digit hex */
                        if (c[k] >= 0x20 && c[k] <= 0x7e) UT_shlf("%c", c[k]);
                        else UT_shlf("\\%02x", (unsigned)c[k]);
                    }
                    UT_shlf(" ");
                    hh=hh->hh_next;
                }
                UT_shlf("\n");
            }

            h++;
            j++;
        }
        free(hashes);
        return SHL_OK;
}

int UT_mem_init_shl( void ) {
        UT_shl_cmd_create( "mem", "memory pool usage summary",
                (UT_shlcmd *)UT_mem_shlcmd, (UT_helpcb*)UT_mem_usage);
        return 0;
}

