/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
* mem.c                                                                        *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include "libut/ut_internal.h"


#define MALLOC_FAILED 1
#define BAD_POOL_NAME 2
#define BUF_SCAN_FAILED 3

extern int UT_mem_init_shl(void);

/* A global structure used to keep the master memory list */
UT_mem_global_type UT_mem_global = { 
    .UT_mem_pools_all = NULL
};

/******************************************************************************
 * UT_mem_init()                                                              *
 * Most importantly, this creates the first memory pool: ut_hash_tbl.         *
 * As this will be the first element in the global memory pool list, the hash *
 * on the list will also be created as a side effect. Here's the catch: the   *
 * hash table will be created in the very memory pool we're creating.         *
 * To avoid the chicken-and-egg problem, the macro used to find the pool      *
 * resorts to a linear scan of the memory pools list when the hash is not yet *
 * created. So despite the non-existence of the hash, it can find the pool to *
 * create the hash in.  This is really the only tricky part of bootstrapping. *
 *****************************************************************************/
void UT_mem_init() {
    UT_mem_pool_create( UTHASHPOOL, sizeof(UT_hash_table), 10 );
    UT_mem_init_shl();
}

/******************************************************************************
 * UT_malloc_failure()                                                        *
 *****************************************************************************/
void UT_malloc_failure( int error) {
    switch (error) {
        case MALLOC_FAILED:
            UT_LOG( Fatal, "Malloc failure. Stopping.");
            break;
        case BAD_POOL_NAME:
            UT_LOG( Fatal, "Bad pool name in buf alloc. Stopping.");
            break;
        case BUF_SCAN_FAILED:
            UT_LOG( Fatal, "Buf scan failure. Stopping.");
            break;
    }

    exit(-1);
}

/******************************************************************************
 * UT_add_mem_pool_row()                                                      *
 * The row is created with eiether min_bufs or the standard bufs-per-row for  *
 * this pool, whichever is greater.                                           *
 *****************************************************************************/
UT_mem_pool_row *UT_add_mem_pool_row( UT_mem_pool *pool, unsigned min_bufs ) {
    UT_mem_pool_row *row;
    unsigned num_bufs,num_chars,i,bkt;

    num_bufs = ( pool->bufs_per_row > min_bufs ? pool->bufs_per_row : min_bufs);

    if (!(row = (UT_mem_pool_row *)malloc( sizeof( UT_mem_pool_row ))))
        UT_malloc_failure( MALLOC_FAILED );

    row->num_bufs = num_bufs;
    row->max_extent = num_bufs;
    row->max_extent_idx = 0;
    row->next = NULL;

    /* Allocate the usage bitmap for this row (tracks which bufs are in use) */
    num_chars = (num_bufs >> 3) + ((num_bufs & 7) ? 1 : 0); /* ceil(num_bufs/8)*/
    if (!(row->usage_bitmap = (char *)malloc(num_chars * sizeof(char)))) {
        free(row);
        UT_malloc_failure( MALLOC_FAILED );
    }
    for (i=0; i < num_chars; i++) row->usage_bitmap[i] = 0;
    
    /* Allocate the data buffers for this row */
    if (!(row->data = malloc(num_bufs * pool->buf_size ))) {
        free(row->usage_bitmap);
        free(row);
        UT_malloc_failure( MALLOC_FAILED );
    }

    /* find bucket this row belongs in, and prepend to that list. */
    GET_BKT(bkt, row->max_extent);
    row->next = pool->rows[bkt];
    pool->rows[bkt] = row;
    return row;
}

/******************************************************************************
 * UT_mem_pool_create()                                                          *
 *****************************************************************************/
UT_API void UT_mem_pool_create( char *poolname, int buf_size, int exhaust_incr ) {
    UT_mem_pool *pool, *pooltemp;
    int i;

    if (!(pool = (UT_mem_pool *)malloc(sizeof(UT_mem_pool))))
        UT_malloc_failure( MALLOC_FAILED );

    /* Set up the basic information for this pool, name etc. */     
    UT_strncpy( pool->name, poolname, POOL_NAME_MAX_LEN);
    pool->buf_size = buf_size;
    pool->bufs_per_row = exhaust_incr;
    pool->allocd_buf_stats = 0;
    pool->freed_buf_stats = 0;
    for(i=0; i<UT_MEM_ROW_NUM_BKTS; i++) pool->rows[i] = NULL;
    pool->next = NULL;

    /* Set up an initial row of memory buffers for this pool. */
    UT_add_mem_pool_row( pool, exhaust_incr );

    /* This pool is all set up. Add it to the global list of pools. */
    HASH_ADD_STR( UT_mem_global.UT_mem_pools_all, pooltemp, name, pool);
}

/******************************************************************************
 * UT_update_max_extent()                                                     *
 * Following an alloc or a free we need to update the max extent in the row   *
 *****************************************************************************/
int UT_update_max_extent( UT_mem_pool *pool, UT_mem_pool_row *row ) {
    unsigned mark, i, start, end, max_ext, max_ext_idx,old_bkt,new_bkt;
    UT_mem_pool_row *tmp;

    mark = 0;
    start = 0;
    end = 0;
    max_ext = 0;
    max_ext_idx = 0;

    GET_BKT(old_bkt,row->max_extent);

    /* in this logic below, mark means we found an open buf and are scanning
     * for contiguous buffers. if the mark is 0, we haven't found an open buf. */
    for (i=0; i < row->num_bufs; i++) {
        if (! BUF_USED(row->usage_bitmap,i)) {
            if (mark) end = i;
            else {
                start = i;
                end = i;
                mark = 1;
            }
        } else if (mark) {
            if (max_ext < (end-start+1)) {
              max_ext = end-start+1;
              max_ext_idx = start;
            }
            mark = 0; 
        }
    }

    if (mark) { 
        if (max_ext < (end-start+1)) {
           max_ext = (end-start+1);
           max_ext_idx = start;
        }
    }

    row->max_extent     = max_ext;
    row->max_extent_idx = max_ext_idx;

    /* relocate the row to a different hash bucket if necessary. */
    GET_BKT(new_bkt,row->max_extent);
    if (new_bkt != old_bkt) {
        LL_DEL(pool->rows[old_bkt], tmp, row);
        row->next = pool->rows[new_bkt];
        pool->rows[new_bkt] = row;
    }
}




/******************************************************************************
 * UT_mem_pool_free()                                                         *
 *****************************************************************************/
void UT_mem_pool_free( char *poolname ) {
    UT_mem_pool *pool,*temp;
    UT_mem_pool_row *row,*rowtemp;
    unsigned bkt;

    HASH_FIND_STR( UT_mem_global.UT_mem_pools_all, pool, name, poolname);
    if (!pool) UT_LOG(Fatal, "bad pool name in buf alloc [%s]", poolname);

    /* Free up all the rows in this pool */
    for(bkt=0; bkt < UT_MEM_ROW_NUM_BKTS; bkt++) {
        row = pool->rows[bkt];
        while (row) {
            rowtemp = row->next;
            free (row->data);
            free(row);
            row = rowtemp;
        }
    }
    HASH_DEL( UT_mem_global.UT_mem_pools_all, temp, pool );
    free( pool );
}

/******************************************************************************
 * UT_mem_alloc()                                                             *
 * Allocate a contiguous number of buffers from the given pool. In principle  *
 * this is just a matter of scanning the pool's rows for one having enough    *
 * contiguous free buffers. To do this more efficiently, since the number of  *
 * rows could grow very large, the rows are stored in a hash. The max_extent  *
 * (greatest number of free contiguous buffers for a row) determines which    *
 * bucket the row is in. So to allocate a number of bufs, we jump straight to *
 * that bucket, or the higher buckets (having greater max extents) to find a  *
 * suitable row. Finally, if no sufficient row is found, we make a new one.   *
 *****************************************************************************/
UT_API  void *UT_mem_alloc( char *poolname, int bufs_rqstd ) {
    UT_mem_pool *pool;
    UT_mem_pool_row *row;
    int start,end;
    unsigned bkt,b,i;
    void *bufaddr;

    if (bufs_rqstd == 0) {
        UT_LOG(Error, "Bad mem allocation request for 0 buffers");
        return NULL;
    }

    HASH_FIND_STR(UT_mem_global.UT_mem_pools_all, pool, name, poolname);
    if (!pool) UT_LOG(Fatal, "bad pool name in buf alloc [%s]", poolname);

    /* Scan buckets for a row with sufficient free bufs. */
    GET_BKT(bkt,bufs_rqstd);
    for(b=bkt; b < UT_MEM_ROW_NUM_BKTS; b++) {
        row = pool->rows[b];
        while (row) {
            if (row->max_extent >= bufs_rqstd) {
                start = row->max_extent_idx;
                end   = row->max_extent_idx + bufs_rqstd - 1;
                for (i=start; i <= end; i++) {
                    BUF_SET(row->usage_bitmap,i);
                    pool->allocd_buf_stats++;
                }
                UT_update_max_extent(pool, row);
                bufaddr = ( (char*)row->data + start*pool->buf_size );
                return bufaddr;
            }
            row = row->next;
        }

        /* if we failed to find a row in the last bucket, make new row. */
        if (b == UT_MEM_ROW_NUM_BKTS - 1) {
            UT_add_mem_pool_row(pool, bufs_rqstd);
            b = bkt - 1;   /* 'for' loop will b++; new row will be found */
        }
    }
    /* not reached */
}

/******************************************************************************
 * UT_mem_free()                                                              *
 *****************************************************************************/
UT_API void UT_mem_free( char *poolname, void *buf, int num_bufs ) {
    UT_mem_pool *pool;
    UT_mem_pool_row *row;
    void *lo,*hi;
    unsigned i,bkt;

    HASH_FIND_STR(UT_mem_global.UT_mem_pools_all, pool, name, poolname);
    if (!pool) UT_LOG(Fatal, "bad pool name in buf alloc [%s]", poolname);

    /* figure out what row this buf belongs to. this could be slow; lin-scan. */
    for(bkt=0; bkt < UT_MEM_ROW_NUM_BKTS; bkt++) {
        row = pool->rows[bkt];
        while (row) {
            /* compute the lowest/highest struct start addr in row */
            lo = row->data;
            hi = (char*)row->data + (row->num_bufs-1) * pool->buf_size;
            if (buf >= lo && buf <= hi ) {
                /* buf is in this row. find its index */
                for (i=0; i < row->num_bufs; i++ ) {
                    if (buf == ((char*)lo + i*(pool->buf_size))) {
                        if (! BUF_USED(row->usage_bitmap,i)) {
                            UT_LOG(Error, "Freed free buf.");
                        }
                        /* free the buf and advance to next */
                        BUF_CLR(row->usage_bitmap,i);
                        pool->freed_buf_stats++;
                        if (--num_bufs) {
                            buf = (char*)buf + pool->buf_size;
                        } else {
                            UT_update_max_extent(pool, row);
                            return;
                        }
                    }
                }
            } else row = row->next;
        }
    }

    /* if we get here, we didn't find the buf in any rows  */
    UT_LOG( Error, "Can't free unallocated buf from %s", poolname);
}

/* Find the memory pool, and create a malloc'd array of pointers
 * to all of the pool's in-use buffers. The caller must free() 
 * the pool when done. This function should be used very carefully
 * because the buffer pointers are only valid until something 
 * modifies that pool, i.e. use these pointers only as long as
 * you know the pool hasn't been modified yet. It is best to do
 * whatever needs to be done on them and then immediately free
 * the buffer pointers and then return, keeping no references. */
void**UT_mem_get_buf_map(char *poolname) {
    UT_mem_pool *pool;
    UT_mem_pool_row *row;
    unsigned i,bkt,bufs_used=0;
    void **buflist,**b;

    HASH_FIND_STR(UT_mem_global.UT_mem_pools_all, pool, name, poolname);
    if (!pool) return NULL;
     
    /* Loop over the rows, counting the # of bufs in use */
    for(bkt=0; bkt < UT_MEM_ROW_NUM_BKTS; bkt++) {
        row = pool->rows[bkt];
        while (row) {
            for (i=0; i<row->num_bufs;i++) 
                    if (BUF_USED(row->usage_bitmap,i)) bufs_used++;
            row=row->next;
        }
    }

    buflist = (void**)malloc( (bufs_used+1) * sizeof(void*) );
    b=buflist;
    if (!b) { UT_LOG(Fatal,"Malloc failure"); }

    /* Loop over the rows again, this time saving ptrs to bufs */
    for(bkt=0; bkt < UT_MEM_ROW_NUM_BKTS; bkt++) {
        row = pool->rows[bkt];
        while (row) {
            for (i=0; i<row->num_bufs;i++) 
                    if (BUF_USED(row->usage_bitmap,i)) 
                       *b++ = (char*)row->data + i*(pool->buf_size);
            row=row->next;
        }
    }
    *b++ = NULL; 
    return buflist;
}


/****************************************************************************
 * UT_hash_rescale()                                                        *
 * Increase a hash's # of buckets if necessary to try to keep items per bkt *
 * w/in threshold. If a bucket exceeds threshold, double the num of buckets *
 * and re-distribute the items among the new buckets ("re-bucket" the items)*
 ***************************************************************************/
int UT_hash_rescale(UT_hash_table *h) {
    UT_hash_handle*hh,*next_hh,*lst_hh;
    unsigned rescale,i,j,old_numbkts,total_count,cnt;
    UT_hash_bucket *old_buckets, *new_bucket;
    double delta,sum_of_deltas,hash_quality;

    UT_prf_bgn("libut","prf-rescale");

    /* determine the hash quality to decide whether to increase # buckets */
    total_count = 0;
    sum_of_deltas = 0;
    for(i=0; i < h->num_buckets; i++) total_count += h->buckets[i].count;
    for(i=0; i < h->num_buckets; i++) {
        delta = h->buckets[i].count - (total_count * 1.0 / h->num_buckets);
        if (delta < 0) delta *= -1.0; 
        sum_of_deltas += delta;
    }
    hash_quality = sum_of_deltas / (2.0 * total_count);  /* 0=best,1=worst */

    UT_LOG(Debugk, "Hash quality is %f (0=best,1=worst)", hash_quality);

    /* Give up on expanding this hash if its distribution is poor and the
     * number of buckets is already large. There's no theoretical reason for
     * the specific choice of hq=0.5 and #bkts=65535 as the cutoff. */
    if (hash_quality > 0.5 && h->num_buckets >= 65536) {
        rescale = 0;
        UT_LOG(Info, "Hash expansion inhibited; see mem -h");
        h->inhibit_expansion=1;
    } else {
        for(i=0,rescale=0; i < h->num_buckets && !rescale; i++) {
            cnt=0;  /* count items in this bucket */
            hh=h->buckets[i].hh_head;
            while(hh) {cnt++; hh=hh->hh_next;}
            if (cnt > UT_HASH_RESCALE_THRESHOLD) {
               /* Form a idea of how many buckets we want after scaling this hash */
               rescale= (cnt * h->num_buckets / UT_HASH_RESCALE_THRESHOLD);
               j=0;  /* make it a power of two */
               while (rescale >> j) j++;
               rescale = 1 << j;   
            }
        }
    }

    if (rescale) {
       old_numbkts = h->num_buckets;
       old_buckets = h->buckets;

       h->num_buckets = rescale;  /* no max #bkts.. hmm */
       h->buckets = malloc((h->num_buckets) * sizeof(struct UT_hash_bucket));
       if (! h->buckets) { UT_LOG( Fatal, "malloc failure" ); }
       memset(h->buckets, 0, (h->num_buckets) *sizeof(struct UT_hash_bucket));

       /* Traverse the buckets, putting every item in each bucket into a new one */
       for(i=0; i < old_numbkts; i++) {
          hh = old_buckets[i].hh_head; /* traverse hash handles in bucket[i] */
          while (hh) {
             next_hh = hh->hh_next;
             hh->hh_next   = NULL;
             hh->bkt_next = NULL;
             hh->tbl->key = hh->key;
             hh->tbl->keylen = hh->keylen;
             hh->tbl->bkt = 0;
             UT_HASH( hh->tbl->key, hh->tbl->keylen, hh->tbl->num_buckets, hh->tbl->bkt);

             new_bucket = &(hh->tbl->buckets[ hh->tbl->bkt ]);
             new_bucket->count++;
             if (new_bucket->link_head) {
                /* find last hh in the eew bucket and append */
                lst_hh = new_bucket->hh_head;
                while (lst_hh->hh_next) lst_hh=lst_hh->hh_next;
                lst_hh->bkt_next = hh->link;
                lst_hh->hh_next   = hh;
             } else {
                new_bucket->link_head = hh->link;
                new_bucket->hh_head = hh;
             }

             hh = next_hh;
          }
       }

       UT_LOG(Debug, "Rescaled hash from %d to %d buckets", old_numbkts, h->num_buckets);
       free( old_buckets );
    }

    UT_prf_end("libut","prf-rescale");
    return rescale;
}


