/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef UT_H
#define UT_H 

#include <sys/types.h>  /* for pid_t */
#include <netinet/in.h> /* INET_ADDRSTRLEN */
#include <unistd.h>     /* size_t */

#if defined __cplusplus
extern "C" {
#endif


/* Linked lists and hashes */
#include "libut/ll.h"
#include "libut/hash.h"

/* initialization flags */
#define INIT_UNUSED      1
#define INIT_SHL_IPPORT  2
#define INIT_LOGFILE     3
#define INIT_LOGLEVEL    4
#define INIT_ARGCV       5
#define INIT_JOBNAME     6
#define INIT_CONFIGFILE  7
#define INIT_DETACH      8
#define INIT_SIGNALS(...) INIT_SIGNAL_BGN,__VA_ARGS__,INIT_SIGNAL_END
#define INIT_SIGNAL_BGN  9
#define INIT_SIGNAL_END 10
#define INIT_END         0

/* UT_fd flags */
#define UTFD_PIPE              (1 << 0)
#define UTFD_SOCKET            (1 << 1)
#define UTFD_OTHER             (1 << 2)
#define UTFD_R                 (1 << 3)
#define UTFD_W                 (1 << 4)
#define UTFD_SOCKET_LISTENING  (1 << 5)
#define UTFD_SOCKET_ACCEPTED   (1 << 6)
#define UTFD_SOCKET_CONNECTING (1 << 7)
#define UTFD_SOCKET_CONNECTED  (1 << 8)
#define UTFD_SOCKET_LOCALADDR  (1 << 9)
#define UTFD_SOCKET_REMOTEADDR (1 << 10)
#define UTFD_IS_READABLE       (1 << 11)
#define UTFD_IS_WRITABLE       (1 << 12)
#define UTFD_IS_NEWACCEPT      (1 << 13)
#define UTFD_IS_CONSUCCESS     (1 << 14)
#define UTFD_IS_CONFAILURE     (1 << 15)
#define UTFD_IS_REQSUCCESS     (1 << 16)
#define UTFD_IS_REQFAILURE     (1 << 17)
#define UTFD_WRITE_PENDING     (1 << 18)
#define UTFD_GET_FLAGS         (1 << 19)
#define UTFD_SET_FLAGS         (1 << 20)
#define UTFD_GET_DATA          (1 << 21)
#define UTFD_SET_DATA          (1 << 22)
#define UTFD_GET_AUX           (1 << 23)
#define UTFD_SET_AUX           (1 << 24)
#define UTFD_CLOSE_AFTER_WRITE (1 << 25)
#define UTFD_CLOSED_IN_CB      (1 << 26)

/* flags for UT_net_connect */
#define UT_CONNECT_FROM_IPPORT    (1 << 0)
#define UT_CONNECT_TO_IPPORT      (1 << 1)

#define UT_IPPORT_MAXLEN (INET_ADDRSTRLEN + 6)

/* The log levels. These are Fatal, Error, Warning, Info, Debug and Debugk. */
#define LOGLEVELS EL(Fatal) EL(Error) EL(Warning) EL(Info) EL(Debug) EL(Debugk) 
#define EL(x) x,
typedef enum UT_loglevel { LOGLEVELS } UT_loglevel;
#undef EL

/* The supported prf scales. The integer bucket count is used internally.*/
#define PRF_SCALES EL(te_log_1u_10s, 10)        \
                   EL(te_10u_100u, 11)          \
                   EL(te_100u_1m, 11)           \
                   EL(te_log_1m_10s, 6)         \
                   EL(te_minute, 8)             \
                   EL(te_hour, 8)               \
                   EL(te_non_1s_15min, 8) 
#define EL(x,y) x,
typedef enum UT_prf_scale { PRF_SCALES } UT_prf_scale;
#undef EL

/* Supported shl var types */
#define SHL_VAR_TYPES EL(UT_var_int)     \
                      EL(UT_var_double)  \
                      EL(UT_var_string)
#define EL(x) x,
typedef enum UT_var_type { SHL_VAR_TYPES } UT_var_type;
#undef EL
typedef enum UT_var_restrict_type { 
    range, 
    strenum, 
    readonly 
} UT_var_restrict_type;

#define UT_JOB_VARNAME "ut_jobname"

#define UT_LOG( level, msg, ...) UT_log( level, __func__, __FILE__, __LINE__, \
                                         msg, ##__VA_ARGS__ )
#define UT_API 

typedef struct UT_iob {
        size_t len;
        char *buf;
        struct UT_iob *next;
} UT_iob;

/* auxillary information stored for socket fd's */
struct UT_fd_aux_socket {
    char local_ipport[UT_IPPORT_MAXLEN];
    char remote_ipport[UT_IPPORT_MAXLEN];
};

/* container for various kinds of aux info (just sockets for now) */
typedef union {
    struct UT_fd_aux_socket socket;
} UT_fd_aux_type;

/* Prototypes for application-supplied callbacks */
typedef int (UT_fd_cb)(int fd, char *name, int flags, void *data);
typedef int (UT_tmr_cb)(char *name, unsigned msec, void *data);
typedef int (UT_coproc)(char *name, void *data);
typedef int (UT_exithdlr)(pid_t pid, int sts, char *name, void *data);
typedef int (UT_var_upd_cb)(char *name, void *data);
typedef int (UT_net_request_cb)(char *name, void *data, int flags, UT_iob*io[2]);
typedef int (UT_signal_cb)(int signum);
typedef int (UT_shlcmd)(int argc,char *argv[]);
typedef int (UT_helpcb)(int argc, char *argv[]);

/* Public prototypes */
extern UT_API int UT_fd_reg(int fd, char *name, UT_fd_cb *cb, void *data, int flags, ...);
extern UT_API int UT_fd_unreg(int fd);
extern UT_API int UT_fd_write(int fd, void *buf, size_t len);
extern UT_API int UT_fd_cntl(int fd, int flags, ...);
extern UT_API void UT_log(UT_loglevel level, const char *func, char *file, unsigned line_num, char *msg, ...);
extern UT_API void UT_mem_pool_create(char *poolname, int buf_size, int exhaust_incr);
extern UT_API void*UT_mem_alloc(char *poolname, int bufs_rqstd);
extern UT_API void UT_mem_free(char *poolname, void *buf, int num_bufs);
extern UT_API void UT_prf_create(char *prf_name, char *desc, UT_prf_scale t);
extern UT_API void UT_prf_bgn(char *prf_name, char *row_name);
extern UT_API void UT_prf_end(char *prf_name, char *row_name);
extern UT_API int UT_init(int opt1, ...);
extern UT_API void UT_event_loop(void);
extern UT_API void UT_event_once(unsigned wait_msec);
extern UT_API void UT_tmr_set(char *name, unsigned msec, UT_tmr_cb *cb, void *data);
extern UT_API int UT_tmr_clear(char *name);
extern UT_API int UT_net_listen(char *name, char *ipport, UT_fd_cb *cb, void *data );
extern UT_API int UT_net_listen_close(int fd);  
extern UT_API int UT_net_connect(char*name,UT_fd_cb*cb,void *data, int flags,...);
extern UT_API char*UT_net_resolve(char *hostport);   
extern UT_API int UT_net_request( char *name, UT_net_request_cb*cb, void *data, UT_iob *io[2], int flags, ...);
extern UT_API int UT_stridx(char *str, char *list[]);
extern UT_API int UT_var_create(char *name, char *desc, UT_var_type type, ...);
extern UT_API int UT_var_set(char *name, ...);
extern UT_API int UT_var_get(char *name, void *val);
extern UT_API int UT_var_reg_cb(char *name, UT_var_upd_cb *cb, void *data);
extern UT_API int UT_var_bindc(char *name, void *cvar);
extern UT_API int UT_var_restrict(char *name, UT_var_restrict_type type, ...);
extern UT_API UT_iob *UT_iob_create(void);
extern UT_API int UT_iob_printf(UT_iob *iob, char *fmt, ...);
extern UT_API int UT_iob_append(UT_iob *iob, void *b, size_t n );
extern UT_API int UT_iob_free(UT_iob *iob);
extern UT_API int UT_iob_writefd(UT_iob *iob, int fd);
extern UT_API int UT_iob_bufcmp(UT_iob *iob, char *buf, size_t buflen);
extern UT_API char* UT_iob_flatten(UT_iob *iob, size_t *out_len);
extern UT_API size_t UT_iob_len(UT_iob *iob);
extern UT_API pid_t UT_fork_coprocess(char *name, UT_coproc *fcn, void *data, UT_exithdlr *exhdlr, int fds[2]);
extern UT_API int UT_shl_cmd_create(char*name,char*desc,UT_shlcmd*cb,UT_helpcb*hb);
extern UT_API int UT_shl_hdr(char *cols[]);
extern UT_API int UT_shlf(char *fmt, ...);
extern UT_API int UT_shle(char *fmt, ...);
extern UT_API char* UT_strncpy(char *dst, char *src, size_t n);
extern UT_API int UT_signal_reg( UT_signal_cb *cb);

/* return codes for shlcmds */
#define SHL_OK     0 
#define SHL_ERROR -1

#if defined __cplusplus
    }
#endif

#endif /* UT_H */

