/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
 *  Command shell support                                                      *
 ******************************************************************************/


#include <netinet/in.h>  /* in_addr_t */
#include <sys/types.h>   /* size_t */
#include <stdarg.h>      /* va_list */

#define NUM_BOOT_TMP_CMDS 10

#define SHL_SESSION_POOL "ut_shl_session"
#define SHL_CMD_POOL "ut_shl_cmd"
#define SHL_IPPORT_VARNAME "ut_control_port"
#define SHL_IPPORT_DEFAULT "127.0.0.1:4445"
#define SHL_BASEDIR_VARNAME "ut_basedir"

#define SHLCMD_MAXLEN 20
#define SHLCMD_DESC_MAXLEN 80

#define SHL_PORT_MAX_INPUT 100


typedef struct UT_shl_cmd {
  char name[SHLCMD_MAXLEN];
  char desc[SHLCMD_DESC_MAXLEN];
  UT_shlcmd *cb;
  UT_helpcb *helpcb;
  struct UT_shl_cmd *next;
} UT_shl_cmd;

/* Function signatures for glue functions */
typedef int UT_shl_lang_init_fcn(struct UT_shl_cmd *cmds);
typedef int UT_shl_lang_addcmd_fcn(struct UT_shl_cmd *cmd);
typedef int UT_shl_lang_eval_fcn(char *text);
typedef int UT_shl_lang_eval_file_fcn(char *filename);
typedef int UT_shl_lang_result_to_iob_fcn(UT_iob *outiob);


/*******************************************************************************
* struct UT_shl_lang_glue                                                      *
* One of these structs exists for each supported "glue", i.e. language binding *
* for the command shell. Commands entered in the shell are interpreted in the  *
* installed shell language via the "glue" function pointers herein.            *
*******************************************************************************/
typedef struct UT_shl_lang_glue {
    char *name; /* static */
    UT_shl_lang_init_fcn *init;
    UT_shl_lang_addcmd_fcn *addcmd;
    UT_shl_lang_eval_fcn *eval;
    UT_shl_lang_eval_file_fcn *feval;
    UT_shl_lang_result_to_iob_fcn *toiob,*errtoiob;
} UT_shl_lang_glue;

typedef struct UT_shl_session {
  int fd;
  UT_iob *in,*out,*err;
  struct UT_shl_session *next;
} UT_shl_session;

/* A global instance keeps track of all shl state */
typedef struct UT_shl_global_type {
  int inited;
  int listen_fd;
  UT_shl_session *sessions;
  UT_shl_session *active_session;
  UT_shl_cmd *cmds;
  UT_shl_lang_glue *glue;
} UT_shl_global_type;

/* Prototypes */

void UT_shl_init(char *ipport);
int UT_shl_install_lang(char *name);
int UT_shl_eval(char*in);
char** UT_shl_build_argv(char *in,int*argc);
UT_shl_session* UT_shl_create_session(int fd);
int UT_shl_close_session(UT_shl_session*);
int UT_shl_open_port(char *ipport);
int UT_shl_close_port(void);
int UT_shl_upd_port(char *name,void *data);
int UT_shl_io_cb(int fd,void*,void*,int);
int UT_shl_help_cmd(int argc,char *argv[]);
int UT_shl_exit_cmd(int argc,char *argv[]);
int UT_shl_ipport_parse(char*str,in_addr_t *ip_out,int *port_out);

/* Macros used in the initialization of the glue */
#define GLUE_SHL .name ="shl",\
                 .init =NULL,\
                 .addcmd =NULL,\
                 .eval =NULL,\
                 .feval =NULL,\
                 .toiob =NULL,\
                 .errtoiob =NULL
#define GLUE_NULL .name =NULL,\
                  .init =NULL,\
                  .addcmd =NULL,\
                  .eval =NULL,\
                  .feval =NULL,\
                  .toiob =NULL,\
                  .errtoiob =NULL
#define GLUES {GLUE_SHL},{GLUE_NULL}



/*****************************************************************************
 * UT_clproc                                                                 * 
 * Command-line-procedure support, used in implementing shl commands. The    *
 * cmdtable maps a list of names to corresponding procedures. The procs      *
 * have the UT_clproc prototype.                                             *
 ****************************************************************************/
typedef int UT_clproc(int argc, char *argv[]);
typedef struct UT_cmdtable {
                char *name;
                UT_clproc *proc;
} UT_cmdtable;

extern UT_clproc *UT_dispatch_subcmd(char *name, UT_cmdtable *c);
extern char* UT_dsprintf(char *, va_list ap);
