/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
* init.c                                                                       *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/
static const char id[]="$Id: init.c,v 1.19 2005/11/07 02:05:26 thanson Exp $";

#include <time.h>
#include <stdarg.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include "libut/ut_internal.h"

extern UT_loop_global_type UT_loop_global;
extern UT_shl_global_type UT_shl_global;
extern char *UT_loglevel_strs[];

/* a scratch area to write a config file name into.  */
static char config_file_buf[CONFIG_FILENAME_MAXLEN];   
static char *config_file = NULL;

/*******************************************************************************
* UT_job_init()                                                                *
* The whole existence of a variable that defines the "job name" is really      *
* for future expansion along the lines of server-to-server messaging           *
*******************************************************************************/
void UT_job_init(char *jobname) {
    UT_var_create(UT_JOB_VARNAME, "job name", UT_var_string, jobname);
}


/*******************************************************************************
* UT_signal_handler()                                                          *
* The signal handler. It is only safe herein to set a flag.                    *
*******************************************************************************/
void UT_signal_handler(int signum) {
    UT_loop_global.sig_flag++;
    sigaddset( &UT_loop_global.sig_set, signum);
}

/*******************************************************************************
* UT_signal_init()                                                             *
* Initialize the signals specified in set to be handled by UT_signal_handler.  *
*******************************************************************************/
void UT_signal_init(sigset_t *set) {
    sigset_t max_set;
    int i;
    struct sigaction sa;

    /* calculate max signal number */
    sigemptyset(&max_set);
    i=1; 
    while (sigaddset(&max_set,i) != -1) i++;
    UT_loop_global.sig_max = i-1;

    sa.sa_handler = UT_signal_handler;
    sigemptyset( &sa.sa_mask );
    sa.sa_flags = 0;

    /* Iterate over signals in the set, setting handler for each one. */
    for (i=1; i < UT_loop_global.sig_max; i++) {
        if (sigismember(set, i) == 1) {
            if (sigaction( i, &sa, NULL ) == -1) {
                UT_LOG(Fatal, "can't catch signal %d (%s)", i, strsignal(i));
            }
        }
    }
}

/*******************************************************************************
* UT_signal_update()                                                           *
* Invoked from the event loop to check for and handle any new signals.         *
*******************************************************************************/
int UT_signal_update() {
    int i;

    if (UT_loop_global.sig_flag == 0) return 0;

    /* Iterate over signals in the set. Log each, invoke cb if app requested. */
    for(i=1; i < UT_loop_global.sig_max; i++) {
        if ( sigismember( &UT_loop_global.sig_set, i) == 1) {
            UT_LOG(Info,"received signal %d (%s)", i, strsignal(i));
            if (UT_loop_global.sig_cb) UT_loop_global.sig_cb(i);
        }
    }

    UT_loop_global.sig_flag = 0;
    sigemptyset( &UT_loop_global.sig_set );
    return 0;
}

/*******************************************************************************
* UT_signal_reg()                                                              *
* Register a callback to be invoked in response to any INIT_SIGNALS(...) sigs. *
*******************************************************************************/
UT_API int UT_signal_reg( UT_signal_cb* cb) {
    UT_loop_global.sig_cb = cb;
    return 0;
}

/******************************************************************************
 * UT_init()                                                                  *
 * Application must call this prior to any other UT functions.                *
 * Initializes the various UT subsystems.                                     *
 *****************************************************************************/
int UT_API UT_init(int opt1, ...) {
    int opt,argc,i,sig,detach=0;
    signed char c;
    char **argv;
    va_list ap;
    char  *shl_ipport     = SHL_IPPORT_DEFAULT;
    UT_loglevel log_level = LOGLEVEL_DEFAULT;
    char *log_file        = LOGFILE_DEFAULT,
         *cfg_file        = CONFIGFILE_DEFAULT,
         *job_name        = UT_JOB_DEFAULT;
    UT_per_loop_cb_t *pscb,*pscb_tmp;
    struct stat stat_buf;
    pid_t pid;
    sigset_t set;

    /* Line-buffer stderr so forked coprocesses send complete lines to the 
       logging pipe. This improves the readability of their log messages.  */
    setvbuf(stderr,NULL,_IOLBF,0); 

    if (sigemptyset(&set) == -1) UT_LOG(Fatal,"sigempty set failed");
    if (sigaddset(&set,SIGPIPE) == -1) UT_LOG(Fatal,"sigaddset SIGPIPE failed");

    /* option processing */
    va_start(ap,opt1);
    for( opt = opt1; opt != INIT_END; opt = va_arg(ap,int) ) {
        switch (opt) {
            case INIT_DETACH:
                detach = 1;
                break;
            case INIT_SHL_IPPORT:
                shl_ipport = va_arg(ap,char *);
                break;
            case INIT_LOGFILE:
                log_file = va_arg(ap,char *);
                break;
            case INIT_LOGLEVEL:
                log_level = va_arg(ap,unsigned);
                break;
            case INIT_ARGCV:
                argc = va_arg(ap, int);
                argv = (char**)va_arg(ap, void*);
                opterr = 0;     /* prevent getopt from writing to stderr */
                while ( (c = getopt(argc, argv, ":p:l:v:j:c:bh")) != -1) {
                    switch (c) {
                        case 'p':
                            shl_ipport = optarg;
                            break;
                        case 'l':
                            log_file = optarg;
                            break;
                        case 'v':
                            i = UT_stridx(optarg,UT_loglevel_strs);
                            if (i >= 0) log_level = i;
                            else UT_LOG(Fatal, "Invalid log level [%s]",optarg);
                            break;
                        case 'j':
                            job_name = optarg;
                            break;
                        case 'c':
                            cfg_file = optarg;
                            break;
                        case 'b':
                            detach = 1;
                            break;
                        case 'h':  /* help */
                        case ':':  /* required param missing */
                            UT_LOG(Fatal,"options: [-p <ip:port>] "
                                "[-l <logfile>] [-v <loglevel>] [-j <jobname>] "
                                "[-c <configfile>] [-b]");
                            break;
                        case '?':
                            UT_LOG(Debug,"non-UT option: %c", (char)optopt);
                            break;
                    }
                }
                break;
            case INIT_JOBNAME:
                job_name = va_arg(ap,char *);
                break;
            case INIT_CONFIGFILE:
                cfg_file = va_arg(ap,char *);
                break;
            case INIT_SIGNAL_BGN:
                while ( (sig = va_arg(ap,int)) != INIT_SIGNAL_END) {
                    if (sigaddset(&set,sig) == -1) 
                        UT_LOG(Fatal,"sigaddset err");
                }
                break;
            default:
                fprintf(stderr,"invalid UT_init option\n");
                exit(-1);
        }
    }
    va_end(ap);

    /* Set up the config filename so we can evaluate it later. 
     * If the config file hasn't been specified, check if the
     * default config file actually exists before allowing it. 
     * If the config file is set to "-" no config file is used. */
    if (!strcmp(cfg_file,CONFIGFILE_DEFAULT)) {
        if (stat(cfg_file, &stat_buf) == -1) cfg_file = "-";
    }
    if (!strcmp(cfg_file, "-")) config_file = NULL;
    else if (strlen(cfg_file) + 1 < CONFIG_FILENAME_MAXLEN) {
        UT_strncpy(config_file_buf, cfg_file, CONFIG_FILENAME_MAXLEN);
        config_file = config_file_buf;
    } else {
        UT_LOG(Fatal,"config file name too long [%s]", cfg_file);
    }

    /* Background mode. See daemon_init, Stevens APUE p418. */
    if (detach) {
        pid = fork();
        if (pid < 0) UT_LOG(Fatal, "fork error: %s", strerror(errno));
        else if (pid != 0) exit(0); /* parent exits */
        setsid();                   /* child becomes session leader */
    }

    /* Set up signal handlers for the signals specified in set. */
    UT_signal_init(&set);

    /* Init the UT subsystems */
    UT_mem_init();
    UT_var_init();
    UT_log_init( log_level, log_file );
    UT_fd_init();
    UT_net_listen_init();
    UT_tmr_init();
    UT_prf_init();
    UT_job_init( job_name );
    UT_coproc_init();
    UT_shl_init( shl_ipport ); 
    UT_iob_init(); 
    UT_net_request_init(); 
    UT_loop_init();

    /* Set the initial time of day */
    gettimeofday( &UT_loop_global.TOD, NULL);

    /* Create a prf to track libut statistics such as loop times.*/
    UT_prf_create( "libut", "libut statistics", te_log_1m_10s );

    /* set up a initialization callback to take place at first loop */
    pscb = (UT_per_loop_cb_t*)UT_mem_alloc(PER_LOOP_CB_POOL,1);
    pscb->cb = UT_first_loop_cb;
    pscb->next = NULL;
    LL_ADD(UT_loop_global.per_loop_cbs, pscb_tmp, pscb);

    return 0;
}

/*******************************************************************************
* UT_first_loop_cb()                                                           *
* This is an initialization callback that's invoked on the very first UT loop. *
* Its main function is to read the runtime configuration file, which has to be *
* held off until now so that the app could define its own variables after      *
* UT_init(). Those variables may be referenced in the runtime config file.     *
*******************************************************************************/
int UT_first_loop_cb() {
    char *ipport;

    UT_LOG(Debugk, "completing final initialization");

    /* source config file */
    if (config_file) UT_shl_eval_file(config_file);

    /* open the shl listener port if needed. May be open already due to the
     * var callback being invoked while parsing the config file. */
    if (UT_shl_global.listen_fd == -1) {
        UT_var_get(SHL_IPPORT_VARNAME , &ipport );
        UT_shl_open_port( ipport );
    }

    return 0;  /* indicates that cb should not be invoked again */
}

