/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/******************************************************************************
 * utfd.c                                                                     *
 * Copyright (c) 2003-2005  Troy Hanson                                       *
 * Functions for registering callbacks for file descriptor read/writability   *
 *****************************************************************************/
static const char id[]="$Id: fd.c,v 1.22 2005/11/07 01:45:58 thanson Exp $";

#include <limits.h>
#include <stdarg.h>
#include <stdlib.h>         /* malloc */
#include <unistd.h>         /* size_t */
#include <errno.h>          /* errno */
#include <string.h>         /* strerror,memcpy */
#include "libut/ut_internal.h"

extern UT_loop_global_type UT_loop_global;

/*******************************************************************************
* UT_fd_reg()                                                                  *
* Register a file-descriptor for select (I/O monitoring) in libut's main loop. *
* If called for an already-registered fd, this function updates the name, cb,  *
* data and flags to the newly-given values. This makes it easy to change a cb  *
* or update the flags (socket connecting --> connected) etc.                   *
*******************************************************************************/
UT_API int UT_fd_reg(int fd, char *name, UT_fd_cb *cb, void *data, 
        int flags, ...) {
    UT_fd *fdx, *tmp;
    va_list ap;
    char *ipport;
    int update;

    /* sanity check; a program that reg's for neither UTFD_R/UTFD_W is broken */
    if (!(flags & UTFD_R || flags & UTFD_W)) 
        UT_LOG(Fatal,"fd %d (%s) registered for neither r/w I/O.", fd, name);

    HASH_FIND_INT( UT_loop_global.fds, fdx, fd, &fd);
    if ( (update = (fdx ? 1 : 0)) == 1) {
        UT_LOG(Debug, "updating registration for fd %d: %s",fd, fdx->name);
    } else fdx = (UT_fd *)UT_mem_alloc( FD_POOL, 1 );

    UT_strncpy( fdx->name, name, FD_NAME_MAXLEN );
    fdx->fd = fd;
    fdx->flags = flags;
    fdx->cb = cb;
    if (!update) fdx->cb_count = 0;
    fdx->data = data;
    fdx->write_pendng = NULL;
    if (!update) memset(&fdx->aux,0,sizeof(fdx->aux));
    if (!update) fdx->next = NULL;

    /* Two varargs are permitted: socket local and remote address. */
    va_start(ap, flags);
    if (flags & UTFD_SOCKET) {
        if (flags & UTFD_SOCKET_LOCALADDR) {
            if ( (ipport = va_arg(ap,char*)) != NULL ) 
              UT_strncpy(fdx->aux.socket.local_ipport,ipport,UT_IPPORT_MAXLEN);
        }
        if (flags & UTFD_SOCKET_REMOTEADDR) {
            if ( (ipport = va_arg(ap,char*)) != NULL ) 
              UT_strncpy(fdx->aux.socket.remote_ipport,ipport,UT_IPPORT_MAXLEN);
        }
    }
    va_end(ap);

    if (!update) {
        HASH_ADD_INT(UT_loop_global.fds, tmp, fd, fdx );
    }
    return 0;
}

/*******************************************************************************
* UT_fd_unreg()                                                                *
* Instructs UT to stop monitoring the fd for I/O.                              *
* The caller is still responsible for closing the fd when desired.             *
*******************************************************************************/
UT_API int UT_fd_unreg(int fd) {
     UT_fd *fdx, *tmp;

     HASH_FIND_INT( UT_loop_global.fds, fdx, fd, &fd);
     if (!fdx) {
         UT_LOG(Error, "failed to unregister descriptor %d", fd);
         return -1;
     }

     HASH_DEL(UT_loop_global.fds, tmp, fdx);
     UT_mem_free( FD_POOL, fdx, 1 );
     UT_LOG( Debugk, "unregistered descriptor (%d)", fd);
     return 0;
}


/*******************************************************************************
* UT_fds_shlchmd()                                                             *
* Print a table of the monitored file descriptors with their flag information. *
*******************************************************************************/
int UT_fds_shlcmd( int argc, char* argv[] ) {
    UT_fd *fdx;
    char *type,flg[3]="\0\0\0",rw[3],*loc,*rem;

    char *cols[] = { "fd", "name            ", "type","flg","io-count","md",
                     "local/remote address                ", NULL };

    UT_shl_hdr(cols);
    for(fdx=UT_loop_global.fds; fdx; fdx=fdx->next) {

        if (fdx->flags & UTFD_SOCKET) type="socket";
        else if (fdx->flags & UTFD_PIPE) type="pipe";
        else if (fdx->flags & UTFD_OTHER) type="other";
        else type="?";

        if (fdx->flags & UTFD_SOCKET && fdx->flags & UTFD_SOCKET_LISTENING) *flg='L';
        else if (fdx->flags & UTFD_SOCKET && fdx->flags & UTFD_SOCKET_ACCEPTED) *flg='A';
        else if (fdx->flags & UTFD_SOCKET && fdx->flags & UTFD_SOCKET_CONNECTING) *flg='G';
        else if (fdx->flags & UTFD_SOCKET && fdx->flags & UTFD_SOCKET_CONNECTED) *flg='C';
        else *flg=' ';
        if (fdx->flags & UTFD_WRITE_PENDING) strcat(flg,"W");
        
        rw[0] = (fdx->flags & UTFD_R) ? 'r' : ' ';
        rw[1] = (fdx->flags & UTFD_W) ? 'w' : ' ';
        rw[2] = '\0';

        loc = (fdx->flags & UTFD_SOCKET && fdx->flags & UTFD_SOCKET_LOCALADDR) ?
            fdx->aux.socket.local_ipport : "";
        rem = (fdx->flags & UTFD_SOCKET && fdx->flags & UTFD_SOCKET_REMOTEADDR) ?
            fdx->aux.socket.remote_ipport : "";

        UT_shlf("%2d %-16s %-4.4s %-3s %8d %-2s %s %s\n", fdx->fd, 
                fdx->name, type, flg, fdx->cb_count, rw, loc, rem );
    }

    return SHL_OK;
}

int UT_fd_init() {
    UT_mem_pool_create( FD_POOL, sizeof(UT_fd), 10 );
    UT_shl_cmd_create( "fds", "list selected file descriptors", 
            (UT_shlcmd *)UT_fds_shlcmd, NULL);
}

/*******************************************************************************
* UT_fd_write()                                                                *
* A write that accomodates "delayable" receivers i.e. those that may not be    *
* able to read all the data as fast as we could write it to them w/o blocking. *
* If the buffer isn't fully written immediately, the remaining portion is      *
* appended to the "pending write" output buffer for this fd. As the fd becomes *
* writable (according to select() ) the pending output is written out.         *
*******************************************************************************/
UT_API int UT_fd_write(int fd, void *buf, size_t len) {
    int n,nwritten=0;
    UT_fd *fdx;

    if (len == 0) {
        UT_LOG(Warning, "write of 0-length buffer to fd %d", fd);
        return 0;
    }

    HASH_FIND_INT(UT_loop_global.fds, fdx, fd, &fd);
    if (!fdx) {
        UT_LOG(Error, "Can't find UT_fd for fd %d", fd);
        return -1;
    }

    /* If there is already pending output, just add to it, don't write now */
    if (fdx->flags & UTFD_WRITE_PENDING) {
        if (!fdx->write_pendng) UT_LOG(Fatal, "write pending flag w/null iob");
        UT_iob_append(fdx->write_pendng, (char*)buf, len);
        return 0;
    }

    /* start by trying to write the full buffer */
    if ( (n = write(fd, buf, len)) == -1 ) {
        if (errno != EINTR && errno != EAGAIN) {
            UT_LOG(Error, "Write error to fd %d: %s", fd, strerror(errno));
            return -1;
        }
        n = 0;  /* eintr/eagain */
    }
    nwritten += n;

    /* did we write whole buffer in one fell swoop? */
    if (len-n == 0) {
        if (fdx->flags & UTFD_CLOSE_AFTER_WRITE) {
            UT_fd_unreg(fd);
            close(fd);
        }
        return nwritten;  
    }

    /* buffer wasn't fully written. write remainder as fd becomes writable. */
    fdx->write_pendng = UT_iob_create();
    UT_iob_append(fdx->write_pendng, (char*)buf + n, len-n);
    fdx->flags |= UTFD_WRITE_PENDING;
    return nwritten;
}

/*******************************************************************************
* UT_fd_write_pending()                                                        *
* This function is used internally from within the UT main loop to write out   *
* a pending-output buffer to a file descriptor which has become writable.      *
* It may only succeed in writing a partial buffer which is ok, the process     *
* just repeats until the pending output is completely written.                 *
*******************************************************************************/
int UT_fd_write_pending(UT_fd *fdx) {
    int n;
    UT_iob *iob;
    UT_fd *fdm; /* master UT_fd corresponding to copy fdx */
    void *buf;

    UT_LOG(Debugk, "writing pending output for fd %d", fdx->fd); 

    HASH_FIND_INT(UT_loop_global.fds, fdm, fd, &fdx->fd);
    if (!fdm) {
        UT_LOG(Error, "Failed to find UT_fd for fd %d", fdx->fd);
        return -1;
    }

    /* sanity check first */
    if (!(fdx->flags & UTFD_WRITE_PENDING)) {
        UT_LOG(Fatal, "pending write invoked with inconsistent flags (fd %d)", fdx->fd);
        return -1;
    }

    while ( (iob = fdx->write_pendng) != NULL ) {
        if ( (n = write(fdx->fd, iob->buf, iob->len)) == -1 ) {
            if (errno != EINTR && errno != EAGAIN) {
                UT_LOG(Error, "Write error to fd %d: %s", fdx->fd, strerror(errno));

                /* fdx is a copy of the master flags, so use fdm to
                 * switch the "write pending" bit off in the master. */
                fdm->flags &= ~UTFD_WRITE_PENDING;
                UT_iob_free(iob);
                return -1;
            }
            n = 0;  /* eintr/eagain. nothing written */
            return n;  /* try again later */
        }

        /* handle partial write, by copying the unwritten part of the iob's
         * internal buffer to a newly malloc'd buffer and free'ing the old */
        if (n < iob->len) {
            buf = malloc(iob->len - n);
            if (!buf) UT_LOG(Fatal, "malloc error");
            memcpy(buf, (char*)iob->buf + n, iob->len - n);
            free(iob->buf);
            iob->buf = buf;
            iob->len -= n;
            return n;
        }

        /* A full iob was written. Advance to the next iob. However, we must 
         * advance the master (not just the copy, fdx) because if we write only
         * a partial buffer, it will pick up next time from the master iob. */
        fdx->write_pendng = iob->next;  /* copy */
        fdm->write_pendng = iob->next;  /* master */

        /* free the single "former head" link in the iob chain */
        iob->next = NULL;
        UT_iob_free(iob);               
    }

    /* the pending output has been completely written. set master bit off */
    UT_LOG(Debugk, "completed pending write (fd %d)", fdx->fd);
    fdm->flags &= ~UTFD_WRITE_PENDING;
    fdm->write_pendng = NULL;
    if (fdm->flags & UTFD_CLOSE_AFTER_WRITE) {
        UT_fd_unreg(fdx->fd);
        close(fdx->fd);
        fdx->flags |= UTFD_CLOSED_IN_CB;
    }
    return n;
}

/*******************************************************************************
* UT_fd_cntl()                                                                 *
* This is a catchall inspired by fcntl. Supports get/set fd flags, cb or aux.  *
*******************************************************************************/
UT_API int UT_fd_cntl(int fd, int opflag, ...) {
    va_list ap;
    int *iptr,i;
    UT_fd *fdx;
    void **vpp,*vptr;

    HASH_FIND_INT(UT_loop_global.fds, fdx, fd, &fd);
    if (!fdx) {
        UT_LOG(Error,"failed to UT_fd_cntl: unknown fd %d", fd);
        return -1;
    }

    va_start(ap,opflag);
    switch (opflag) {
        case UTFD_GET_FLAGS:
            iptr = va_arg(ap, int*);
            *iptr = fdx->flags;
            break;
        case UTFD_SET_FLAGS:
            i = va_arg(ap, int);
            fdx->flags = i;
            break;
        case UTFD_GET_DATA:
            vpp = (void**)va_arg(ap,void*);
            *vpp = fdx->data;
            break;
        case UTFD_SET_DATA:
            vptr = va_arg(ap,void*);
            fdx->data = vptr;
            break;
        case UTFD_GET_AUX:
            vptr = va_arg(ap,void*);
            memcpy(vptr, &fdx->aux, sizeof(UT_fd_aux_type));
            break;
        case UTFD_SET_AUX:
            vptr = va_arg(ap,void*);
            memcpy(&fdx->aux, vptr, sizeof(UT_fd_aux_type));
            break;
        default:
            UT_LOG(Error, "unsupported UT_fd_cntl %d (fd %d)", opflag, fd);
            return -1;
    }
    va_end(ap);

    return 0;
}
