/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
 * Coprocess support                                                           *
 ******************************************************************************/
static const char id[]="$Id: coproc.c,v 1.12 2005/10/21 01:59:44 thanson Exp $";

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "libut/ut_internal.h"

extern int UT_coproc_init_shl(void);
extern UT_log_global_type UT_log_global;
extern UT_shl_global_type UT_shl_global;

/* Global encapsulating coprocess state */
UT_coprocess_global_type UT_coproc_global = { 
    .coprocs = NULL,
};

void UT_coproc_init() {
  UT_mem_pool_create( COPROC_POOL, sizeof(UT_coprocess), 10);
  UT_coproc_init_shl();
}

/* Collect a coprocess that we think has exited, trying later if not */
/* This function has this prototype because it can be a timer callback */
int UT_coproc_cleanup(char *tmr_name,unsigned msec,void*pid_in) {
   pid_t pid;
   int sts,i;
   UT_coprocess *coproc,*tmp;

   pid = waitpid((pid_t)pid_in,&sts,WNOHANG);  /* don't block */
   if (pid < 0) { 
     UT_LOG(Error, "waitpid (pid %d) failure: %s", pid_in, strerror(errno)); 
   } else if (pid == 0) {
       /* Child still running. Try again in a bit. Auto-reschedule. */
       return 0;
   } else { /* pid > 0, child exited */
     if (WIFSIGNALED(sts)) {
       UT_LOG(Info,"Pid %d killed by signal %d", pid, WTERMSIG(sts));
     } else if (WIFEXITED(sts)) {
       UT_LOG(Info,"Pid %d exited with status %d", pid, WEXITSTATUS(sts));
     } else {
       UT_LOG(Error,"Pid %d neither exited nor signalled (?)", pid);
     }
     /* clean up UT_coprocess structure */
     LL_FIND_BY_MEMBER(UT_coproc_global.coprocs,coproc,pid,(pid_t)pid_in);
     if (!coproc) {
       UT_LOG(Error, "Internal error: failed to find UT_coprocess for pid %d",pid);
       return -1;
     }
     /* Remove the structure from the linked list */
     LL_DEL(UT_coproc_global.coprocs,tmp,coproc);

     /* close pipes to/from coprocess */
     for (i=0;i<3;i++) close(coproc->fds[i]);

     /* Call the application's coprocess exit handler */
     UT_LOG(Debug,"Invoking exit handler for pid %d (%s)", pid, coproc->name);
     if (coproc->exit_cb) coproc->exit_cb(pid,sts,coproc->name,coproc->data);

     UT_mem_free(COPROC_POOL, coproc, 1);
   }
   return -1;  /* don't auto-reschedule timer */
}

int UT_coproc_logger(int fd, char*name, int flags, void*pid_in) {
  char buf[101],*b,*cleaner="coprocess_collector";
  int rc,i,unwritten;
  while( (rc = read(fd,buf,100)) > 0) {
    b = buf;
    unwritten=0;
    for(i=0;i<rc;i++) {
      if (buf[i] == '\n') {
        buf[i] = '\0';
        UT_LOG(Info,"%s: [%s]", name,b);
        b = &buf[i+1];
        unwritten = 0;
      } else unwritten=1;
    }
    if (unwritten) { /* trailing non-newline-terminated part */
      buf[i] = '\0'; /* ok since buf size is 1>max-read-size */
      UT_LOG(Info,"%s: [%s]*", name, b);
    }
  }

  if (rc == 0 || (rc == -1 && errno != EAGAIN)) {
    /* close(fd); */ /* now doing this in the cleanup fcn */
    UT_fd_unreg(fd);
    if (rc == -1) UT_LOG(Error, "read error %s", strerror(errno));
    UT_LOG(Debug,"%s logging closed.", name);
    /* collect child; if its not yet ready then retry periodically */
    if (UT_coproc_cleanup(cleaner,0,pid_in) >= 0) 
        UT_tmr_set(cleaner,COPROC_WAITPID_INTERVAL,UT_coproc_cleanup,pid_in);
  }
}

/*******************************************************************************
* UT_fork_coprocess()                                                          *
* Forks a child which executes the given function and exits w/its return value.*
* The child will automatically be wait()-ed for after it closes the pipe       *
* tying its stderr back to the parent (which logs it). If thecaller passes a   *
* non-null fds argument, it will be populated with file descriptors to write to*
* and read from the child, respectively. The given exit handler will be invoked*
* after the child exits if non-NULL.                                           *
*******************************************************************************/
UT_API pid_t UT_fork_coprocess(char *name, UT_coproc *fcn, void *data,
                               UT_exithdlr *exhdlr, int fds[2]) {
  UT_coprocess *coproc,*tmp;
  int coproc_stdin[2], coproc_stdout[2], coproc_stderr[2], err, i, rc, fdflags;
  pid_t pid;
  UT_shl_session *s;

  err = pipe( coproc_stdin );
  if (!err) err = pipe( coproc_stdout );
  if (!err) err = pipe( coproc_stderr );

  if (err) {
    UT_LOG(Error,"pipe error %s", strerror(errno));
    return -1;
  }

  if ( (pid = fork()) < 0) { 
    UT_LOG(Error, "fork error %s", strerror(errno));
    for(i=0;i<2;i++) close(coproc_stdin[i]);
    for(i=0;i<2;i++) close(coproc_stdout[i]);
    for(i=0;i<2;i++) close(coproc_stderr[i]);
    return -1;
  } else if (pid == 0) {  /* child */
    /* don't hang on to parent's shl port listener or sessions */
    for(s = UT_shl_global.sessions; s; s= s->next) if (s->fd > 0) close(s->fd);
    if (UT_shl_global.listen_fd != -1) close( UT_shl_global.listen_fd );
    UT_log_global.fd = STDERR_FILENO; /* send any logging to parent log pipe */
    dup2(coproc_stdin[1],STDIN_FILENO);
    dup2(coproc_stdout[1],STDOUT_FILENO);
    dup2(coproc_stderr[1],STDERR_FILENO);
    /* close the pre-dup fd and parent end */
    for(i=0;i<2;i++) close(coproc_stdin[i]);
    for(i=0;i<2;i++) close(coproc_stdout[i]);
    for(i=0;i<2;i++) close(coproc_stderr[i]);
    rc = fcn(name,data);
    exit(rc);
  } 

  /* parent here */
  close(coproc_stdin[1]);
  close(coproc_stdout[1]);
  close(coproc_stderr[1]);

  /* coproc_stdin[0] can be used to write to the child.  */
  /* coproc_stdout[0] can be used to read from the child.*/
  /* These two are going to be returned to the caller. */
  if (fds) {
     fds[0] = coproc_stdin[0];
     fds[1] = coproc_stdout[0];
  }

  /* When the coprocess writes to its stderr, we'll log that output */
  UT_fd_reg(coproc_stderr[0],name,(UT_fd_cb*)UT_coproc_logger,(void*)pid,UTFD_R|UTFD_PIPE);

  /* parent uses non-blocking i/o */
  fdflags = fcntl(coproc_stdin[0], F_GETFL, 0);
  fcntl(coproc_stdin[0], F_SETFL, fdflags | O_NONBLOCK);
  fdflags = fcntl(coproc_stdout[0], F_GETFL, 0);
  fcntl(coproc_stdout[0], F_SETFL, fdflags | O_NONBLOCK);
  fdflags = fcntl(coproc_stderr[0], F_GETFL, 0);
  fcntl(coproc_stderr[0], F_SETFL, fdflags | O_NONBLOCK);

  coproc = (UT_coprocess *)UT_mem_alloc( COPROC_POOL, 1);
  UT_strncpy(coproc->name,name,COPROC_NAME_MAXLEN);
  coproc->pid = pid;
  coproc->exit_cb = exhdlr;
  coproc->data = data;
  coproc->fds[0] = coproc_stdin[0];
  coproc->fds[1] = coproc_stdout[0];
  coproc->fds[2] = coproc_stderr[0];
  time(&coproc->start_time);
  coproc->next = NULL;
  LL_ADD(UT_coproc_global.coprocs,tmp,coproc);

  UT_LOG(Debug,"Forked coprocess [%s]: pid %d", name, pid);
  return pid;
}

