/*********************************************************************
 * 
 *   Copyright (C) 2001-2004 Torsten Marek
 *   Parts of the code are taken from the xmms mgp123-Plugin by Espen Skoglund
 *   Code for swapEndian fuctions taken from glib.h
 * 
 * Filename:      id3tag.cpp
 * Description:   implementation of the id3Tag class for accessing ID3v2 tags in mp3 files
 * Author:        Torsten Marek <shlomme@gmx.net>
 *                
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ********************************************************************/

// TODO:
// add encryption and compression as needed
#include <unistd.h>
#include <cstring>
#include <sys/stat.h>
#include "id3tag.hpp"

inline int convertToHeader(unsigned char * buf)
{
    return (buf[0] << 24) + (buf[1] << 16) + (buf[2] << 8) + buf[3];
}

uint16 swapEndian(uint16 val)
{
    return ((val & (uint16) 0x00ffU) << 8) | ((val & (uint16) 0xff00U) >> 8);
}

uint32 swapEndian(uint32 val)
{
    return ((val & (uint32) 0x000000ffU) << 24) | ((val & (uint32) 0x0000ff00U) <<  8) | ((val & (uint32) 0x00ff0000U) >>  8) | ((val & (uint32) 0xff000000U) >> 24);
}

id3Tag::id3Tag(const char* n) : TagEditor(n)
{
    id3Frame::setConverter(converter());
    // initialize header
    id3_version = 4;
    id3_revision = id3_flags = id3_tagsize = id3_pos = 0;
    id3_bufsize = 8192;
    id3_buf = new char[id3_bufsize];
    hasID3v1 = hasID3v2 = false;

    fstream in(fname.c_str(), ios::in);
    if(!in.is_open()) {
        tag_insane = true;
        return;
    }
    file = &in;
    read();
    
    if(!hasID3v2) {
        id3_pos = 0;
        file->seekg(0, ios::beg);
    }
    getSongInfo();
    file->close();
    file = NULL;
}

id3Tag::~id3Tag()
{
    delete[] id3_buf;
    FRAMEMAP::iterator fra;
    for(fra = frames.begin(); fra != frames.end(); ++fra)
        delete(fra->second);
}


void id3Tag::read()
{
    if(readID3v2Tag())
        return;
    else if(readID3v1Tag())
        return;
    else {
        // guess from filename
        frames[ID3_TPE1] = newFrame(ID3_TPE1, NULL);
        int begin = fname.find_last_of("/") + 1;
        int end = fname.find_last_of(".") - 1;
        int divide = fname.find_first_of("-", begin);
        string t = fname.substr(begin, divide - begin);
        if(*t.rbegin() == ' ')
            t.erase(t.length() - 1);
        frames[ID3_TPE1]->setContent(t);
        frames[ID3_TIT2] = newFrame(ID3_TIT2, NULL);
        t = fname.substr(divide + 1,  end - divide);
        if(*t.begin() == ' ')
            t.erase(0, 1);
        frames[ID3_TIT2]->setContent(t);
    }    
}

bool id3Tag::readID3v1Tag() 
{
    id3v1 t;
    memset(&t,0, sizeof(t));
    file->seekg(-128, ios::end);
    file->read((char*)&t, sizeof(t));
    if(memcmp(t.t_tag, "TAG", 3))
        return false;
    fillFrame(ID3_TIT2, t.t_title, min(strlen(t.t_title), sizeof(t.t_title)));
    fillFrame(ID3_TPE1, t.t_name, min(strlen(t.t_name), sizeof(t.t_name)));
    fillFrame(ID3_TALB, t.t_album, min(strlen(t.t_album), sizeof(t.t_album)));
    fillFrame(ID3_TDRC, t.t_year, min(strlen(t.t_year), sizeof(t.t_year)));
    fillFrame(ID3_COMM, t.t_comment, min(strlen(t.t_comment), sizeof(t.t_year)));
    if(t.t_track != 0) {
        stringstream s;
        s << (int)t.t_track;
        fillFrame(ID3_TRCK, s.str().c_str(), s.str().length());
    }
    if(t.t_genre != 0xFF) {
        stringstream s;
        s << "(" << (int)t.t_genre << ")";
        fillFrame(ID3_TCON, s.str().c_str(), s.str().length());
    }
    hasID3v1 = true;
    file->seekg(0, ios::beg);
    return true;
}

void id3Tag::getSongInfo()
{
    int head;
    char tmp[4];
    mp3info frm;
    double tpf, bpf;
    int len;

    file->read(tmp, 4);
    head = convertToHeader((unsigned char*)tmp);
    while (!headCheck(head)) {
        if(file->eof())
            return;
        head <<= 8;
        file->read(tmp, 1);
        head |= (unsigned char)tmp[0];
    }
    if (decodeHeader(&frm, head)) {
        tpf = computeTpf(&frm);
        bpf = computeBpf(&frm);
        file->seekg(0, ios::end);
        len = (long)file->tellg() - (long)(id3_pos + (hasID3v1) ? 128 : 0);
        file_length = (unsigned int)((len / bpf) * tpf);
        file_bitrate = tabsel_123[frm.lsf][frm.lay - 1][frm.bitrate_index];
    }
}

bool id3Tag::headCheck(unsigned long head)
{
    if ((head & 0xffe00000) != 0xffe00000)
        return false;
    if (!((head >> 17) & 3))
        return false;
    if (((head >> 12) & 0xf) == 0xf)
        return false;
    if (!((head >> 12) & 0xf))
        return false;
    if (((head >> 10) & 0x3) == 0x3)
        return false;
    if (((head >> 19) & 1) == 1 &&
        ((head >> 17) & 3) == 3 &&
        ((head >> 16) & 1) == 1)
        return false;
    if ((head & 0xffff0000) == 0xfffe0000)
        return false;
    
    return true;
}

int id3Tag::decodeHeader(mp3info *fr, unsigned long newhead)
{
    if (newhead & (1 << 20)) {
        fr->lsf = (newhead & (1 << 19)) ? 0x0 : 0x1;
        fr->mpeg25 = 0;
    } else {
        fr->lsf = 1;
        fr->mpeg25 = 1;
    }
    fr->lay = 4 - ((newhead >> 17) & 3);
    if (fr->mpeg25) 
        fr->sampling_frequency = 6 + ((newhead >> 10) & 0x3);
    else
        fr->sampling_frequency = ((newhead >> 10) & 0x3) + (fr->lsf * 3);

    fr->padding = ((newhead >> 9) & 0x1);
    fr->bitrate_index = ((newhead >> 12) & 0xf);
    fr->framesize = (long) tabsel_123[fr->lsf][2][fr->bitrate_index] * 144000;
    fr->framesize /= mpg123_freqs[fr->sampling_frequency] << (fr->lsf);
    fr->framesize = fr->framesize + fr->padding - 4;
    if(fr->framesize > MAXFRAMESIZE)
        return 0;
    return 1;
}

double id3Tag::computeBpf(mp3info *fr)
{
    double bpf;
    bpf = tabsel_123[fr->lsf][fr->lay - 1][fr->bitrate_index];
    bpf *= 144000;
    bpf /= mpg123_freqs[fr->sampling_frequency] << (fr->lsf);
    return bpf;
}

double id3Tag::computeTpf(mp3info *fr)
{
    const int bs[4] = {0, 384, 1152, 1152};
    double tpf;
    tpf = bs[fr->lay];
    tpf /= mpg123_freqs[fr->sampling_frequency] << (fr->lsf);
    return tpf;
}

void id3Tag::fillFrame(ID3FrameID ID, const char* p, int len)
{
    string s;
    id3Frame* frame;
    s.assign(p, len);
    reverse(s.begin(), s.end());
    int beg = s.find_first_not_of(" ");
    s.erase(0, beg);
    if(s.length() == 0)
        return;
    reverse(s.begin(), s.end());
    frame = newFrame(ID);
    frame->setContent(s);
    frames[ID] = frame;
}

id3Frame* id3Tag::newFrame(ID3FrameID ID, id3_framehdr_t* hdr)
{
    if(hdr == 0) {
        switch(ID) {
        case ID3_TCON:
            return new contentFrame();
        case ID3_COMM:
            return new commentFrame();
        default:
            return new id3Frame(ID);
        }
    } else {
        switch(ID) {
        case ID3_TCON:
            return new contentFrame(hdr);
        case ID3_COMM:
            return new commentFrame(hdr);
        default:
            return new id3Frame(ID, hdr);
        }
    }
}

bool id3Tag::readID3v2Tag() 
{
    id3_taghdr_t *taghdr;
    id3_exthdr_t *exthdr;
    readFromID3(NULL, 3);
    if (memcmp(id3_buf,"ID3", 3))
        return false;
    id3_tagsize = sizeof(*taghdr) + 3;
    taghdr = (id3_taghdr_t*)readFromID3(NULL, sizeof(*taghdr));
    //id3_version = taghdr->th_version;
    id3_revision = taghdr->th_revision;
    id3_flags = taghdr->th_flags;
    id3_tagsize += ID3_GET_SIZE28(swapEndian(taghdr->th_size));

    // Parse extended header
    if (id3_flags & ID3_THFLAG_EXT) {
        exthdr = (id3_exthdr_t*)readFromID3(NULL, sizeof(*exthdr) );
    }
    hasID3v2 = true;
    // Parse frames
    while(id3_pos < id3_tagsize)
        readID3Frame();
    return true;
}

void id3Tag::readID3Frame()
{
    id3_framehdr_t framehdr;
    bool discard;
    id3Frame* frame;
    ID3FrameID ID;
    char *p;
    // read Frameheader
    readFromID3(&framehdr, sizeof(framehdr));
    // determine the type of the frame.
    ID = (ID3FrameID)swapEndian(framehdr.fh_id);
    ID = frdesc.checkID(ID, &discard);
    if((ID3FrameID)0 == ID) {
        file->seekg(id3_tagsize, ios::beg);
        id3_pos = id3_tagsize;
        return;
    }
    // allocate frame and add to the frame map
    frame = newFrame(ID, &framehdr);
    p = (char*)readFromID3(NULL, frame->getSizeInFile());
    frame->setData(p);
    if(!discard)
        frames[ID] = frame;
    else
        delete(frame);
    return;
}

void* id3Tag::readFromID3(void* buf, size_t size)
{
    if(buf == NULL) {
        // if buffer is to small, allocate new memory
        if(size > id3_bufsize) {
            id3_bufsize = size;
            delete[] id3_buf;
            id3_buf = new char[id3_bufsize];
        }
        buf = id3_buf;
    }
    file->read((char*)buf, size);
    id3_pos += size;
    return buf;
}

size_t id3Tag::tagsize()
{
    unsigned int size = sizeof(id3_taghdr_t) + 3;
    FRAMEMAP::iterator fra;
    for(fra = frames.begin(); fra != frames.end(); ++fra){
        int t = fra->second->getSize();
        if(t)
            size += t + sizeof(id3_framehdr_t);
    }
    return size;
}

// writes id3 tag to file
// return values
// 0: everything ok
// -1: file could not be opened for reading/writing
// -2: no tempfile could be created
// -3: old file could not be removed
int id3Tag::writeTag()
{
    if(!isOK())
        return -1;
    // space for id3-tag at beginning in file fname
    unsigned int oldspace = id3_tagsize, newsize;
    fstream out;
    ifstream old;
    // space needed for tag
    newsize = tagsize();
    // if not enough space present
    if(newsize > oldspace) {
        // open old file
        old.open(fname.c_str());
        old.seekg(oldspace);
        // generate new tagsize
        id3_tagsize = newSpace(newsize);
        // open tempfile for creation of new tag
        out.open((fname + string(".tmp")).c_str(), ios::out);
        if(!out.is_open())
            return -2;
    } else {
        // just open file to write tag
        out.open(fname.c_str(), ios::out | ios::in);
        if(!out.is_open())
            return -1;
    }
    // write tag with padding
    writeID3v2Tag(out, id3_tagsize - newsize);
    // if old file is open, copy content
    if(old.is_open()) {
        out << old.rdbuf();
        old.close();
        out.close();
        // and delete old file
        if(rename((fname + string(".tmp")).c_str(), fname.c_str()) == -1)
            return -3;
    }
    return 0;
}

void id3Tag::writeID3v2Tag(fstream& out, int padding)
{
    FRAMEMAP::iterator fra;
    id3_taghdr_t taghdr;
    // Write tag header.
    taghdr.th_version = id3_version;
    taghdr.th_revision = id3_revision;
    taghdr.th_flags = id3_flags;
    id3_tagsize -= 10;
    taghdr.th_size = swapEndian(ID3_SET_SIZE28(id3_tagsize));
    id3_tagsize += 10;
    out.write("ID3", 3);
    out.write((const char*)&taghdr, sizeof(taghdr));
    //Write frames
    for(fra = frames.begin(); fra != frames.end(); ++fra) {
        id3Frame* fr = fra->second;
        int size = fr->getSize();
        id3_framehdr_t fhdr;
        fr->fillHeader(&fhdr);
        out.write((const char*)&fhdr, sizeof(fhdr));
        // write empty frames, too? default yes
        if(size > 0) {
            const char* p = fr->getData();
            out.write(p, size);
        }
    }
    if(padding)
        out.write(string(padding, '\0').c_str(), padding);
    hasID3v2 = true;
    tag_altered = 0;
}

int id3Tag::removeField(FieldID ID)
{
    FRAMEMAP::iterator it = frames.find(frdesc.translateField(ID));
    if(it != frames.end()) {
        delete(it->second);
        frames.erase(it);
        tag_altered = 1;
        return 0;
    }
    return 1;
}

string id3Tag::getField(FieldID ID) const
{
    static string empty("");
    FRAMEMAP::const_iterator it = frames.find(frdesc.translateField(ID));
    if(it != frames.end()) {
        id3Frame* fr = it->second;
        return fr->getContent();
    } else
        return empty;
}

int id3Tag::setField(FieldID ID, const string& s)
{
    ID3FrameID id3ID = frdesc.translateField(ID);
    string conv;
    //  1 == no conversion
    //  0 == conversion successfull
    // -1 == conversion failed
    int convState = 1;
    id3Frame *frame;
    if(frdesc.checkID(id3ID) == -1)
        return -1;
    FRAMEMAP::iterator it = frames.find(id3ID);
    if(it == frames.end()) {
        frame = newFrame(id3ID);
        frames[id3ID] = frame;
    } else
        frame = it->second;
    
    if(frame->encoding() == '\0') 
        convState = converter()->fromUTF8(s, conv);
    if(1 == convState)
        conv = s;
    else if(-1 == convState)
        return -1;
    frame->setContent(conv);
    tag_altered = 1;
    return 0;
}

