/*********************************************************************
 * 
 *   Copyright (C) 2001-2004 Torsten Marek
 * 
 * Filename:      flactag.cpp
 * Description:   implementation of the flacTag class
 * Author:        Torsten Marek <shlomme@gmx.net>
 *                
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ********************************************************************/
#include "flactag.hpp"
#include <iostream>

using namespace std;

flacTag::flacTag(const char* n) : vorbisComment(n)
{
    readFlacTag();
}

flacTag::~flacTag()
{
}


int flacTag::readFlacTag()
{
    FLAC::Metadata::Chain flacfile;
    flacfile.read(fname.c_str());
    if(!flacfile.is_valid()) {
        tag_insane = 1;
        return -1;
    }
    FLAC::Metadata::Iterator meta_iter;
    meta_iter.init(flacfile);
    FLAC::Metadata::StreamInfo* info = 0;
    FLAC::Metadata::VorbisComment* vc = 0;
    do {
        switch(meta_iter.get_block_type()) {
        case FLAC__METADATA_TYPE_STREAMINFO:
            info = (FLAC::Metadata::StreamInfo*)meta_iter.get_block();
            file_length = info->get_total_samples() / info->get_sample_rate();
            // not on the disk, but since flac is lossless, bitrate is virtually correct
            file_bitrate = info->get_sample_rate() * info->get_channels() * info->get_bits_per_sample()/1000;
            break;
        case FLAC__METADATA_TYPE_VORBIS_COMMENT:
            vc = (FLAC::Metadata::VorbisComment*)meta_iter.get_block();
            for(int i = 0; i < vc->get_num_comments(); ++i) {
                FLAC::Metadata::VorbisComment::Entry e = vc->get_comment(i);
                string tag(e.get_field_name(), e.get_field_name_length());
                string conv;
                string value(e.get_field_value(), e.get_field_value_length());
                string::iterator it = tag.begin();
                for(; it != tag.end(); ++it)
                    *it = toupper(*it);
                comments[tag] = value;
            }
            break;
        }
    } while(meta_iter.next() && (info == 0 || vc == 0));
	return 0;
}

// writes flac tag to file
// return values
// 0: everything ok
// -1: an error occurred while opening the file
// -2: an error occurred while writing the file
int flacTag::writeTag() 
{
    FLAC::Metadata::Chain flacfile;
    flacfile.read(fname.c_str());
    if(!flacfile.is_valid())
        return -1;

    FLAC::Metadata::Iterator meta_iter;
    meta_iter.init(flacfile);
    int found = 0;
    do {
        if(meta_iter.get_block_type() == FLAC__METADATA_TYPE_VORBIS_COMMENT) {
            found = 1;
            break;
        }
    } while(meta_iter.next());
    FLAC::Metadata::VorbisComment* vc = new FLAC::Metadata::VorbisComment;
    map<string, string>::iterator it = comments.begin();
	for(; it != comments.end(); ++it) {
        string c = it->second;
        FLAC::Metadata::VorbisComment::Entry e;
        e.set_field_name(it->first.c_str());
        e.set_field_value(c.c_str(), c.length());
        vc->insert_comment(vc->get_num_comments(), e);
	}
    if(found)
        meta_iter.set_block(vc);
    else
        meta_iter.insert_block_after(vc);
    // write comment block
    flacfile.sort_padding();
    int ret = flacfile.write();
    // that's it
    if(ret) {
        tag_altered = 0;
        return 0;
    }
    else
        return -2;
}
