# include <string.h>
# include <pnet6.h>

/*----------------------------------------------------------------------*/
/* Example program, demonstrating how multicasting works		*/
/* This is a simple program that binds to port 9999 and prints out 	*/
/* everything it receives. If it receives 'sd', it shuts down.		*/
/*----------------------------------------------------------------------*/
static int read_callback( PNETSOCK ps, void * data );

static PNETADDR	pa;
static const char * ifname= "eth0";

int
main( int argc, const char ** argv )
{
    PNETSOCK	ps;
    const char * host = "224.0.0.1";
    const char * sport= "9999";
    int		pfam = PNET_IPv4;
    int 	c;
    int		hops = 0;
    int		mhops = 1;
    char 	addrbuf[ PNET_ADDR_BUFSIZ ];

    while ( (c = pnetGetopt( argc, argv, ":f:h:i:m:p:" )) != -1 )
    {
	switch ( c )
	{
	case 'f':
	    pfam = !strcmp( pnetOptarg, "inet6" ) ? PNET_IPv6 :
		   !strcmp( pnetOptarg, "inet4" ) ? PNET_IPv4 : -1;
	    if ( pfam == -1 )
	    {
		fprintf(stderr, "Unknown address family %s, "
				"use inet4|inet6\n",pnetOptarg);
		return -1;
	    }
	    break;

	case 'h':	host  = strdup( pnetOptarg );	break;
	case 'i':	ifname= strdup( pnetOptarg );   break;
	case 'm':	mhops = atoi( pnetOptarg); 	break;
	case 'p':	sport = strdup( pnetOptarg );	break;
	case '!': 	fprintf(stderr, "Option %c requires and argument\n",
				pnetOptopt);
			return 1;
	default:
	    fprintf(stderr,"usage: mcast -h <host> -p <port> "
			   "-i <inteface> -f <address family>\n");
	    return 1;
	}
    }

    /* Step one: resolve multicast host address, i.e. the address that we
     * wish to receive multicasts on. 				*/

    if ( ! (pa = pnetAddrResolve( pfam, host )) )
	return -1; /* Probably can't resolve */

    /* Set the port on which multicasts will be received 		*/
    pnetAddrLookupPort( pa, "udp", sport );

    printf("Resolved '%s' -> '%s'\n", host,
	    pnetAddrToString( pa, addrbuf, sizeof( addrbuf ) ) );
    /* Step 2: Create socket on which to listen for multicasts 		*/
    /*         This socket must be UDP, since only UDP supports it.	*/

    if ( ! (ps = pnetUDPSocketX( pfam )) )
	{ pnetAddrFree( pa ); return -1; }

    /* Join multicast group */
    if ( pnetMCastJoin( ps, pa, ifname ) )
    {
	/* Joined failed. 			*/
	/* Was the address a multicast address? */
	if ( pfam == PNET_IPv6 )
	    printf("Is interface %s IPv6 configured?\n", ifname );
	pnetAddrFree( pa );
	return -1;
    }

    /* Get value of multicast hops */
    if ( pnetMCastGetHops( ps, &hops ) )
	printf("Can't read default value for multicast hops\n");
    else
	printf("Current value of multicast hops = %d\n", hops );

    if ( mhops > 1 && mhops != hops )
	if ( !pnetMCastSetHops( ps, mhops ) )
	    printf("Multicast hops set to %d\n", mhops );

    /* Get value of multicast loop */
    if ( pnetMCastGetLoop( ps, &hops ) )
	printf("Can't read default value for multicast loop\n");
    else
	printf("Current value of multicast loop = %d\n", hops );
    if ( !pnetMCastSetLoop( ps, 0 ) )
	printf("Multicast loop set to %d\n", 0 );

    pnetSockAddReadcallback( ps, read_callback, NULL );
    /* Start listening */

    if ( pnetListen( ps, NULL  ) )
    {
	pnetAddrFree( pa );
	return -1;
    }

    pnetStartListen( PNET_LISTEN_THREAD );

    return 0;
}

static int read_callback( PNETSOCK ps, void * data )
{
    PNETADDR    from;
    PNETADDR    to  ;
    char 	buf[128];
    char	addrbuf[ PNET_ADDR_BUFSIZ ];
    int ret;
    int n;
    PNETSOCK_AUXINFO paux ;

    /* Wait for input on the socket for up to 10 seconds */
    ret = pnetReadTimed(ps,buf,sizeof(buf),0,10000);

    from = pnetSockGetPeerAddr(ps);

    /* All PNET read routines return PNET_READ_TIMED_OUT if no input    */
    /* is present after the given time period                           */
    if (ret == PNET_READ_TIMED_OUT)
    {
        printf("Timed out on client %s\n",
		pnetAddrToString( from, addrbuf, sizeof( addrbuf ) ) );
        return -1;
    }
    else if (ret == PNET_READ_ERROR)
    {
        printf("Error reading data from client %s.\n",
		pnetAddrToString( from, addrbuf, sizeof( addrbuf ) ) );
        return -1;
    }
    to   = pnetSockGetLocalAddr(ps);

    printf("Source       %s\n",
    	   pnetAddrToString( from, addrbuf, sizeof( addrbuf ) ) );
    paux = pnetSockAuxInfo(ps);

    /* 
     * A UDP socket can also contain additional information, if it has been
     * created with the pnetUDPSocketX() function, which marks it for
     * receiving ancillary data such as interface index on which the data-
     * gram has been received, and the actual datagram destination address.
     * This also works for IPv6.
     */
    if ( paux )
    {
	PNETIF pif;

	/* Get actual datagram address */
	printf("Actual datagram destination address: %s\n",
		pnetAddrToString( pnetAuxDstAddr(paux),
				  addrbuf, sizeof( addrbuf ) ) );

	/* Get interface index on which datagram arrived */
	pif = pnetAuxRecvIf(paux);
	if ( pif )
	    printf("Datagram received on interface %s\n",pnetIfName(pif));
    }

    
    n = ret;
    buf[n] = 0;
    printf("Got multicast message: %s\n",buf);

    if ( ! strncmp( buf, "sd", 2 ) )
    {
	printf(" Shutting down\n" );
	if ( pnetMCastLeave( ps, pa, ifname ) )
	    printf("Error leaving multicast group\n");
	pnetAddrFree( pa );
	pnetClose( ps );
	exit( 0 );
    }
    return 0;
}
