/*
 * This file is part of
 *
 * PNET6: a Portable Network Library
 *
 * PNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * $Id: pkt-win32.c,v 1.14 2002/10/03 06:24:54 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:		pkt-win32.c
 * created on:		Thu Jul 18 09:28:50 EDT 2002
 * created by:		peter
 * project: 		PNET6
 *----------------------------------------------------------------------*/

# include "../local.h"

# if defined PNET_HAVE_WIN32_PACKET		/* { */

# include "../pkt.h"
# include "../pnet6pkt.h"

# if PNET_WIN32_USE_PACKET_DLL			/* {{ */

/*----------------------------------------------------------------------*/
/* Monitoring packets on the network, using packet.dll (see README.txt)	*/
/*----------------------------------------------------------------------*/

int
pkt_get_ll_type( struct pnet_pktacc *pa )
{
    NetType 	net;

    if ( PacketGetNetType( pa->pa_fd, &net ) == FALSE )
	{ FATALERR("PacketGetNetType()"); return -1; }

    printf(" Net type = %d, Net speed = %d\n", net.LinkType, net.LinkSpeed );

    switch ( net.LinkType)
    {
    case NdisMediumWan:
    case NdisMedium802_3:
	pa->pa_llh_len = 14;
	pa->pa_iftype  = DLT_EN10MB;
	break;

    case NdisMediumFddi:
	pa->pa_llh_len = 21;
	pa->pa_iftype  = DLT_FDDI;
	break;
    }

    pdbg(E_INFO,"pkt_get_ll_type(): interface type: %s\n",
		 pkt_get_ll_name(pa));

    return 0;
}
int
pkt_set_promiscuous( struct pnet_pktacc * pa, int on )
{
    int ret = 0;
    int flag;

    pdbg(E_INFO,"pkt_set_promiscuous(pa=%X,on=%s)\n",XX(pa),PONOFF(on));

    if ( on )
	flag  =  NDIS_PACKET_TYPE_PROMISCUOUS;
    else
    {
    	flag &= ~NDIS_PACKET_TYPE_PROMISCUOUS;
	flag |=  NDIS_PACKET_TYPE_ALL_LOCAL;
    }

    ret = PacketSetHwFilter( pa->pa_fd, flag );

    if (ret == FALSE)
	{ FATALERR("PacketSetHWFilter()"); return -1; }

    if (ret == TRUE)
	pa->pa_ifisprom = on != 0;

    return - (ret != 0);
}
int
pkt_set_read_timeout( PNetPktAccess *pa, int msec )
{
    LLOG( msec );
    return - (PacketSetReadTimeout( pa->pa_fd, msec ) == FALSE);
}
int
pkt_get_stats( PNetPktAccess * pa, pnet_uint * caught, pnet_uint * dropped )
{
    struct bpf_stat     stats;

    if ( ! PacketGetStats( pa->pa_fd, &stats ) )
	{ FATALERR("PacketGetStats()"); return -1; }

    if (caught)
	*caught = stats.bs_recv;
    if (dropped)
	*dropped = stats.bs_drop;
    return 0;
}
struct pnet_pktacc*
pkt_open_dev( const char *ifname, int tries, int glen )
{
    struct pnet_pktacc *	pa;
    LPADAPTER			adapter;
    int				kb = 512;
    char 			buf[128];

    sprintf(buf,"%s",ifname );
    sprintf( buf,"\\Device\\Packet_{1A28EAB5-D40C-4CA6-92B4-5DEF2964AD57}");

    pdbg(E_DBG4,"pkt_open_dev(): trying device %s\n", buf );

    adapter = PacketOpenAdapter( buf );

    if ( ! adapter || (adapter->hFile == INVALID_HANDLE_VALUE) )
	{ NETERR("pkt_open_dev()"); return NULL; }

    DBG(dbg("Bound to interface %s\n",ifname));

    /* Set a 512K buffer in the driver */
    if( PacketSetBuff( adapter, kb * 1024 ) == FALSE )
    {
	perr(E_FATAL,"Cannot set kerner packet buffer to %dK\n", kb);
	PacketCloseAdapter( adapter );
	return NULL;
    }

    STDMALLOC( pa, sizeof(struct pnet_pktacc), NULL );

    strncpy( pa->pa_ifname, ifname, IFNAMSIZ );

    pa->pa_fd	= adapter;
    pa->pa_pkt	= PacketAllocatePacket();
    pa->pa_buflen= PKTBUFSIZ;


    if ( !pa->pa_pkt )
    {
	perr(E_FATAL,"Cannot initialize packet structure\n");
	PacketCloseAdapter( adapter );
	return NULL;
    }
    pdbg(E_INFO,"pkt_open_dev(): using capture buffer size of %d\n",
	 pa->pa_buflen);

    STDMALLOC( pa->pa_buf, pa->pa_buflen, NULL );

    PacketInitPacket( pa->pa_pkt, pa->pa_buf, pa->pa_buflen );

    pa->pa_glen = glen;
    pa->pa_cooked = PKTACC_NOT_COOKED;

    pkt_get_ll_type( pa );

    return pa;
}
void
pkt_close_dev( PNetPktAccess * pa )
{
    pdbg(E_DBG4,"pkt_close_dev(): closing device %s\n", pa->pa_ifname );

    if ( pa->pa_ifisprom )
	pkt_set_promiscuous( pa, 0 );

    PacketFreePacket( pa->pa_pkt );
    PacketCloseAdapter( pa->pa_fd );
    STDFREE( pa->pa_buf );
    STDFREE( pa );
}
int
pkt_set_bpf_filter( PNetPktAccess *pa, BPF_FILTER *insn, int len )
{
    struct bpf_program 	prgm;

    prgm.bf_len = len;
    prgm.bf_insns = insn;

    if ( ! PacketSetBpf( pa->pa_fd, &prgm ) )
	{ FATALERR("PacketSetBpf()"); return -1; }

    pdbg(E_DBG4,"pkt_set_bpf_filter(): installed filter ok\n");

    return 0;
}
#include <dbgon.h>

int
pkt_next_pkt( PNetPktAccess *pa, PNetPacket *pkt )
{
    struct bpf_hdr * bh;

    if ( pa->pa_sbuf >= pa->pa_ebuf )
    {
	/* Read new data from interface */
	int got;

	if ( PacketReceivePacket( pa->pa_fd, pa->pa_pkt, TRUE) == FALSE )
	    { NETERR("pkt_next_pkt()"); return -1; }

	got = pa->pa_pkt->ulBytesReceived;

	LLOG( got );

	if ( got <= 0 )
	    return got; 	/* timeout or error */

	/* Mark start and end of buffer */
	pa->pa_sbuf = pa->pa_buf;
	pa->pa_ebuf = pa->pa_sbuf + got;
    }
    else
    {
	/* pa's buffer contains more packets. Skip to the next one here */

	bh = (struct bpf_hdr*) pa->pa_sbuf;	/* Current pkt header */

	/* Advance pa's buffer start pointer to just behind this packet */
	pa->pa_sbuf += Packet_WORDALIGN( bh->bh_hdrlen + bh->bh_caplen );

	if ( pa->pa_sbuf >= pa->pa_ebuf )
	{
	    pa->pa_sbuf = pa->pa_ebuf;
	    return 0;			/* No more packets */
	}
    }

    /* Get data from next packet */
    bh = (struct bpf_hdr*) pa->pa_sbuf;

    pkt->pkt_tssec  = bh->bh_tstamp.tv_sec;
    pkt->pkt_tsusec = bh->bh_tstamp.tv_usec;
    pkt->pkt_grablen= bh->bh_caplen;
    pkt->pkt_datalen= bh->bh_datalen;

    DBG( dbg("pkt_next_pkt(): pkt: grablen=%d, datalen=%d\n",
	     pkt->pkt_grablen,pkt->pkt_datalen ) );

    /* Skip past bpf header */
    pkt->pkt_buf = pa->pa_sbuf + bh->bh_hdrlen;
    pkt->pkt_pa	 = pa;
   
    /* pkt_set_type( pkt );*/

    return 1;		/* More packet data present */
}
int
pkt_output( PNetPktAccess *pa, pnet_byte *buf, pnet_uint blen )
{
    PACKET pkt;

    pkt.Buffer = (void*)buf;
    pkt.Length = blen;
# include <dbgon.h>
LLOG( pkt.Length );

    if ( ! PacketSendPacket( pa->pa_fd, &pkt, 0 ) )
	{ NETERR( "pkt_output()" ); return -1; }

    return blen;

}
# if 0 	/* { */
int
pkt_set_type( PNetPacket * pkt )
{
    byte * b;

    if ( ! pkt->pkt_pa )
	return -1;

    b = pkt_get_payload( pkt );

    switch ( pkt->pkt_pa->pa_iftype )
    {
    case DLT_NULL:

	if ( pkt->pkt_buf[0] == AF_INET )
	{
	    pkt->pkt_type = PNET_PKTTYPE_IPv4;
	    pkt->pkt_stype= ((struct pnet_ip*) b)->ip_p;
	}
	else if ( pkt->pkt_buf[0] == AF_INET6 )
	{
	    pkt->pkt_type = PNET_PKTTYPE_IPv6;
	    pkt->pkt_stype= ((struct pnet_ip6*)b)->ip6_nxt;
	}
	return 0;

    case DLT_EN10MB:
	{
	    struct ether_header *eh = (struct ether_header*) pkt->pkt_buf;

	    if ( eh->ether_type == ETHERTYPE_IP )
	    {
		pkt->pkt_type = PNET_PKTTYPE_IPv4;
		pkt->pkt_stype= ((struct pnet_ip*) b)->ip_p;
	    }
	    else if ( eh->ether_type == ETHERTYPE_IPV6 )
	    {
		pkt->pkt_type = PNET_PKTTYPE_IPv6;
		pkt->pkt_stype= ((struct pnet_ip6*)b)->ip6_nxt;
	    }
	    else if ( eh->ether_type == ETHERTYPE_ARP  )
	    {
		pkt->pkt_type = PNET_PKTTYPE_ARP;
		pkt->pkt_stype= ((struct arphdr*)b)->ar_hrd;
	    }
	    else if ( eh->ether_type == ETHERTYPE_REVARP  )
	    {
		pkt->pkt_type= PNET_PKTTYPE_RARP;
	    }
	}
	return 0;
    }

    return -1;
}
# endif /* } */

# else 					/* }{ */

# include <mstcpip.h>
/*----------------------------------------------------------------------*/
/* Monitoring packets on the network RCV_ALL ioctl			*/
/*----------------------------------------------------------------------*/

int
pkt_get_ll_type( struct pnet_pktacc *pa )
{
    return 0;
}
int
pkt_set_promiscuous( struct pnet_pktacc * pa, int on )
{
    return 0;
}
int
pkt_set_read_timeout( PNetPktAccess *pa, int msec )
{
    return 0;
}
int
pkt_get_stats( PNetPktAccess *pa, pnet_uint * caught, pnet_uint * dropped )
{
    return 0;
}
struct pnet_pktacc*
pkt_open_dev( const char *ifname, int tries, int glen )
{
    struct pnet_pktacc * pa;
    InetAddr		ia;
    int			sd;
    int			on = 1;

    pdbg(E_DBG4,"pkt_open_dev(): %s using RCVALL ioctl \n", ifname );

    if ( net_get_if_ipaddr( AF_INET, ifname, (SockAddr*)&ia ) )
    {
	perr(E_FATAL,"pkt_open_dev(): no such interface %s\n", ifname );
	return NULL;
    }

    pnetInit();
    sd = WSASocket( AF_INET, SOCK_RAW, IPPROTO_IP , NULL, 0, 0 );
    if ( sd == INVALID_SOCKET )
	{ NETERR("pkt_open_dev()"); return NULL; }

    if ( bind( sd, (SockAddr*) &ia, sizeof( ia ) ) )
	{ NETERR("pkt_open_dev()"); closesocket( sd ); return NULL; }


    if ( WSAIoctl( sd, SIO_RCVALL, (void*)&on, sizeof( on ),
    		   NULL, 0, &on, 0, 0 ) )
     	{ NETERR("pkt_open_dev()" ); closesocket( sd ); return NULL; }

    STDMALLOC( pa, sizeof(struct pnet_pktacc), NULL );

    strncpy( pa->pa_ifname, ifname, IFNAMSIZ );
    pa->pa_fd	= sd;
    pa->pa_ifindex = 0;
    pa->pa_buflen= PKTBUFSIZ;

    pdbg(E_INFO,"pkt_open_dev(): using grab buffer size of %d\n",pa->pa_buflen);

    STDMALLOC( pa->pa_buf, pa->pa_buflen, NULL );

    pa->pa_glen = glen;
    pa->pa_cooked = PKTACC_TRUE_COOKED;

    tries--;
    pkt_get_ll_type( pa );

    return pa;
}
void
pkt_close_dev( PNetPktAccess * pa )
{
    pdbg(E_DBG4,"pkt_close_dev(): closing device %s\n", pa->pa_ifname );

    if ( pa->pa_ifisprom )
	pkt_set_promiscuous( pa, 0 );

    closesocket( pa->pa_fd );
    STDFREE( pa->pa_buf );
    STDFREE( pa );
}
int
pkt_set_bpf_filter( PNetPktAccess *pa, BPF_FILTER *insn, int len )
{
    pa->pa_bpf = insn;

    pdbg(E_DBG4,"pkt_set_bpf_filter(): installed filter ok\n");

    return 0;
}

extern pnet_uint bpf_filter( BPF_FILTER*, pnet_byte*, pnet_uint, pnet_uint );
int
pkt_next_pkt( PNetPktAccess *pa, PNetPacket *pkt )
{
    struct sockaddr_in	sa;
    struct timeval 	tv;
    socklen_t		len;
    int			got = -1;

    len = sizeof( sa );
    errno = EINTR;

    /* Read new data off the interface */
    while ( got == -1 && errno == EINTR )
    {
	got = recvfrom( pa->pa_fd, pa->pa_buf, pa->pa_buflen, 0, 
			(SockAddr*) &sa, &len );

	/* Get the timestamp */
	tv.tv_sec= 0; tv.tv_usec= 0;

	gettimeofday( &tv, NULL );
    }

    if ( got <= 0 )
	return 0; 		/* timeout or error */

    DBG(dbg("Got %d bytes from interface %s\n",got,pa->pa_ifname));

    /*
     * Run through bpf filter. If result is 0, then packet failed the 
     * filter.
     */
    if ( bpf_filter( pa->pa_bpf, (pnet_byte*)pa->pa_buf, got, got ) == 0 )
	return 0;

    /* Mark start and end of buffer */
    pa->pa_sbuf = pa->pa_buf;
    pa->pa_ebuf = pa->pa_sbuf;

    pkt->pkt_tssec  = tv.tv_sec;
    pkt->pkt_tsusec = tv.tv_usec;
    pkt->pkt_grablen= got;
    pkt->pkt_datalen= 0;
    pkt->pkt_buf 	= pa->pa_sbuf;
    pkt->pkt_pa	= pa;
    pkt_set_type( pkt );

    return 1;		/* More packet data present */
}

int
pkt_output( PNetPktAccess *pa, pnet_byte *buf, pnet_uint blen )
{
    perr(E_FATAL,"pkt_output() not implemented on this system.\n");
    return 0;
}
int
pkt_set_type( PNetPacket * pkt )
{
    byte * b;

    if ( ! pkt->pkt_pa )
	return -1;

    b = pkt_get_payload( pkt );

    switch ( pkt->pkt_pa->pa_iftype )
    {
    case DLT_NULL:

	if ( pkt->pkt_buf[0] == AF_INET )
	{
	    pkt->pkt_type = PNET_PKTTYPE_IPv4;
	    pkt->pkt_stype= ((struct pnet_ip*) b)->ip_p;
	}
	else if ( pkt->pkt_buf[0] == AF_INET6 )
	{
	    pkt->pkt_type = PNET_PKTTYPE_IPv6;
	    pkt->pkt_stype= ((struct pnet_ip6*)b)->ip6_nxt;
	}
	return 0;

    case DLT_EN10MB:
	{
	    /*
	    struct ether_header *eh = (struct ether_header*) pkt->pkt_buf;

	    if ( eh->ether_type == ETHERTYPE_IP )
	    {
		pkt->pkt_type = PNET_PKTTYPE_IPv4;
		pkt->pkt_stype= ((struct pnet_ip*) b)->ip_p;
	    }
	    else if ( eh->ether_type == ETHERTYPE_IPV6 )
	    {
		pkt->pkt_type = PNET_PKTTYPE_IPv6;
		pkt->pkt_stype= ((struct pnet_ip6*)b)->ip6_nxt;
	    }
	    else if ( eh->ether_type == ETHERTYPE_ARP  )
	    {
		pkt->pkt_type = PNET_PKTTYPE_ARP;
		pkt->pkt_stype= ((struct arphdr*)b)->ar_hrd;
	    }
	    else if ( eh->ether_type == ETHERTYPE_REVARP  )
	    {
		pkt->pkt_type= PNET_PKTTYPE_RARP;
	    }
	    */
	}
	return 0;
    }

    return -1;
}

# endif 					/* }} */

# endif						/* } */
