/*
 * This file is part of
 *
 * PNET6: a Portable Network Library
 *
 * PNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pkt-linux.c,v 1.14 2002/10/04 14:11:47 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:		pkt-linux.c
 * created on:		Sun Jul  7 21:30:24 CEST 2002
 * created by:		peter
 * project: 		PNET6
 *----------------------------------------------------------------------*/

# include "../local.h"
# include "../pkt.h"
# include "../pnet6pkt.h"

# if defined PNET_HAVE_PACKET_ACCESS && defined PNET_HAVE_PF_PACKET	/* { */

/*----------------------------------------------------------------------*/
/* Monitoring packets on the network, Linux PF_PACKET style.		*/
/*----------------------------------------------------------------------*/

static int
get_ll_type( int fd, const char * ifname, int *phl )
{
    struct ifreq	ifr;
    int			iftype;

    memset( &ifr, 0, sizeof( ifr ) );
    strncpy( ifr.ifr_name, ifname, sizeof( ifr.ifr_name ) );

    if ( ioctl( fd, SIOCGIFHWADDR, &ifr ) )
    {
	perr(E_FATAL,"Cannot obtain datalink type for iface %s.\n",ifname );
	LOGSYSERR("pkt_get_ll_type()");

	return -1;
    }

    switch ( ifr.ifr_hwaddr.sa_family )
    {
    case ARPHRD_ETHER:
    case ARPHRD_LOOPBACK:
    	iftype 	= DLT_EN10MB;
	*phl	= sizeof( struct ether_header );
	break;

    case ARPHRD_SLIP:
    	iftype	= DLT_SLIP;
	*phl	= -1;
	break;

    case ARPHRD_FDDI:
	iftype	= DLT_FDDI;
	*phl	= -1;
	break;

    case ARPHRD_PPP:
    	iftype	= DLT_PPP;
	*phl	= -1;
	break;

    default:
	iftype	= -1;
	*phl    = -1;
	break;
    }

    return iftype;
}
int
pkt_get_ll_type( struct pnet_pktacc *pa )
{
    pa->pa_iftype = get_ll_type( pa->pa_fd, pa->pa_ifname, &pa->pa_llh_len );
    pdbg(E_INFO,"pkt_get_ll_type(): interface type: %s\n",pkt_get_ll_name(pa));

    return 0;
}
int
pkt_set_promiscuous( struct pnet_pktacc * pa, int on )
{
    struct ifreq 	ifr;
    int			prom;

    DBG(dbg("pkt_set_promiscuous(pa=%X,on=%s)\n",XX(pa),ponoff(on)));

    if ( (! pa->pa_ifisprom && ! on ) || ( pa->pa_ifisprom && on ) )
    {
	pdbg( E_DBG4,"Device %s is in requested promiscuous state (%s).\n",
		pa->pa_ifname, ponoff(on) );
	return 0;
    }
    strncpy( ifr.ifr_name, pa->pa_ifname, IFNAMSIZ );

    if ( ioctl( pa->pa_fd, SIOCGIFFLAGS, &ifr ) < 0 )
	{ FATALERR("ioctl( ..., SIOCGIFFLAGS, ... )"); return -1; }

    prom = ifr.ifr_flags & IFF_PROMISC;

    pdbg( E_DBG4,"Device %s is %s promiscuous mode.\n",
    	  pa->pa_ifname, prom ? "in" : "not in" );

    if ( on )
	ifr.ifr_flags |= IFF_PROMISC;
    else
	ifr.ifr_flags &= ~IFF_PROMISC;

    if ( ioctl( pa->pa_fd, SIOCSIFFLAGS, &ifr ) < 0 )
	{ FATALERR("ioctl( ..., SIOCSIFFLAGS, ... )"); return -1; }

    pdbg( E_DBG4,"Promiscuous mode for %s turned %s.\n",
		 pa->pa_ifname, on ? "on" : "off");

    pa->pa_ifisprom = !!on;

    return 0;
}
int
pkt_set_read_timeout( PNetPktAccess *pa, int msec )
{
    pdbg(E_INFO,"packet read timeout off.\n");

    return 0;
    pa = pa;
    msec = msec;
}
# ifndef HAVE_STRUCT_TPACKET_STATS		/* { */
#define PACKET_STATISTICS               6
struct tpacket_stats
{
    unsigned int    tp_packets;
    unsigned int    tp_drops;
};
# endif						/* } */
int
pkt_get_stats( PNetPktAccess *pa, pnet_uint * caught, pnet_uint * dropped )
{
    struct tpacket_stats	stats;
    socklen_t			len = sizeof( struct tpacket_stats );

    memset( &stats, 0 , sizeof( stats ) );
    if ( getsockopt( pa->pa_fd, SOL_SOCKET,
		     PACKET_STATISTICS, &stats, &len ) )
    {
	FATALERR("getsockopts( ..., SOL_SOCKET, PACKET_STATISTICS, ... )");
	return -1;
    }

    /* tp_packets: means all packets that passed our filter. 		   */
    /* tp_drops:   means packets that were dropped because our application */
    /*             could not keep up with the flow of packets returned by  */
    /*             the filter.						   */
    /*             This does not mean packets dropped because they did not */
    /*             _pass_ the filter. Of those, we shall never learn.      */

    if ( caught )	*caught = stats.tp_packets;
    if ( dropped )	*dropped= stats.tp_drops;

    return 0;
}
/*----------------------------------------------------------------------*/
/* Open a given interface for packet access.				*/
/* We need to check the interface's type prior to deciding if we can	*/
/* use RAW mode (which requires that we know the link layer header	*/
/* length), or cooked mode, which'll strip the header off the packet.	*/
/*----------------------------------------------------------------------*/

struct pnet_pktacc*
pkt_open_dev( const char *ifname, int tries, int glen )
{
    struct sockaddr_ll		sll;
    struct pnet_pktacc *	pa;
    struct ifreq 		ifr;
    int				fd = -1;
    int				cooked= PKTACC_NOT_COOKED;
    int				ll_hlen = 0;
    
    pdbg(E_DBG4,"pkt_open_dev(): trying device %s\n", ifname );

    if ( (fd = socket( PF_PACKET, SOCK_RAW, htons(ETH_P_ALL))) < 0 )
	{ FATALERR("pkt_open_dev()"); return NULL; }

    if ( get_ll_type( fd, ifname, &ll_hlen ) < 0 )
    {
	/* Don't know link layer header length, use COOKED mode */
	close( fd );

	pdbg(E_DBG1,"Using true cooked mode for iface %s\n", ifname );

	if ( (fd = socket( PF_PACKET, SOCK_DGRAM, htons(ETH_P_ALL))) < 0 )
	    { FATALERR("pkt_open_dev()"); return NULL; }

	cooked = PKTACC_TRUE_COOKED;
    }

    /* Bind to this device 				*/

    strncpy( ifr.ifr_name, ifname, IFNAMSIZ );

    if ( ioctl( fd, SIOCGIFINDEX, &ifr ) < 0 )
	{ LOGSYSERR("pkt_open_dev()"); close(fd); return NULL; }

    memset( &sll, 0, sizeof( sll ) );
    sll.sll_family 		= AF_PACKET;
    sll.sll_ifindex		= ifr.ifr_ifindex;
    sll.sll_protocol		= htons( ETH_P_ALL );

    if ( bind( fd, (struct sockaddr*) &sll, sizeof( sll ) ) == -1 )
    {
	perr(E_FATAL,"pkt_open_dev(%s): cannot bind() to device\n", ifname );
	LOGSYSERR("pkt_open_dev()");
	close( fd );
	return NULL;
    }

    STDMALLOC( pa, sizeof(struct pnet_pktacc), NULL );

    pa->pa_buflen	= PKTBUFSIZ;

    STDMALLOC( pa->pa_buf, pa->pa_buflen, NULL );

    strncpy( pa->pa_ifname, ifname, IFNAMSIZ );
    pa->pa_fd		= fd;
    pa->pa_ifindex 	= ifr.ifr_ifindex;
    pa->pa_glen 	= glen;
    pa->pa_cooked	= cooked;
    pkt_get_ll_type( pa );

    pdbg(E_INFO,"pkt_open_dev(): using capture buffer size of %d\n",
	 pa->pa_buflen);

    tries--;

    return pa;
}
void
pkt_close_dev( PNetPktAccess * pa )
{
    pdbg(E_DBG4,"pkt_close_dev(): closing device %s\n", pa->pa_ifname );

    STDFREE( pa->pa_buf );
    STDFREE( pa );
    close( pa->pa_fd );

    return;
}
int
pkt_set_bpf_filter( PNetPktAccess *pa, BPF_FILTER *insn, int len )
{
    BPF_PROG  prog;

    prog.bf_len = len;
    prog.bf_insns = insn;

    if ( setsockopt( pa->pa_fd, SOL_SOCKET, SO_ATTACH_FILTER,
		     &prog, sizeof( prog ) ) )
        { FATALERR("setsockopt(...,SO_ATTACH_FILTER,...)"); return -1; }

    pdbg(E_DBG4,"pkt_set_bpf_filter(): installed filter ok\n");

    return 0;
}
int
pkt_next_pkt( PNetPktAccess *pa, PNetPacket *pkt )
{
    struct sockaddr_ll	sll;
    struct timeval 	tv;
    socklen_t		len;
    int			got = -1;

    len = sizeof( sll );
    errno = EINTR;

    /* Read new data off the interface. If all goes well, passing MSG_TRUNC
     * will tell us the actual length of the packet on the wire, even if its
     * longer than the buffer. Our buffer is long enough to accept most 
     * packet (way bigger than an ethernet MTU), but you never know.
     */
    while ( got == -1 && errno == EINTR ) 
	got = recvfrom( pa->pa_fd, pa->pa_buf, pa->pa_buflen, MSG_TRUNC,
			(SockAddr*) &sll, &len );

    DBG(dbg("Got %d bytes from interface %s\n",got,pa->pa_ifname));

    /* Mark start and end of buffer */
    pa->pa_sbuf = pa->pa_buf;
    pa->pa_ebuf = pa->pa_sbuf;

    tv.tv_sec = 0;
    tv.tv_usec= 0;
    /* Get the timestamp */
    ioctl( pa->pa_fd, SIOCGSTAMP, &tv );
	
    pkt->pkt_tssec  = tv.tv_sec;
    pkt->pkt_tsusec = tv.tv_usec;
    pkt->pkt_grablen= got;
    pkt->pkt_datalen= got;
    pkt->pkt_buf    = pa->pa_sbuf;
    pkt->pkt_pa	    = pa;
    /* pkt_set_type( pkt );*/

    return 1;		/* More packet data present */
}
int
pkt_output( PNetPktAccess *pa, pnet_byte * buf, pnet_uint blen )
{
    return write( pa->pa_fd, buf, blen );
}
# if 0		/* { */
int
pkt_set_type( PNetPacket * pkt )
{
    struct ethhdr *eh;
    byte * b;

    if ( ! pkt->pkt_pa )
	return -1;

    eh = (struct ethhdr*) pkt->pkt_buf;
    b  = pkt_get_payload( pkt );
    
    switch ( htons(eh->h_proto) )
    {
	case ETHERTYPE_IP:
	    pkt->pkt_type = PNET_PKTTYPE_IPv4;
	    pkt->pkt_stype= ((struct pnet_ip*) b)->ip_p;
	    return 0;
	case ETHERTYPE_IPV6:
	    pkt->pkt_type = PNET_PKTTYPE_IPv6;
	    pkt->pkt_stype= ((struct pnet_ip6*) b)->ip6_nxt;
	    return 0;
	case ETHERTYPE_ARP:
	    pkt->pkt_type = PNET_PKTTYPE_ARP;
	    pkt->pkt_stype= ((struct arphdr*) b)->ar_hrd;
	    return 0;
	case ETHERTYPE_REVARP:
	    pkt->pkt_type = PNET_PKTTYPE_RARP;
	    return 0;
    }

    return -1;
}
# endif 	/* } */
# endif 				/* } */
