/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: connect.c,v 1.19 2002/11/05 08:35:35 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            connect.c
 * created on:          Sat May 25 13:10:16 CEST 2002
 * created by:          teddykgb
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

static int
make_local_address( PNetAddr *pa, const char *path )
{
# ifdef PNET_HAVE_LOCAL
    return addr_set_ip_addr(pa,AF_LOCAL,path,strlen(path));
# else
    return -1;
# endif
}

/*----------------------------------------------------------------------*/
/* Make a temporary file name.						*/
/* Since we can't use tmpnam(), what do we do?				*/
/* We make an address that contains the current time. I wonder if this	*/
/* is a good way of doing this? If you got an idea for something better */
/* share it, please.							*/
/*----------------------------------------------------------------------*/

# ifdef PNET_HAVE_LOCAL         /* { */
#   include <time.h>
# endif 			/* } */

static int
make_tmp_address(PNetAddr *pa)
{
# ifdef PNET_HAVE_LOCAL         /* { */
    struct sockaddr_un  *       un;
    char 			name[128];
 
    un = &pa->pa_unaddr;

    memset( name , 0, sizeof( name ) );
    snprintf( name, sizeof( name ) - 1, "/tmp/skt_%10lu",
		   (unsigned long) time( NULL ) );

    un->sun_family = AF_LOCAL;

    strncpy(un->sun_path, name, sizeof( un->sun_path ) - 1);

    pa->pa_len = SUN_LEN(un);
    pa->pa_family = AF_LOCAL;

# endif                         /* } */
    return 0;
}

/*----------------------------------------------------------------------*/
/* Family can be any of the PNET_* family constants, sock type can be	*/
/* either DGRAM or STREAM. If family is PNET_LOCAL, then the hostname	*/
/* parameter is ignored. If family is PNET_LOCAL, then hname is ignored,*/
/* and service is assumed to be a file name on the system.		*/
/* If hname is a LINK LOCAL address, then it must be postfixed by a	*/
/* %ifname.								*/
/*----------------------------------------------------------------------*/

static int
net_connect_socket( PNetSocket *ps  ,
		    const char *hname, const char *serv,
		    const char *lhost, const char * lport,
		    int do_connect  )
{
    int 		pfam 	= fam2pfam(ps->fam);
    const char *        prot_type;
    int                 port;
    char 		buf[ PNET_ADDR_STRLEN ];

    DBG( dbg( "net_connect_socket()\n" ) );

    if (ps->type == SOCK_STREAM)
	prot_type = "tcp";
    else if (ps->type == SOCK_DGRAM)
    	prot_type = "udp";
    else
	prot_type = "none";

    if (pfam == PNET_LOCAL)
    {
	/* Fill in the destination address */
	make_local_address( &ps->peer_addr, serv );

        /* Bind our src address to a temp file. */
        make_tmp_address( &ps->local_addr );

        sock_bind( ps, &ps->local_addr );

        goto conn_lbl;
    }

    /* 1. Fill in remote address */

    /* If not local, try to either convert the peer address from a string */
    /* or try to resolve it 						*/

    if ( pnet_pton( pfam, hname, &ps->peer_addr ) 	&&
         net_gethostbyname( ps->fam, hname, &ps->peer_addr ) )
    {
        perr(E_FATAL,"net_connect_socket(): cannot resolve '%s'\n",hname);
        return -1;
    }

    /* Fill in remote peer port */

    if ( ( port = net_servicetoport( serv, prot_type ) ) < 0 )
    {
        perr(E_FATAL,"net_connect_socket(): '%s': %s\n",serv,neterr());
        return -1;
    }

    addr_set_port( &ps->peer_addr, (pnet_ushort) port );

    if ( lhost || lport )
    {
	/* 2. Fill in local address */

	addr_make_listen( ps->fam, &ps->local_addr, lhost, lport );

	if ( sock_bind( ps, &ps->local_addr ) )
	{
	    perr(E_FATAL,"net_connect_socket(): local %s:%s: %s\n",
			 lhost, lport, neterr());
	    return -1;
	}
    }

conn_lbl:

    if (do_connect)
    {
	if ( connect( ps->sd, &ps->peer_addr.pa_saddr, ps->peer_addr.pa_len) )
	{
	    if ( errno == EINPROGRESS )
	    {
		/*
		 * We are trying to connect a non-blocking socket
		 * Will need to use pnetWaitForConnect()
		 */

		pdbg(E_DBG4,"PNET_EINPROGRESS for %lX\n",XX( ps ) );
		return PNET_EINPROGRESS;
	    }
# ifdef STDWin32	/* } */
	    if ( WSAGetLastError() != WSAEWOULDBLOCK )
	    {
		pdbg(E_DBG4,"PNET_EINPROGRESS for %lX\n",XX( ps ) );
		return PNET_EINPROGRESS;
	    }
# endif			/* { */
	    if (pfam == PNET_LOCAL)
	    {
		perr(E_FATAL,"net_connect_socket(): %s: %s\n",serv,neterr());
	    }
	    else
	    {
		perr(E_FATAL,"net_connect_socket(%s): %s\n",
		     pnet_ntop( &ps->peer_addr, buf, sizeof( buf ) ),
		     neterr());
	    }
	    return -1;
	}

	ps->connected = 1;

	if ( sock_getsockname( ps, &ps->local_addr ) )
	    perr(E_FATAL,"net_connect_socket(): sock_get_sockname() %s: %s\n",
		 serv,neterr());

	pdbg(E_DBG4,"CONNECT: remote %s\n",
		     pnet_ntop( &ps->peer_addr, buf, sizeof( buf ) ));
	pdbg(E_DBG4,"CONNECT: local  %s\n",
		     pnet_ntop( &ps->local_addr, buf, sizeof( buf ) ));
    }

    return 0;
}
static PNETSOCK
net_connect( int pfam, int sock_type,
	     const char * hname, const char * serv,
	     const char * lname, const char * lserv )
{
    PNetSocket *        ps;
    int			ret;

    DBG( dbg( "net_connect()\n" ) );
    /* Create a socket of the desired type and family */

    if (sock_type == SOCK_STREAM)
        ps = pnetTCPSocket2(pfam);
    else if (sock_type == SOCK_DGRAM)
        ps = pnetUDPSocket2(pfam);
    else
    {
        PROTOERR("net_connect()");
        return NULL;
    }   

    if (!ps)
    {
        perr(E_FATAL,"net_connect(): Cannot create socket\n");
        return NULL;
    }

    ret = net_connect_socket( ps, hname, serv,
			          lname, lserv, sock_type == SOCK_STREAM );

    if ( ret && ret != PNET_EINPROGRESS )
	{ pnetClose(ps); return NULL; }
 
    return ps;
}

int
pnetWaitForConnect( PNETSOCK ps, int msec )
{
    fd_set		rds,wrs;
    struct timeval 	tv,*ptv;
    int			ret;
    char 		buf[ PNET_ADDR_STRLEN ];
    
    DBG( dbg( "pnetWaitForConnect()\n" ) );

    ptv = NULL;

    if ( msec >= 0 )
    {
	tv.tv_sec	= msec / 1000;
	tv.tv_usec	= msec % 1000 * 1000;
	ptv		= &tv;
    }
    
    FD_ZERO( &rds );
    FD_ZERO( &wrs );

    FD_SET( ps->sd, &rds );
    FD_SET( ps->sd, &wrs );

    ret = select( ps->sd + 1, &rds, &wrs, (fd_set*) 0, ptv );

    if ( ret == 0 )			/* Time out */
	return PNET_ETIMEDOUT;

    if ( ret == -1 )
	{ NETERR("pnetWaitForConnect()"); return -1; }

    if ( FD_ISSET( ps->sd, &rds ) || FD_ISSET( ps->sd, &wrs ) )
    {
	/* Socket is readable/writable. Check out what the connection 
	 * status is. */
	int e   = 0;
	socklen_t len = sizeof(e);

	if ( getsockopt( ps->sd, SOL_SOCKET, SO_ERROR, (void*)&e, &len ) < 0 )
	    e = errno;	/* Solaris does this: Actual error returned in errno */
	if ( e )
	    { NETERR("pnetWaitForConnect()"); return -1; }
    }
    else
    {
	perr(E_FATAL,"pnetWaitForConnect(): fd not set by select\n");
	return -1;
    }

    /* At this point, the connection has been successfully established. */
    /* Mark socket connected and return 0.				*/

    ps->connected = 1;

    if (sock_getsockname(ps,&ps->local_addr))
	perr(E_FATAL,"pnetWaitForConnect(): sock_get_sockname(): %s\n",
	     neterr());

    pdbg(E_DBG4,"NBCONNECT: remote %s\n",
    		 pnet_ntop( &ps->peer_addr, buf, sizeof( buf ) ));
    pdbg(E_DBG4,"NBCONNECT: local  %s\n",
    		 pnet_ntop( &ps->local_addr,buf, sizeof( buf ) ));

    return 0;
}
int
pnetSockConnected( PNETSOCK ps )
{
    return ps->connected ;
}
int
pnetConnect(PNETSOCK ps,const char *host,const char *serv)
{
    DBG( dbg("pnetConnect(ps=%lX,host=%s,serv=%s)\n",XX(ps),host,serv) );
    return net_connect_socket(ps,host,serv,NULL,NULL,ps->type == SOCK_STREAM);
}
int
pnetConnectX(PNETSOCK ps,const char * host,const char * serv,
			 const char *lhost,const char *lport )
{
    DBG( dbg("pnetConnectX(ps=%lX,host=%s,serv=%s,lhost=%s,lport=%s)\n",
			   XX(ps),host,serv,lhost,lport) );
    return net_connect_socket(ps,host,serv,lhost,lport,ps->type == SOCK_STREAM);
}
PNETSOCK
pnetTCPConnect2(int pfam,const char *host,const char *serv)
{
    DBG( dbg("pnetTCPConnect2(pfam=%d,host=%s,serv=%s)\n",pfam,host,serv) );
    return net_connect(pfam,SOCK_STREAM,host,serv,NULL,NULL);
}
PNETSOCK
pnetTCPConnectX(int pfam,const char * host,const char * serv,
			 const char *lhost,const char *lserv )
{
    DBG( dbg("pnetTCPConnect2(pfam=%d,host=%s,serv=%s,lhost=%s,lserv=%s)\n",
	      pfam,host,serv,lhost,lserv) );
    return net_connect(pfam,SOCK_STREAM,host,serv,lhost,lserv);
}
PNETSOCK
pnetTCPConnect(const char *host,const char *serv)
{
    DBG( dbg("pnetTCPConnect(host=%s,serv=%s)\n",host,serv) );
    return net_connect(PNET_IPv6,SOCK_STREAM,host,serv,NULL,NULL);
}
PNETSOCK
pnetUDPConnect2(int pfam,const char *host,const char *serv)
{
    DBG( dbg("pnetUPDConnect2(pfam=%d,host=%s,serv=%s)\n",pfam,host,serv) );
    return net_connect(pfam,SOCK_DGRAM,host,serv,NULL,NULL);
}
PNETSOCK
pnetUDPConnect(const char *host,const char *serv)
{
    DBG( dbg("pnetUDPConnect(host=%s,serv=%s)\n",host,serv) );
    return net_connect(PNET_IPv6,SOCK_DGRAM,host,serv,NULL,NULL);
}
PNETSOCK
pnetUDPConnectX(int pfam,const char * host,const char * serv,
			 const char *lhost,const char *lserv )
{
    DBG( dbg("pnetUDPConnectX(pfam=%d,host=%s,serv=%s,lhost=%s,lserv=%s)\n",
	      pfam,host,serv,lhost,lserv) );
    return net_connect(pfam,SOCK_DGRAM,host,serv,lhost,lserv);
}
