/*
 * Monitor tcp connections. 
 *
 * What we do is note when a TCP packet contains a SYN flag only, mark
 * that as a new connection, log the time. When we see a FIN, RST, we mark it
 * as closed, and log the time as well. The idea is to be able to see what's 
 * happening on a given interface.
 * 
 * $Id: tcpmon.c,v 1.5 2002/12/09 17:56:38 kingofgib Exp $
 */
# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <errno.h>

# include <pnet6.h>
# include <pnet6pkt.h>
# include <pnet6tlp.h>

static PNET_PKTACCESS	pa = 0;
static FILE*		fp = 0;

/*
 * Length of packet to capture is 60 byte IP header + 20 bytes TCP header,
 * not that this will also work for IPv6. The 14 bytes are for the Ethernet
 * header, but that would fit a PPP link device too (with header
 * length of 4 bytes).
 */

static int		grab_length = 14 + 60 + 20;

static int
open_interface( const char * ifname )
{
    pa = pnetPktOpenInterface( ifname, 1, 500, grab_length );

    if ( ! pa )
    {
	fprintf( stderr, "Cannot open \"%s\": %s\n",ifname,strerror( errno ) );
	return -1;
    }
    return 0;
}

static int
close_interface( void )
{
    if ( pa )
	pnetPktCloseInterface( pa );
    return 0;
}

static void
usage( void )
{
    printf( "usage: tcpmon [-i interface_name ]\n" );
}

static void
do_packet( pnet_byte * buf, int len )
{
    pnet_ip *		ip;
    pnet_tcp *		tcp;
    char 		tmp[64];
    char 		addr[ PNET_ADDR_BUFSIZ ];
    PNETADDRSTORAGE 	pas1;
    PNETADDRSTORAGE 	pas2;
    int			synpkt,synack,finpkt,rstpkt;

    ip = (pnet_ip *) buf;

    if ( ip->ip_p != 6 )
	return ;

    tcp= (pnet_tcp*) ( buf + pnetIP_HeaderLength( ip ) );
    len -= pnetIP_HeaderLength( ip );

    if ( len < (int)sizeof( pnet_tcp ) )
	return;

    synpkt = tcp->th_flags & TH_SYN;
    synack = synpkt && (tcp->th_flags & TH_ACK);
    finpkt = tcp->th_flags & TH_FIN;
    rstpkt = tcp->th_flags & TH_RST;

    if ( ! synpkt && !finpkt && !rstpkt && !synack )
	return;

    /*
     * Get source and destination
     */
    pnetIP_Source( ip, PNET_SADDR( pas1) );
    pnetIP_Destination( ip, PNET_SADDR( pas2 ) );

    if ( synpkt )
	fprintf( fp, "\n+---------\n");
    fprintf( fp, "%s: %s -> ",
	pnetGetTimeStamp( tmp, sizeof( tmp ) ),
	pnet_ntop( PNET_SADDR( pas1 ), addr, sizeof( addr ) ) );
    fprintf( fp, "%s: %s: src=%d, dst=%d %c\n",
	pnet_ntop( PNET_SADDR( pas2 ), addr, sizeof( addr ) ),
	synack ? "SYN/ACK" : synpkt ? "SYN" : finpkt ? "FIN" : "RST", 
	pnetTCP_SrcPort( tcp ), pnetTCP_DstPort(tcp), 
	synpkt ? '{' : '}' );

    return;
}

static void
tcpmon( void )
{
    PNetPacket	pkt;

    memset( &pkt, 0, sizeof( pkt ) );

    while ( 1 )
	if ( pnetPktNextPacket( pa, &pkt ) )
	    do_packet( (pnet_byte*)pkt.pkt_buf, (int)pkt.pkt_grablen );
}

int
main( int argc, const char ** argv )
{
    const char * ifname = strdup( "tun0" );
    const char * expr	= strdup( "tcp" );	/* Match any TCP packets */
    const char * fname  = strdup( "tcpmon.log" );
    int		 c, ret = 0;

    pnetInit();

    while ( (c = pnetGetopt( argc, argv, ":i:f:" )) != -1 )
	switch ( c )
	{
	case 'i': 	/* Interface	*/
	    ifname = strdup( pnetOptarg );
	    break;
	case 'f':
	    free( (void*)fname );
	    fname = strdup( pnetOptarg );
	    break;

	case '!': case ':':
	    fprintf( stderr, "Option '%c' requires an argument\n", pnetOptopt);
	    usage();
	    ret = 1;
	    goto quit;
	default:
	    fprintf( stderr, "Unknown argument '%c'\n", pnetOptopt );
	    usage();
	    ret = 1;
	    goto quit;
	}

    fp = fopen( pnetOptarg, "r" );
    if ( ! fp )
	{ fp = stdout; fname = "stdout"; }
    printf( "tcpmon: using interface %s\n", ifname );
    printf( "tcpmon: output goes to %s\n", fname );

    if ( open_interface( ifname ) )
	{ ret = 1; goto quit; }
    /*
     * We're not concerned with the link level header, so we do the
     * packet capture in cooked mode.
     */
    pnetPktSetReadMode( pa, PNET_READ_COOKED );

/*
    if ( pnetPktAddFilter( pa, expr ) )
    {
        fprintf( stderr, "Failed to install packet filter.\n" );
        return 1;
    }
*/

    tcpmon();

    ret = 0;
quit:
    if ( fp )
	fclose( fp );

    close_interface();
    free( (void*)ifname );
    free( (void*)expr );
    return 1;
}
