
/*
 * Copyright (c) 2001-2002 Packet Design, LLC.
 * All rights reserved.
 * 
 * Subject to the following obligations and disclaimer of warranty,
 * use and redistribution of this software, in source or object code
 * forms, with or without modifications are expressly permitted by
 * Packet Design; provided, however, that:
 * 
 *    (i)  Any and all reproductions of the source or object code
 *         must include the copyright notice above and the following
 *         disclaimer of warranties; and
 *    (ii) No rights are granted, in any manner or form, to use
 *         Packet Design trademarks, including the mark "PACKET DESIGN"
 *         on advertising, endorsements, or otherwise except as such
 *         appears in the above copyright notice or in the software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED BY PACKET DESIGN "AS IS", AND
 * TO THE MAXIMUM EXTENT PERMITTED BY LAW, PACKET DESIGN MAKES NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, REGARDING
 * THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY AND ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT.  PACKET DESIGN DOES NOT WARRANT, GUARANTEE,
 * OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF, OR THE RESULTS
 * OF THE USE OF THIS SOFTWARE IN TERMS OF ITS CORRECTNESS, ACCURACY,
 * RELIABILITY OR OTHERWISE.  IN NO EVENT SHALL PACKET DESIGN BE
 * LIABLE FOR ANY DAMAGES RESULTING FROM OR ARISING OUT OF ANY USE
 * OF THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE, OR CONSEQUENTIAL
 * DAMAGES, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, LOSS OF
 * USE, DATA OR PROFITS, HOWEVER CAUSED AND UNDER ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF PACKET DESIGN IS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Archie Cobbs <archie@freebsd.org>
 */

#ifndef _PDEL_STRUCTS_XML_H_
#define _PDEL_STRUCTS_XML_H_

/*
 * Callback function type used by structs_xml_input()
 */
typedef void	structs_xmllog_t(int sev, const char *fmt, ...);

/* Special pre-defined loggers for structs_xml_input() */
#define STRUCTS_LOGGER_NONE	((structs_xmllog_t *)0)	    /* discard output */
#define STRUCTS_LOGGER_STDERR	((structs_xmllog_t *)1)	    /* log to stderr */
#define STRUCTS_LOGGER_ALOG	((structs_xmllog_t *)2)	    /* log to alog() */

/* Flags to structs_xml_input() */
#define STRUCTS_XML_UNINIT	0x0001	/* data object needs initialization */
#define STRUCTS_XML_LOOSE	0x0002	/* unknown tags, nested attrib. ok */
#define STRUCTS_XML_SCAN	0x0004	/* don't try to decode data structure */
#define STRUCTS_XML_COMB_TAGS	0x0008	/* allow combined tags */

/* Flags to structs_xml_output() */
#define STRUCTS_XML_FULL	0x0001	/* even output default values */

__BEGIN_DECLS

/*
 * Create a data structure instance from XML input.
 *
 * The XML document element must match "elem_tag" and may have attributes.
 * No other tags may have attributes.
 *
 * If "attrp" is not NULL, any attributes associated with "elem_tag"
 * are stored there in a string allocated with memory type "attr_mtype".
 * Attributes are stored in a single buffer like so:
 *
 *	name1 '\0' value1 '\0'
 *	name2 '\0' value2 '\0'
 *		...
 *	nameN '\0' valueN '\0'
 *	'\0'
 *
 * It is assumed that "data" points to enough space to hold an item
 * of type "type".
 *
 * If successful (only), "data" must eventually be freed e.g., by calling
 * structs_free(type, NULL, data).
 *
 * "flags" bits are defined above. STRUCTS_XML_UNINIT means that the item
 * is not initialized and must be initialized first; otherwise it is
 * assumed to be already initialized and only those subfields specified
 * in the XML will be changed. STRUCTS_XML_LOOSE means any XML tags that
 * are unrecognized or nested (non-top level) attributes cause a warning
 * to be emitted but are otherwise ignored and non-fatal. Without it, these
 * will cause a fatal error.
 *
 * Returns 0 if successful, otherwise -1 and sets errno.
 */
extern int	structs_xml_input(const struct structs_type *type,
			const char *elem_tag, char **attrp,
			const char *attr_mtype, FILE *fp, void *data,
			int flags, structs_xmllog_t *logger);

/*
 * Output a data structure as XML.
 *
 * The XML document element is an "elem_tag" element with attributes
 * supplied by "attrs" (if non NULL) as described above.
 *
 * If "elems" is non-NULL, it must point to a NULL terminated list of
 * elements to output. Only elements appearing in the list are output.
 *
 * If STRUCTS_XML_FULL is not included in flags, then elements are omitted
 * if they are equal to their initialized (i.e., default) values.
 *
 * Returns 0 if successful, otherwise -1 and sets errno.
 */
extern int	structs_xml_output(const struct structs_type *type,
			const char *elem_tag, const char *attrs,
			const void *data, FILE *fp, const char **elems,
			int flags);

__END_DECLS

#endif	/* _PDEL_STRUCTS_XML_H_ */

