
/*
 * Copyright (c) 2001-2002 Packet Design, LLC.
 * All rights reserved.
 * 
 * Subject to the following obligations and disclaimer of warranty,
 * use and redistribution of this software, in source or object code
 * forms, with or without modifications are expressly permitted by
 * Packet Design; provided, however, that:
 * 
 *    (i)  Any and all reproductions of the source or object code
 *         must include the copyright notice above and the following
 *         disclaimer of warranties; and
 *    (ii) No rights are granted, in any manner or form, to use
 *         Packet Design trademarks, including the mark "PACKET DESIGN"
 *         on advertising, endorsements, or otherwise except as such
 *         appears in the above copyright notice or in the software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED BY PACKET DESIGN "AS IS", AND
 * TO THE MAXIMUM EXTENT PERMITTED BY LAW, PACKET DESIGN MAKES NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, REGARDING
 * THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY AND ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT.  PACKET DESIGN DOES NOT WARRANT, GUARANTEE,
 * OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF, OR THE RESULTS
 * OF THE USE OF THIS SOFTWARE IN TERMS OF ITS CORRECTNESS, ACCURACY,
 * RELIABILITY OR OTHERWISE.  IN NO EVENT SHALL PACKET DESIGN BE
 * LIABLE FOR ANY DAMAGES RESULTING FROM OR ARISING OUT OF ANY USE
 * OF THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE, OR CONSEQUENTIAL
 * DAMAGES, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, LOSS OF
 * USE, DATA OR PROFITS, HOWEVER CAUSED AND UNDER ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF PACKET DESIGN IS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Archie Cobbs <archie@freebsd.org>
 */

#ifndef _PDEL_NET_IF_UTIL_H_
#define _PDEL_NET_IF_UTIL_H_

struct sockaddr_dl;

__BEGIN_DECLS

/*
 * Gets the names of all system interfaces, returned as an array of
 * char * pointers. Caller must free each pointer and the array itself.
 * Returns the length of the array.
 *
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_get_list(char ***listp, const char *mtype);

/*
 * Get the interface type.
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_get_type(const char *ifname);

/*
 * Get the flags associated with an interface.
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_get_flags(const char *ifname);

/*
 * Set the flags associated with an interface.
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_set_flags(const char *ifname, int flags);

/*
 * Get the MTU associated with an interface.
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_get_mtu(const char *ifname);

/*
 * Set the MTU associated with an interface.
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_set_mtu(const char *ifname, u_int mtu);

/*
 * Get the link address associated with an interface.
 * Caller must free the returned structure.
 *
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_get_link_addr(const char *ifname,
			struct sockaddr_dl **sdlp, const char *mtype);

/*
 * Get a list of all IP addresses and netmasks configured on an interface.
 * Caller must free both lists. Returns the length of the arrays.
 *
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_get_ip_addrs(const char *ifname, struct in_addr **iplistp,
			struct in_addr **nmlistp, const char *mtype);

/*
 * Get the first IP address and netmask configured on an interface.
 * "ipp" and/or "nmp" may be NULL. Returns -1 and sets errno on error;
 * in particular, if there are no addresses, then errno will be ENOENT.
 */
extern int	if_get_ip_addr(const char *ifname,
			struct in_addr *ipp, struct in_addr *nmp);

/*
 * Add an IP address assignment to a broadcast or p2p interface.
 * The "dest" should be 0.0.0.0 for non-p2p interfaces.
 *
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_add_ip_addr(const char *iface, struct in_addr ip,
			struct in_addr mask, struct in_addr dest);

/*
 * Remove an IP address assignment from a broadcast or p2p interface.
 * The "dest" should be 0.0.0.0 for non-p2p interfaces.
 *
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_del_ip_addr(const char *iface, struct in_addr ip,
			struct in_addr mask, struct in_addr dest);

/*
 * Get an ARP entry.
 *
 * Returns -1 and sets errno if there was a problem.
 * In particular, errno = ENOENT if entry not found.
 */
extern int	if_get_arp(struct in_addr ip, u_char *ether);

/*
 * Set (or remove, if ether == NULL) an ARP entry.
 *
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_set_arp(struct in_addr ip, const u_char *ether,
			int temp, int publish);

/*
 * Flush all ARP entries.
 *
 * Returns -1 and sets errno if there was a problem.
 */
extern int	if_flush_arp(void);

__END_DECLS

#endif	/* _PDEL_NET_IF_UTIL_H_ */

