/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __libnd_protocol_plugin_h
#define __libnd_protocol_plugin_h

#include <ltdl.h>
#include <libnd_protocol.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Protocol plugins -- Netdude itself doesn't know any protocols,
   they're all implemented as plugins to keep things modularized. */

typedef const char *(*LND_ProtoPluginInfoFunc)(void);

typedef struct lnd_protoplugin
{
  const char   *(*name) (void);
  const char   *(*description) (void);
  const char   *(*author) (void);
  const char   *(*version) (void);

  /* Generic initialization/cleanup. Use as needed. */

  LND_Protocol *(*init) (void);
  int           (*cleanup) (void);

  /* full path to shared lib filename */
  char         *filename;
  lt_dlhandle   lt;

} LND_ProtoPlugin;


/**
 * LND_ProtoPluginFunc - signature of plugin iteration callbacks.
 * @plugin: iterated plugin.
 * @user_data: arbitrary user data.
 *
 * This is the signature of functions that can be passed to
 * libnd_plugin_foreach().
 */
typedef void (*LND_ProtoPluginFunc)(LND_ProtoPlugin *plugin,
				    void *user_data);


/**
 * libnd_proto_plugin_init - initializes protocol plugins.
 *
 * The function scans the plugin directories, reads in the
 * plugins it finds and hooks them into the main window
 * and the plugin registry.
 */
void          libnd_proto_plugin_init(void);

/**
 * libnd_proto_plugin_add - creates, registers, and returns a plugin structure.
 * @name: plugin name info hook.
 * @description: plugin description hook.
 * @author: plugin author hook.
 * @version: version info hook.
 *
 * The function returns a pointer to a new plugin structure,
 * registered internally as a new plugin, but with all function
 * pointers pointing to the dummy callbacks. Any of the hooks may be
 * NULL, in which case the default hooks are used.
 */
LND_ProtoPlugin *libnd_proto_plugin_add(LND_ProtoPluginInfoFunc name,
					LND_ProtoPluginInfoFunc description,
					LND_ProtoPluginInfoFunc author,
					LND_ProtoPluginInfoFunc version);

/**
 * libnd_proto_plugin_get_name - returns plugin name.
 * @pp: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string with the plugin's name.
 *
 * Returns: name of the protocol plugin.
 */
const char   *libnd_proto_plugin_get_name(LND_ProtoPlugin *pp);

/**
 * libnd_proto_plugin_get_description - returns plugin description string.
 * @pp: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string describing the plugin.
 *
 * Returns: description string.
 */
const char   *libnd_proto_plugin_get_description(LND_ProtoPlugin *pp);

/**
 * libnd_proto_plugin_get_version - returns plugin version string.
 * @pp: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string describing the plugin version.
 *
 * Returns: version string.
 */
const char   *libnd_proto_plugin_get_version(LND_ProtoPlugin *pp);

/**
 * libnd_proto_plugin_get_author - returns plugin author.
 * @pp: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string with the author's name.
 *
 * Returns: author name string.
 */
const char   *libnd_proto_plugin_get_author(LND_ProtoPlugin *pp);


/**
 * libnd_proto_plugin_foreach - protocol plugin iteration.
 * @callback: callback to call for each protocol plugin.
 * @user_data: arbitrary user data passed through to @callback.
 *
 * The function iterates over the registered protocol plugins and applies
 * a callback function to each of them.
 */
void          libnd_proto_plugin_foreach(LND_ProtoPluginFunc callback,
					 void *user_data);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
