/* -*- Mode: C++; c-default-style: "k&r"; indent-tabs-mode: nil; tab-width: 2; c-basic-offset: 2 -*- */

/* libmwaw
* Version: MPL 2.0 / LGPLv2+
*
* The contents of this file are subject to the Mozilla Public License Version
* 2.0 (the "License"); you may not use this file except in compliance with
* the License or as specified alternatively below. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
*
* Software distributed under the License is distributed on an "AS IS" basis,
* WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
* for the specific language governing rights and limitations under the
* License.
*
* Major Contributor(s):
* Copyright (C) 2002 William Lachance (wrlach@gmail.com)
* Copyright (C) 2002,2004 Marc Maurer (uwog@uwog.net)
* Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
* Copyright (C) 2006, 2007 Andrew Ziem
* Copyright (C) 2011, 2012 Alonso Laurent (alonso@loria.fr)
*
*
* All Rights Reserved.
*
* For minor contributions see the git repository.
*
* Alternatively, the contents of this file may be used under the terms of
* the GNU Lesser General Public License Version 2 or later (the "LGPLv2+"),
* in which case the provisions of the LGPLv2+ are applicable
* instead of those above.
*/

/*
 * Parser to Canvas5 text document ( style part )
 *
 */
#ifndef CANVAS5_STYLE_MANAGER
#  define CANVAS5_STYLE_MANAGER

#include <string>
#include <utility>
#include <vector>

#include <librevenge/librevenge.h>

#include "libmwaw_internal.hxx"

#include "MWAWDebug.hxx"
#include "MWAWGraphicStyle.hxx"
#include "MWAWInputStream.hxx"
#include "MWAWPosition.hxx"

class MWAWFont;
class MWAWParagraph;

namespace Canvas5Structure
{
struct Stream;
}

namespace Canvas5StyleManagerInternal
{
struct ColorStyle;
struct State;
}

class Canvas5Graph;
class Canvas5Parser;

/** \brief the main class to read the style part of Canvas 5 file
 *
 *
 *
 */
class Canvas5StyleManager
{
  friend class Canvas5Graph;
  friend class Canvas5Parser;

public:
  //! a structure use to store a list of styles
  struct StyleList {
    //! constructor
    StyleList()
      : m_fonts()
      , m_paragraphs()
    {
    }
    //! the fonts and the paragraph id
    std::vector<std::pair<MWAWFont, int> > m_fonts;
    //! the paragraph list and the tab id
    std::vector<std::pair<MWAWParagraph, int> > m_paragraphs;
  };

  //! constructor
  explicit Canvas5StyleManager(Canvas5Parser &parser);
  //! destructor
  virtual ~Canvas5StyleManager();

  /** returns the file version */
  int version() const;

protected:
  // interface

  /** try to update the line style given the stroke id and returns the number of lines(plin)

      \note in case of plin, after retrieving the number of lines, use lineId to define the line's style to set
      and give a offset pointer to retrieve this line offset
   */
  bool updateLineStyle(int sId, MWAWGraphicStyle &style, int &numLines, int lineId=-1, float *offset=nullptr);
  //! try to update the line color given the color id
  bool updateLineColor(int cId, MWAWGraphicStyle &style);
  //! try to update the surface color given the color id
  bool updateSurfaceColor(int cId, MWAWGraphicStyle &style);

  //
  // Intermediate level
  //

  //! try to read the arrows zones
  bool readArrows(Canvas5Structure::Stream &stream);
  //! try to read a character style
  bool readCharStyle(Canvas5Structure::Stream &stream, int id, std::pair<MWAWFont, int> &fontId);
  //! try to read the character styles
  bool readCharStyles(Canvas5Structure::Stream &stream);
  //! try to read the ink color zones
  bool readColors(Canvas5Structure::Stream &stream);
  //! try to read the dashes
  bool readDashes(Canvas5Structure::Stream &stream);
  //! try to read the fonts names
  bool readFonts(Canvas5Structure::Stream &stream, int numFonts);
  //! read the list of formats, mainly an unit's conversion table
  bool readFormats(Canvas5Structure::Stream &stream);
  //! read the pen size (header file)
  bool readPenSize(Canvas5Structure::Stream &stream);
  //! read the pen styles
  bool readPenStyles(Canvas5Structure::Stream &stream);
  //! try to read the stroke styles
  bool readStrokes(Canvas5Structure::Stream &stream);
  //! try to read a style
  bool readStyle(Canvas5Structure::Stream &stream, int id, StyleList *styles=nullptr);
  //! try to read the styles
  bool readStyles(Canvas5Structure::Stream &stream);

  //
  // Windows RSRC
  //

  //
  // Low level
  //

  //! try to read an arrow
  bool readArrow(Canvas5Structure::Stream &stream, MWAWGraphicStyle::Arrow &arrow, long len);
  //! try to read a color: 12 unknown 4 components followed by a name
  bool readColor(Canvas5Structure::Stream &stream, MWAWColor &color, std::string &extra);
  //! try to read a color style
  bool readColorStyle(Canvas5Structure::Stream &stream, Canvas5StyleManagerInternal::ColorStyle &color, long len);
  //! try to read a symbol
  bool readSymbol(Canvas5Structure::Stream &stream, Canvas5StyleManagerInternal::ColorStyle &color, long len);
  //! try to read a texture
  bool readTexture(Canvas5Structure::Stream &stream, Canvas5StyleManagerInternal::ColorStyle &color, long len);
  /** try to read an unknown vectorized graphic format used to store symbol, texture and arrow

      \note because this format is unknown, we fill the texture.m_texture and texture.m_color when we find a texture
            and will fill the list of type (expected 11) is this vector is defined.
   */
  bool readVKFL(Canvas5Structure::Stream &stream, Canvas5StyleManagerInternal::ColorStyle &texture, long len, std::vector<int> *typeList=nullptr);

private:
  Canvas5StyleManager(Canvas5StyleManager const &orig) = delete;
  Canvas5StyleManager &operator=(Canvas5StyleManager const &orig) = delete;

protected:
  //
  // data
  //
  //! the parser state
  MWAWParserStatePtr m_parserState;

  //! the state
  std::shared_ptr<Canvas5StyleManagerInternal::State> m_state;

  //! the main parser;
  Canvas5Parser *m_mainParser;
};
#endif
// vim: set filetype=cpp tabstop=2 shiftwidth=2 cindent autoindent smartindent noexpandtab:
