/* -*- Mode: C++; c-default-style: "k&r"; indent-tabs-mode: nil; tab-width: 2; c-basic-offset: 2 -*- */

/* libmwaw
* Version: MPL 2.0 / LGPLv2+
*
* The contents of this file are subject to the Mozilla Public License Version
* 2.0 (the "License"); you may not use this file except in compliance with
* the License or as specified alternatively below. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
*
* Software distributed under the License is distributed on an "AS IS" basis,
* WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
* for the specific language governing rights and limitations under the
* License.
*
* Major Contributor(s):
* Copyright (C) 2002 William Lachance (wrlach@gmail.com)
* Copyright (C) 2002,2004 Marc Maurer (uwog@uwog.net)
* Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
* Copyright (C) 2006, 2007 Andrew Ziem
* Copyright (C) 2011, 2012 Alonso Laurent (alonso@loria.fr)
*
*
* All Rights Reserved.
*
* For minor contributions see the git repository.
*
* Alternatively, the contents of this file may be used under the terms of
* the GNU Lesser General Public License Version 2 or later (the "LGPLv2+"),
* in which case the provisions of the LGPLv2+ are applicable
* instead of those above.
*/

#include <strings.h>

#include <cmath>
#include <iomanip>
#include <iostream>
#include <limits>
#include <map>
#include <set>
#include <sstream>
#include <stack>

#include <librevenge/librevenge.h>

#include "MWAWFont.hxx"
#include "MWAWFontConverter.hxx"
#include "MWAWGraphicListener.hxx"
#include "MWAWGraphicStyle.hxx"
#include "MWAWParagraph.hxx"
#include "MWAWParser.hxx"

#include "Canvas5Parser.hxx"

#include "Canvas5Structure.hxx"
#include "Canvas5StyleManager.hxx"

#include "libmwaw_internal.hxx"

/** Internal: the structures of a Canvas5StyleManager */
namespace Canvas5StyleManagerInternal
{

////////////////////////////////////////
//! Internal: the color style of a Canvas5StyleManager
struct ColorStyle {
  //! constructor
  ColorStyle()
    : m_type(1)
    , m_nameId(0)
    , m_color()
    , m_gradient()
    , m_hatch()
    , m_textureDim(0,0)
    , m_texture()
    , m_isBad(false)
  {
  }
  friend std::ostream &operator<<(std::ostream &o, ColorStyle const &color)
  {
    if (color.m_type==0)
      o << "transparent,";
    else if (color.m_type==1)
      o << color.m_color << ",";
    else
      o << "type=" << Canvas5Parser::getString(color.m_type) << ",";
    if (color.m_nameId)
      o << "N" << color.m_nameId << ",";
    return o;
  }
  //! the type
  unsigned m_type;
  //! the name id
  int m_nameId;
  //! the basic color
  MWAWColor m_color;
  //! the gradient
  MWAWGraphicStyle::Gradient m_gradient;
  //! the hatch
  MWAWGraphicStyle::Hatch m_hatch;
  //! the texture dimension
  MWAWVec2i m_textureDim;
  //! the embedded objet (texture)
  MWAWEmbeddedObject m_texture;
  //! a flag to know if the color is read
  bool m_isBad;
};

////////////////////////////////////////
//! Internal: the pen style of a Canvas5StyleManager
struct PenStyle {
  /// a line of a Canvas5StyleManager pen style
  struct Line {
    //! constructor
    Line()
      : m_size(1,1)
      , m_offset(0)
      , m_color(MWAWColor::black())
    {
    }
    //! the line width
    MWAWVec2f m_size;
    //! the offset
    float m_offset;
    //! the line color
    MWAWColor m_color;
  };
  //! constructor
  PenStyle()
    : m_type(1)
    , m_size(1,1)
    , m_usePenColor(true)
    , m_lines()
  {
    for (auto &c : m_colors) c=MWAWColor::black();
  }
  friend std::ostream &operator<<(std::ostream &o, PenStyle const &pen)
  {
    if (pen.m_type!=1)
      o << "type=" << Canvas5Parser::getString(pen.m_type) << ",";
    if (pen.m_size!=MWAWVec2f(1,1))
      o << "size=" << pen.m_size << ",";
    return o;
  }
  //! the type
  unsigned m_type;
  //! the pen size
  MWAWVec2f m_size;
  //! the neo color
  MWAWColor m_colors[2];
  //! use pen ink
  bool m_usePenColor;
  //! the plin lines
  std::vector<Line> m_lines;
};

////////////////////////////////////////
//! Internal: the stroke style of a Canvas5StyleManager
struct Stroke {
  //! constructor
  Stroke()
    : m_type(1)
    , m_penId(0)
    , m_dashId(0)
  {
    for (auto &id : m_arrowId) id=0;
  }
  friend std::ostream &operator<<(std::ostream &o, Stroke const &stroke)
  {
    if (stroke.m_type!=1)
      o << "type=" << Canvas5Parser::getString(stroke.m_type) << ",";
    if (stroke.m_penId)
      o << "Pe" << stroke.m_penId << ",";
    if (stroke.m_dashId)
      o << "Da" << stroke.m_dashId << ",";
    for (int i=0; i<2; ++i) {
      if (!stroke.m_arrowId[i]) continue;
      o << (i==0 ? "beg" : "end") << "=Ar" << stroke.m_arrowId[i] << ",";
    }
    return o;
  }
  //! the type
  unsigned m_type;
  //! the pen id
  int m_penId;
  //! the dash id
  int m_dashId;
  //! the arrow id (beg/end)
  int m_arrowId[2];
};

////////////////////////////////////////
//! Internal: the state of a Canvas5StyleManager
struct State {
  //! constructor
  State()
    : m_idToArrow()
    , m_idToColor()
    , m_idToPen()
    , m_idToDash()
    , m_idToStroke()
  {
  }

  //! the id to arrow map
  std::map<int, MWAWGraphicStyle::Arrow> m_idToArrow;
  //! the id to color style map
  std::map<int, ColorStyle> m_idToColor;
  //! the id to pen style map
  std::map<int, PenStyle> m_idToPen;
  //! the id to dash map
  std::map<int, std::vector<float> > m_idToDash;
  //! the id to stroke style map
  std::map<int, Stroke> m_idToStroke;
};

}

////////////////////////////////////////////////////////////
// constructor/destructor, ...
////////////////////////////////////////////////////////////
Canvas5StyleManager::Canvas5StyleManager(Canvas5Parser &parser)
  : m_parserState(parser.getParserState())
  , m_state(new Canvas5StyleManagerInternal::State)
  , m_mainParser(&parser)
{
}

Canvas5StyleManager::~Canvas5StyleManager()
{ }

int Canvas5StyleManager::version() const
{
  return m_parserState->m_version;
}

////////////////////////////////////////////////////////////
//
// Intermediate level
//
////////////////////////////////////////////////////////////

bool Canvas5StyleManager::readArrows(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  long pos=input->tell();

  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  f << "Entries(Arrow):";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  if (!m_mainParser->readUsed(stream, "Arrow"))
    return false;

  if (!m_mainParser->readIndexMap(stream, "Arrow",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &, long len) {
  auto lInput=lStream.input();
    long lPos=lInput->tell();
    libmwaw::DebugStream lF;
    auto &asciiFile=lStream.ascii();
    lF << "Arrow" << id << ",";
    MWAWGraphicStyle::Arrow arrow;
    if (!readArrow(lStream, arrow, len))
      lF << "###";
    else
      m_state->m_idToArrow[id]=arrow;
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  }))
  return false;
  return true;
}

bool Canvas5StyleManager::readArrow(Canvas5Structure::Stream &stream, MWAWGraphicStyle::Arrow &arrow, long len)
{
  auto input=stream.input();
  long pos=input->tell();
  if (len<24 || !input->checkPosition(pos+24)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readArrow: file is to short\n"));
    return false;
  }
  auto &ascFile=stream.ascii();
  libmwaw::DebugStream f;
  long dataLen=long(input->readULong(4));
  if (dataLen==len) dataLen=len-24; // can happen sometimes
  if (pos+24+dataLen<pos+24 || 24+dataLen>len) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readArrow: can not read the arrow's data size\n"));
    return false;
  }
  ascFile.addDelimiter(input->tell(),'|');
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  input->seek(pos+24, librevenge::RVNG_SEEK_SET);

  if (24+dataLen!=len) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readArrow: find unknown length data\n"));
    ascFile.addPos(pos+24+dataLen);
    ascFile.addNote("Arrow-End:###");
  }

  if (dataLen==0) {
    arrow=MWAWGraphicStyle::Arrow();
    return true;
  }
  Canvas5StyleManagerInternal::ColorStyle texture;
  std::vector<int> typeList;
  if (!readVKFL(stream, texture, dataLen, &typeList))
    return false;
  arrow=MWAWGraphicStyle::Arrow::plain();
  if (typeList.size()==1) {
    switch (typeList[0]) {
    case 1:
    case 2:
      arrow=MWAWGraphicStyle::Arrow(5, MWAWBox2i(MWAWVec2i(0,0),MWAWVec2i(20,30)), "M1013 1491l118 89-567-1580-564 1580 114-85 136-68 148-46 161-17 161 13 153 46z", false);
      break;
    case 7:
      arrow=MWAWGraphicStyle::Arrow(5, MWAWBox2i(MWAWVec2i(0,0),MWAWVec2i(1131,1131)), "M462 1118l-102-29-102-51-93-72-72-93-51-102-29-102-13-105 13-102 29-106 51-102 72-89 93-72 102-50 102-34 106-9 101 9 106 34 98 50 93 72 72 89 51 102 29 106 13 102-13 105-29 102-51 102-72 93-93 72-98 51-106 29-101 13z", false);
      break;
    case 10:
      arrow=MWAWGraphicStyle::Arrow(5, MWAWBox2i(MWAWVec2i(0,0),MWAWVec2i(20,30)), "m10 0l-10 30h20z", false);
      break;
    default:
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readArrow: find unexpected arrow with type %d\n", typeList[0]));
      break;
    }
  }
  else if (typeList.size()==2 && typeList[0]==1 && typeList[1]==1)
    arrow=MWAWGraphicStyle::Arrow(10, MWAWBox2i(MWAWVec2i(0,0),MWAWVec2i(40,35)), "m20 0l-20 0 l0 4 l20 0 l-10 30 l20 0 l-10 -30 l20 0 l0 -4z", false);
  else if (typeList.size()==3 && typeList[0]==10)
    arrow=MWAWGraphicStyle::Arrow(5, MWAWBox2i(MWAWVec2i(0,0),MWAWVec2i(20,33)), "m10 0l-10 30 l10 3 l10 -3z", false);
  else {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readArrow: find unexpected arrow with size=%d\n", int(typeList.size())));
  }

  return true;
}


bool Canvas5StyleManager::readColor(Canvas5Structure::Stream &stream, MWAWColor &color, std::string &extra)
{
  auto input=stream.input();
  long pos=input->tell();
  extra="";
  if (!input->checkPosition(pos+24)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readColor: file is to short\n"));
    return false;
  }
  libmwaw::DebugStream f;
  for (int i=0; i<6; ++i) {
    int val=(i>0 && i<5) ? int(input->readULong(2)) : int(input->readLong(2));
    if (val==(i==5 ? -1 : 0)) continue;
    if (i>0 && i<5)
      f << "f" << i << "=" << std::hex << val << std::dec << ",";
    else
      f << "f" << i << "=" << val << ",";
  }
  unsigned char cols[4];
  for (auto &c : cols) c=(unsigned char)(input->readULong(2)>>8);
  // cmyk, rgb, sepp, pton, trum, toyo
  unsigned name=unsigned(input->readULong(4));
  f << Canvas5Parser::getString(name) << ",";
  if (name==0x72676220) // rgb
    color=MWAWColor(cols[0],cols[1],cols[2],(unsigned char)(255-cols[3]));
  else {
    if (name==0x70746f6e) { // pton
      f << "##";
      static bool first=true;
      if (first) {
        MWAW_DEBUG_MSG(("Canvas5StyleManager::readColor: this file contains pantom color, there will ne be converted correctly\n"));
        first=false;
      }
    }
    color=MWAWColor::colorFromCMYK(cols[0],cols[1],cols[2],cols[3]);
  }
  f << color << ",";
  extra=f.str();
  return true;
}

bool Canvas5StyleManager::readColors(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  long pos=input->tell();

  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  f << "Entries(Color):type,";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  if (!m_mainParser->readExtendedHeader(stream, 8, "Color",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &, long) {
  auto lInput=lStream.input();
    long lPos=lInput->tell();
    libmwaw::DebugStream lF;
    auto &asciiFile=lStream.ascii();
    lF << "Co" << id << "[type]:";
    Canvas5StyleManagerInternal::ColorStyle color;
    color.m_type=unsigned(lInput->readULong(4)); // 1, TXUR: texture, ObFl, htch: hatch, vkfl: symbol
    if (color.m_type!=1)
      lF << color << ",";
    color.m_nameId=int(lInput->readLong(4));
    if (color.m_nameId)
      lF << "id[name]=" << color.m_nameId << ",";
    m_state->m_idToColor[id]=color;
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  }))
  return false;

  if (!m_mainParser->readIndexMap(stream, "Color",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &, long len) {
  auto lInput=lStream.input();
    long lPos=lInput->tell();
    libmwaw::DebugStream lF;
    auto &asciiFile=lStream.ascii();
    lF << "Co" << id << ",";
    Canvas5StyleManagerInternal::ColorStyle color;
    auto const &it=m_state->m_idToColor.find(id);
    if (it!=m_state->m_idToColor.end()) color=it->second;
    if (readColorStyle(lStream, color, len))
      m_state->m_idToColor[id]=color;
    else
      lF << "###";
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  }))
  return false;

  if (!m_mainParser->readUsed(stream, "Color"))
    return false;

  std::multimap<int,int> nameIdToColor;
  for (auto &it : m_state->m_idToColor) {
    if (it.second.m_nameId)
      nameIdToColor.insert(std::make_pair(it.second.m_nameId, it.first));
  }

  pos=input->tell();
  f.str("");
  f << "Color:names";
  int N;
  if (!m_mainParser->readDataHeader(stream, 4, N)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readColors: can not read the last zone N\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  f << "N=" << N << ",";
  f << "id=[";
  for (int i=0; i<N; ++i) f << input->readLong(4) << ",";
  f << "],";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  return m_mainParser->readIndexMap(stream, "Color",
  [&nameIdToColor](Canvas5Structure::Stream &lStream, int id, std::string const &, long len) {
    auto lInput=lStream.input();
    long lPos=lInput->tell();
    libmwaw::DebugStream lF;
    auto &asciiFile=lStream.ascii();
    std::string name;
    for (int i=0; i<int(len); ++i) {
      char c=char(lInput->readULong(1));
      if (c==0)
        break;
      name+=c;
    }
    lF << name << ",";
    auto it = nameIdToColor.find(id);
    lF << "[";
    while (it!=nameIdToColor.end() && it->first==id) {
      lF << "Co" << it->second << ",";
      ++it;
    }
    lF << "],";
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  });
}

bool Canvas5StyleManager::readColorStyle(Canvas5Structure::Stream &stream, Canvas5StyleManagerInternal::ColorStyle &color, long len)
{
  auto input=stream.input();
  long pos=input->tell();
  if (len<0 || !input->checkPosition(pos+len)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle: the zone seems too short\n"));
    return false;
  }
  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  switch (color.m_type) {
  case 0:
    if (len==24) { // gray?
      std::string extra;
      if (!readColor(stream, color.m_color, extra)) {
        color.m_isBad=true;
        f << "##";
      }
      f << extra << ",";
      break;
    }
    if (len!=4) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle: unexpected length\n"));
      f << "###";
      break;
    }
    color.m_color=MWAWColor(0,0,0,0);
    for (int i=0; i<2; ++i) {
      int val=int(input->readLong(2));
      if (val)
        f << "f" << i << "=" << val << ",";
    }
    break;
  case 1: {
    if (len!=24) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle: unexpected length\n"));
      f << "###";
      break;
    }
    std::string extra;
    if (!readColor(stream, color.m_color, extra)) {
      color.m_isBad=true;
      f << "##";
    }
    f << extra << ",";
    break;
  }
  case 0x68746368: { // htch
    color.m_isBad=true;
    color.m_color=MWAWColor(0,0,0,0);
    f << "htch,";
    if (len<8+28+76) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle[hatch]: unexpected length\n"));
      f << "###";
      break;
    }
    unsigned sz=unsigned(input->readULong(4));
    f << "first[sz]=" << sz << ",";
    int N=int(input->readULong(2));
    f << "N=" << N << ",";
    if (int(sz)<0 || long(8+sz)>len || (len-long(sz)-8)/(28+76)<N || int(8+sz+(28+76)*unsigned(N))<8+28+76 || len<long(8+sz+(28+76)*unsigned(N))) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle[hatch]: the number of line seems bad\n"));
      f << "###";
      break;
    }
    int val=int(input->readULong(2));
    if (val!=0xf6f6) f << "fl=" << std::hex << val << std::dec << ",";

    libmwaw::DebugStream f2;
    if (sz) {
      long actPos=input->tell();
      f2.str("");
      f2 << "Color[back]:";
      Canvas5StyleManagerInternal::ColorStyle backColor;
      backColor.m_type=unsigned(input->readULong(4));
      f2 << backColor;
      if (!readColorStyle(stream, backColor, sz-4))
        f2 << "###";
      else {
        if (!backColor.m_isBad) {
          color.m_isBad=false;
          color.m_color=backColor.m_color;
        }
        ascFile.addPos(actPos+4);
        ascFile.addNote(f2.str().c_str());
      }

      ascFile.addPos(actPos);
      ascFile.addNote(f2.str().c_str());
      input->seek(actPos+sz, librevenge::RVNG_SEEK_SET);
    }
    if (N<=0)
      break;
    auto &hatch=color.m_hatch;
    hatch.m_type= N==1 ? MWAWGraphicStyle::Hatch::H_Single :
                  N==2 ? MWAWGraphicStyle::Hatch::H_Double : MWAWGraphicStyle::Hatch::H_Triple;
    float offset=0;
    for (int i=0; i<N; ++i) {
      long actPos=input->tell();
      f2.str("");
      f2 << "Color[htch-L" << i << "]:";
      val=int(input->readULong(4));
      if (val!=0x10000)
        f2 << "w=" << float(val)/65536 << ",";
      val=int(input->readLong(4));
      float angl=float(val)/65536;
      if (val)
        f2 << "angle=" << angl << ",";
      if (i==0)
        hatch.m_rotation=90-angl;
      val=int(input->readULong(4));
      offset+=float(val)/65536;
      f << "offset=" << float(val)/65536 << ",";
      val=int(input->readLong(4));
      if (val)
        f << "orig=" << float(val)/65536 << ",";
      ascFile.addDelimiter(input->tell(),'|');
      input->seek(actPos+80, librevenge::RVNG_SEEK_SET);
      ascFile.addPos(actPos);
      ascFile.addNote(f2.str().c_str());

      actPos=input->tell();
      f2.str("");
      f2 << "Color[htch-C" << i << "]:";
      std::string extra;
      MWAWColor col;
      if (!readColor(stream, col, extra))
        input->seek(actPos+24, librevenge::RVNG_SEEK_SET);
      else
        hatch.m_color=col;
      f2 << extra;
      ascFile.addPos(actPos);
      ascFile.addNote(f2.str().c_str());
    }
    hatch.m_distance=offset/float(N)/72;
    break;
  }
  case 0x4f62466c: { // ObFl
    color.m_isBad=true;
    f << "ObFl,";
    if (len<56) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle[ObFL]: unexpected length\n"));
      f << "###";
      break;
    }
    size_t N=size_t(input->readULong(4));
    f << "N=" << N << ",";
    if (long(N)<0 || (len-56)/28<long(N) || long(56+N*28)<56 || len<long(56+N*28)) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle[ObFL]: can not read N\n"));
      f << "###";
      break;
    }
    MWAWGraphicStyle::Gradient &gradient=color.m_gradient;
    int type=int(input->readLong(4));
    if (type>=1 && type<=5) {
      char const *wh[]= {"radial", "directional", "shape", "rectangular",
                         "elliptical"
                        };
      MWAWGraphicStyle::Gradient::Type const types[]= {
        MWAWGraphicStyle::Gradient::G_Radial, MWAWGraphicStyle::Gradient::G_Linear,
        MWAWGraphicStyle::Gradient::G_Rectangular, MWAWGraphicStyle::Gradient::G_Rectangular,
        MWAWGraphicStyle::Gradient::G_Ellipsoid
      };
      gradient.m_type=types[type-1];
      f << wh[type-1] << ",";
    }
    else {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle[ObFL]: find unknown type\n"));
      f << "type=" << type << ",";
    }
    f << "pts=[";
    MWAWVec2f pts[3];
    for (auto &pt : pts) {
      float fDim[2];
      for (auto &d : fDim) d=float(input->readLong(4))/65536;
      pt=MWAWVec2f(fDim[1],fDim[0]);
      f << pt << ",";
    }
    f << "],";
    if (pts[0]!=pts[1]) {
      MWAWVec2f dir=pts[1]-pts[0];
      gradient.m_angle=90-180*std::atan2(dir[1],dir[0])/float(M_PI);
    }
    float fDim[4];
    for (auto &d : fDim) d=float(input->readLong(4))/65536;
    MWAWBox2f box=MWAWBox2f(MWAWVec2f(fDim[1],fDim[0]),MWAWVec2f(fDim[3],fDim[2]));
    gradient.m_percentCenter=box.center();
    f << "box=" << box << ",";
    int val=int(input->readULong(4));
    if (val==1)
      f << "rainbow,";
    else if (val) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle[ObFL]: find unknown rainbow type\n"));
      f << "##rainbow=" << val << ",";
    }
    val=int(input->readULong(2));
    if (val==1)
      f << "rainbow[inverted],";
    else if (val)
      f << "#rainbow[inverted]=" << val << ",";
    val=int(input->readULong(1));
    if (val)
      f << "h0=" << val << ",";
    val=int(input->readULong(1));
    if (val!=0x13)
      f << "h1=" << val << ",";
    input->seek(pos+56, librevenge::RVNG_SEEK_SET);
    libmwaw::DebugStream f2;
    gradient.m_stopList.resize(size_t(N));
    for (size_t i=0; i<size_t(N); ++i) {
      long actPos=input->tell();
      f2.str("");
      f2 << "Color[ObFl" << i << "]:";
      MWAWGraphicStyle::Gradient::Stop &stop=gradient.m_stopList[size_t(N)-1-i];
      stop.m_offset=1-float(input->readLong(4))/100;
      f2 << "pos=" << stop.m_offset << ",";
      std::string extra;
      if (!readColor(stream, stop.m_color, extra))
        input->seek(actPos+24, librevenge::RVNG_SEEK_SET);
      f2 << extra;
      ascFile.addPos(actPos);
      ascFile.addNote(f2.str().c_str());
    }
    break;
  }
  case 0x54585552: // TXUR
    if (!readTexture(stream, color, len))
      f << "###";
    break;
  case 0x766b666c: // vkfl
    if (!readSymbol(stream, color, len))
      f << "###";
    break;
  default: {
    static bool first=true;
    if (first) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readColorStyle: can not read some complex ink color\n"));
      first=false;
    }
    f << "type=" << Canvas5Parser::getString(color.m_type) << "##";
    color.m_isBad=true;
    break;
  }
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  return true;
}

bool Canvas5StyleManager::readVKFL(Canvas5Structure::Stream &stream, Canvas5StyleManagerInternal::ColorStyle &texture, long len,
                                   std::vector<int> *typeList)
{
  if (len==0)
    return true;
  auto input=stream.input();
  long pos=input->tell();
  long endPos=pos+len;
  if (len<180 || !input->checkPosition(endPos)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: the zone seems too short\n"));
    return false;
  }
  auto &ascFile=stream.ascii();
  libmwaw::DebugStream f;

  f << "Entries(Vkfl):";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  MWAWEntry entries[2];
  if (!m_mainParser->readUnknownHeader(stream, entries, len, "Vkfl")) {
    MWAW_DEBUG_MSG(("Canvas5Graph::readVKFL: can not read an effect length\n"));
    ascFile.addPos(pos);
    ascFile.addNote("###");
    return false;
  }
  texture.m_isBad=true;
  std::vector<MWAWEntry> cEntries;
  if (entries[1].valid()) {
    input->seek(entries[1].begin(), librevenge::RVNG_SEEK_SET);
    int shape=0;
    bool first=true;
    while (input->tell()+44<=entries[1].end()) {
      pos=input->tell();
      f.str("");
      f << "Vkfl-s" << ++shape << ":";
      int type=int(input->readLong(4)); // find 1-14
      f << "type=" << type << ",";
      bool hasNoHeader=false;
      if (first && type<5) {
        /* fixme: normally, this zone begins with a node with type=0xb,
           but sometimes, it begins with 3 sub zones: a color, ... */
        if (input->readLong(4)==-1)
          hasNoHeader=true;
        input->seek(hasNoHeader ? -8 : -4, librevenge::RVNG_SEEK_CUR);
      }
      int N=0;
      if (!hasNoHeader) {
        float dim[4];
        for (auto &d : dim) d=float(input->readLong(4))/65536;
        f << "dim=" << MWAWBox2f(MWAWVec2f(dim[1],dim[0]),MWAWVec2f(dim[3],dim[2])) << ",";
        int val=int(input->readULong(4)); // 0|8000
        if (val) f << "fl0=" << std::hex << val << std::dec << ",";
        auto fl=input->readULong(4); // [347]|[01]ff
        if (fl)  f << "fl1=" << std::hex << fl << std::dec << ",";
        long extraLength=0;
        for (unsigned decal=1; decal<0x400; decal*=2) {
          if (fl&decal)
            extraLength+=4;
        }
        auto decal=input->readULong(4);
        long nextPos=decal==0xFFFFFFFF ? entries[1].end() : entries[1].begin()+long(decal);
        if (nextPos<pos+44+extraLength || nextPos>entries[1].end()) {
          input->seek(pos, librevenge::RVNG_SEEK_SET);
          break;
        }
        long dataLength[2];
        dataLength[0]=input->readLong(4);
        unsigned dataType=unsigned(input->readULong(4));
        dataLength[1]=input->readLong(4);
        if (typeList && type!=11)
          typeList->push_back(type==14 ? int(dataType) : type);
        if (dataLength[0]) {
          MWAWEntry cData;
          cData.setId(int(dataType));
          cData.setBegin(dataLength[1]);
          cData.setLength(dataLength[0]);
          f << "data=[" << Canvas5Parser::getString(dataType) << "," << std::hex << dataLength[1] << "->" << dataLength[0]+dataLength[1] << std::dec << "],";
          if (cData.valid() && input->checkPosition(entries[0].begin()+cData.end())) {
            if (cData.end()>entries[0].length()) {
              f << "###";
              MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: the main data seems bad\n"));
            }
            else {
              long actPos=input->tell();

              libmwaw::DebugStream f2;
              f2 << "Vkfl-B[" << Canvas5Parser::getString(unsigned(type)) << ":" << Canvas5Parser::getString(unsigned(dataType)) << "]:";
              input->seek(entries[0].begin()+cData.begin(), librevenge::RVNG_SEEK_SET);

              switch (type) {
              case 1: // poly?
              case 2: { // unsure a list of 16/20 points, many are similar
                f2 << (type==1 ? "poly" : "type2") << ",";
                if (cData.length()<8) {
                  f2 << "###";
                  MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL[poly]: the zone seems too short\n"));
                  break;
                }
                int nPts=int(input->readULong(4));
                f2 << "N=" << nPts << ",";
                if (nPts<0 || 8+8*nPts<0 || nPts>(cData.length()-8)/8 || 8+8*nPts>cData.length()) {
                  f2 << "###";
                  MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL[poly]: can not read the number of points\n"));
                  break;
                }
                val=int(input->readULong(4));
                if (val!=nPts)
                  f2 << "##N1=" << val << ",";
                f2 << "pts=[";
                for (int i=0; i<nPts; ++i) {
                  float coord[2];
                  for (auto &d : coord) d=float(input->readLong(4))/65536;
                  f2 << MWAWVec2f(coord[1], coord[0]) << ",";
                }
                f2 << "],";
                break;
              }
              // 6, 7 no data 6 ? 7 circle ?
              case 9:
                f << "type9,";
                if (cData.length()<16) {
                  f2 << "###";
                  MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL[arc]: the zone seems too short\n"));
                  break;
                }
                f2 << "pts=[";
                for (int i=0; i<2; ++i) {
                  float coord[2];
                  for (auto &d : coord) d=float(input->readLong(4))/65536;
                  f2 << MWAWVec2f(coord[1], coord[0]) << ",";
                }
                f2 << "],";
                break;
              case 10: // pie?
                f << "pie,";
                if (cData.length()<8) {
                  f2 << "###";
                  MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL[arc]: the zone seems too short\n"));
                  break;
                }
                f2 << "angles=[";
                for (int i=0; i<2; ++i) f2 << float(input->readLong(4))/65536 << ",";
                f2 << "],";
                break;
              // 11: no data style?
              case 14: // special
                switch (dataType) {
                case 0x706f626a: // special a pobj which contains a bitmap
                  if (m_mainParser->readBitmap(stream, texture.m_texture, &texture.m_color)) {
                    texture.m_textureDim=MWAWVec2i(int(dim[3]-dim[1]),int(dim[2]-dim[0]));
                    texture.m_isBad=false;
                  }
                  else {
                    f2 << "###";
                    MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: can not retrieve the bitmap\n"));
                  }
                  if (input->tell()!=entries[0].begin()+cData.end()) {
                    ascFile.addPos(input->tell());
                    ascFile.addNote("Vkfl-end");
                  }
                  break;
                case 0x8F909d96: { // special a bitmap in a windows files
                  bool readInverted=input->readInverted();
                  input->setReadInverted(false);
                  if (m_mainParser->readBitmap(stream, texture.m_texture, &texture.m_color)) {
                    texture.m_textureDim=MWAWVec2i(int(dim[3]-dim[1]),int(dim[2]-dim[0]));
                    texture.m_isBad=false;
                  }
                  else {
                    f2 << "###";
                    MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: can not retrieve the bitmap\n"));
                  }
                  input->setReadInverted(readInverted);
                  if (input->tell()!=entries[0].begin()+cData.end()) {
                    ascFile.addPos(input->tell());
                    ascFile.addNote("Vkfl-end");
                  }
                  break;
                }
                default: {
                  std::string extra;
                  if (!m_mainParser->readSpecialData(cData.length(), dataType, extra))
                    f2 << "###";
                  f2 << extra;
                  break;
                }
                }
                break;
              default:
                break;
              }

              ascFile.addPos(entries[0].begin()+cData.begin());
              ascFile.addNote(f2.str().c_str());
              input->seek(actPos, librevenge::RVNG_SEEK_SET);
            }
          }
        }
        if (input->tell()!=pos+44+extraLength)
          ascFile.addDelimiter(input->tell(),'|');
        ascFile.addPos(pos);
        ascFile.addNote(f.str().c_str());
        long sLength=nextPos-pos;
        N=int(sLength-44-extraLength)/24;
        input->seek(pos+44+extraLength, librevenge::RVNG_SEEK_SET);
        first=false;
      }
      else
        N=1;
      for (int i=0; i<N; ++i) {
        pos=input->tell();
        f.str("");
        f << "Vkfl-s" << shape << "-Dt" << i << ":";
        // if type==11, dt0: line color, dt1: surface color, dt2: pen size, dt3: ?
        unsigned dataType=unsigned(input->readULong(4));
        auto unknPos=input->readULong(4);
        if (unknPos!=0xFFFFFFFF) f << "unkn=" << std::hex << unknPos << std::dec << ",";
        long dataLength[2];
        dataLength[0]=input->readLong(4);
        for (int j=0; j<2; ++j) {
          int val=int(input->readULong(4));
          if (val!=j)
            f << "f" << i << "=" << val << ",";
        }
        dataLength[1]=input->readLong(4);
        if (dataLength[0]) {
          MWAWEntry cData;
          cData.setId(int(dataType));
          cData.setBegin(dataLength[1]);
          cData.setLength(dataLength[0]);
          cEntries.push_back(cData);
          f << "data=[" << Canvas5Parser::getString(dataType) << "," << std::hex << dataLength[1] << "->" << dataLength[0]+dataLength[1] << std::dec << "],";
        }
        ascFile.addPos(pos);
        ascFile.addNote(f.str().c_str());
        input->seek(pos+24, librevenge::RVNG_SEEK_SET);
      }
    }
    if (input->tell()<entries[1].end()) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: find unknown data in zone1\n"));
      ascFile.addPos(input->tell());
      ascFile.addNote("Vkfl-B1:###");
    }
  }

  if (entries[0].valid()) {
    long const &begZone=entries[0].begin();
    long lenZone=entries[0].length();
    long endZone=entries[0].end();
    for (auto const &e : cEntries) {
      if (!e.valid()) continue;
      if (e.end()<=lenZone) {
        if (begZone+e.begin()<endZone)
          endZone=begZone+e.begin();
        f.str("");
        f << "Vkfl-C[" << Canvas5Parser::getString(unsigned(e.id())) << "]:";
        input->seek(begZone+e.begin(), librevenge::RVNG_SEEK_SET);
        switch (e.id()) {
        case 1: {
          std::string extra;
          MWAWColor sColor;
          if (e.length()!=24 || !readColor(stream, sColor, extra)) {
            // find also a zone with size=140, maybe a bitmap
            MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: can not read a color field\n"));
            f << "##";
          }
          f << extra;
          break;
        }
        // case 2: size 20: 1, small number, -1,  -1|60, -1|78 maybe stroke?
        case 3: {
          if (e.length()!=32) {
            MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: can not read a pen style field\n"));
            f << "##";
            break;
          }
          float widths[2];
          for (auto &w : widths) w=float(input->readLong(4))/65536.f;
          f << "pen[size]=" << MWAWVec2f(widths[0],widths[1]) << ",";
          ascFile.addDelimiter(input->tell(),'|');
          break;
        }
        case 4: {
          if (e.length()!=72) {
            MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: can not read a matrix field\n"));
            f << "##";
            break;
          }
          for (size_t st=0; st<2; ++st) {
            f << "mat" << st << "=[";
            for (int i=0; i<9; ++i)
              f << double(input->readLong(4))/65536 << ",";
            f << "],";
          }
          break;
        }
        default:
          break;
        }
        ascFile.addPos(begZone+e.begin());
        ascFile.addNote(f.str().c_str());
      }
      else {
        MWAW_DEBUG_MSG(("Canvas5StyleManager::readVKFL: entry C seems bad\n"));
      }
    }
  }
  input->seek(endPos, librevenge::RVNG_SEEK_SET);
  return true;
}

bool Canvas5StyleManager::readSymbol(Canvas5Structure::Stream &stream, Canvas5StyleManagerInternal::ColorStyle &color, long len)
{
  auto input=stream.input();
  long pos=input->tell();
  if (len<36 || !input->checkPosition(pos+len)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readSymbol: the zone seems too short\n"));
    return false;
  }
  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  f << "Entries(Symbol):";
  int val;
  for (int i=0; i<5; ++i) {
    val=int(input->readLong(4));
    if (!val)
      continue;
    char const *wh[]= { "deplX", "deplY", "stagger", "rotation", "scale" };
    f << wh[i] << "=" << float(val)/65536 << ",";
  }
  long sz=input->readLong(4);
  f << "sz=" << sz << ",";
  long endSize=long(input->readULong(4));
  if (endSize)
    f << "sz[end]=" << endSize << ",";
  long endPos=pos+36+sz;
  if (sz<0 || endSize<0 || 36+sz+endSize<0 || 36+sz+endSize>len) {
    f << "###";
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readSymbol: can not read the symbox sz\n"));
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  for (int i=0; i<2; ++i) { // g0=0|2e2e, g1=0|c3|e6e6
    val=int(input->readLong(2));
    if (val)
      f << "g" << i << "=" << std::hex << val << std::dec << ",";
  }
  val=int(input->readLong(1));
  if (val!=1) f << "type?=" << val << ",";
  input->seek(3, librevenge::RVNG_SEEK_CUR);
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  if (sz>0) {
    Canvas5StyleManagerInternal::ColorStyle texture;
    readVKFL(stream, texture, sz);
  }

  if (!endSize)
    return true;

  input->seek(endPos, librevenge::RVNG_SEEK_SET);
  f.str("");
  f << "Symbol-End:";
  if (endSize>4) {
    Canvas5StyleManagerInternal::ColorStyle endColor; // background ?
    endColor.m_type=unsigned(input->readULong(4));
    f << Canvas5Parser::getString(endColor.m_type);
    if (!readColorStyle(stream, endColor, endSize-4))
      f << "###";
    else {
      color.m_isBad=endColor.m_isBad;
      color.m_color=endColor.m_color;
      ascFile.addPos(endPos+4);
      ascFile.addNote(f.str().c_str());
    }
  }
  ascFile.addPos(endPos);
  ascFile.addNote(f.str().c_str());

  return true;
}

bool Canvas5StyleManager::readTexture(Canvas5Structure::Stream &stream, Canvas5StyleManagerInternal::ColorStyle &color, long len)
{
  auto input=stream.input();
  long pos=input->tell();
  if (len<36 || !input->checkPosition(pos+len)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readTexture: the zone seems too short\n"));
    return false;
  }
  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  f << "Entries(Txture):";
  int val;
  for (int i=0; i<10; ++i) { // f4=0|50, f8=50|100
    val=int(input->readLong(2));
    if (val)
      f << "f" << i << "=" << val << ",";
  }
  long sz=input->readLong(4);
  f << "sz=" << sz << ",";
  long endSize=long(input->readULong(4));
  if (endSize)
    f << "sz[end]=" << endSize << ",";
  long endPos=pos+36+sz;
  if (sz<0 || endSize<0 || 36+sz+endSize<0 || 36+sz+endSize>len) {
    f << "###";
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readTexture: can not read the texture sz\n"));
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  for (int i=0; i<2; ++i) { // g0=0|2e2e, g1=0|c3|e6e6
    val=int(input->readLong(2));
    if (val)
      f << "g" << i << "=" << std::hex << val << std::dec << ",";
  }
  val=int(input->readLong(1));
  if (val!=1) f << "type?=" << val << ",";
  input->seek(3, librevenge::RVNG_SEEK_CUR);
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  if (sz>0)
    readVKFL(stream, color, sz);

  if (!endSize)
    return true;


  input->seek(endPos, librevenge::RVNG_SEEK_SET);
  f.str("");
  f << "Texture-End:###";
  MWAW_DEBUG_MSG(("Canvas5StyleManager::readTexture: find extra datan\n"));
  ascFile.addPos(endPos);
  ascFile.addNote(f.str().c_str());

  return true;
}

bool Canvas5StyleManager::readDashes(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  long pos=input->tell();
  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  f << "Entries(Dash):";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  if (!m_mainParser->readUsed(stream, "Dash"))
    return false;

  return m_mainParser->readExtendedHeader(stream, 64, "Dash",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &, long) {
    auto lInput=lStream.input();
    long lPos=lInput->tell();
    libmwaw::DebugFile &asciiFile = lStream.ascii();
    libmwaw::DebugStream lF;
    lF << "Da" << id << ",";
    std::vector<float> dashes;
    int val;
    for (int i=0; i<2; ++i) {
      val=int(lInput->readLong(2));
      if (val)
        lF << "f" << i << "=" << val << ",";
    }
    int N=int(lInput->readLong(2));
    lF << "N=" << N << ",";
    if (N>14) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readDashes: N seems bad\n"));
      lF << "###";
      N=0;
    }
    val=int(lInput->readLong(2));
    if (val!=1) lF << "f2=" << val << ",";
    lF << "dash=[";
    for (int i=0; i<N; ++i) {
      dashes.push_back(float(lInput->readULong(4))/65536.f);
      lF << dashes.back() << ",";
    }
    lF << "],";
    m_state->m_idToDash[id]=dashes;
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  });
}

bool Canvas5StyleManager::readFonts(Canvas5Structure::Stream &stream, int numFonts)
{
  MWAWInputStreamPtr input = stream.input();
  long pos=input->tell();
  if (numFonts<=0 || !input->checkPosition(pos+136*numFonts)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readFonts: zone seems too short\n"));
    return false;
  }

  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  f << "Entries(Font):N=" << numFonts << ",";
  ascFile.addPos(pos-4);
  ascFile.addNote(f.str().c_str());

  auto fontConverter=m_parserState->m_fontConverter;
  std::string const family=m_mainParser->isWindowsFile() ? "CP1252" : "";
  for (int fo=0; fo<numFonts; ++fo) {
    pos=input->tell();
    f.str("");
    f << "Font-" << fo << ":";
    int id=int(input->readULong(2));
    f << "id=" << id << ",";
    for (int i=0; i<3; ++i) {
      int val=int(input->readLong(2));
      if (val!=(i==0 ? 4 : 0)) f << "f" << i << "=" << val << ",";
    }
    int dSz=int(input->readULong(1));
    if (dSz>=127) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readFonts: can not read a name\n"));
      f << "###name";
    }
    else {
      std::string name;
      for (int s=0; s<dSz; ++s) name+=char(input->readULong(1));
      if (!name.empty())
        fontConverter->setCorrespondance(fo+1, name, family);
      f << name << ",";
    }
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+136, librevenge::RVNG_SEEK_SET);
  }
  return true;
}

bool Canvas5StyleManager::readFormats(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  long pos=input->tell();
  if (!input || !input->checkPosition(pos+7*44+4)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readFormats: file is too short\n"));
    return false;
  }

  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  f << "Entries(Format):";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  for (int fo=0; fo<7; ++fo) {
    pos=input->tell();
    f.str("");
    f << "Format-" << fo << ":";
    int val=int(input->readLong(2));
    if (val) // small integer
      f << "f0=" << val << ",";
    val=int(input->readLong(2)); // 1|a|c|10
    f << "f1=" << val << ",";
    for (int i=0; i<2; ++i) { // f2=0|1, f3=0|3|10
      val=int(input->readLong(2));
      if (val)
        f << "f" << i+2 << "=" << val << ",";
    }
    for (int i=0; i<4; ++i) {
      val=int(input->readULong(4));
      if (val!=0x10000)
        f << "dim" << i << "=" << double(val)/double(0x10000) << ",";
    }
    int len=int(input->readULong(1));
    if (len<=19) {
      std::string text;
      for (int i=0; i<len; ++i) text+=char(input->readULong(1));
      f << "name=" << text << ",";
    }
    else {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readFormats: can not read the format name\n"));
      f << "###name,";
    }
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+44, librevenge::RVNG_SEEK_SET);
    if (fo!=0) continue;

    pos=input->tell();
    f.str("");
    f << "Format-unk:";
    for (int i=0; i<2; ++i) { // f0: small number
      val=int(input->readLong(2));
      if (val)
        f << "f" << i << "=" << val << ",";
    }
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
  }
  return true;
}

bool Canvas5StyleManager::readPenSize(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  long pos=input ? input->tell() : 0;
  if (!input || !input->checkPosition(pos+20)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readPenSize: file is too short\n"));
    return false;
  }

  libmwaw::DebugFile &ascFile = stream.ascii();
  libmwaw::DebugStream f;
  f << "Entries(PenSize):sz=[";
  for (int i=0; i<10; ++i)
    f << double(input->readULong(2))/256. << ",";
  f << "],";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  return true;
}

bool Canvas5StyleManager::readPenStyles(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  long pos=input->tell();

  libmwaw::DebugStream f;
  libmwaw::DebugFile &ascFile = stream.ascii();
  f << "Entries(PenStyl):";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  if (!m_mainParser->readExtendedHeader(stream, 4, "PenStyl",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &, long) {
  auto lInput=lStream.input();
    long lPos=lInput->tell();
    libmwaw::DebugFile &asciiFile = lStream.ascii();
    libmwaw::DebugStream lF;
    lF << "Pe" << id << ",";
    Canvas5StyleManagerInternal::PenStyle style;
    style.m_type=unsigned(lInput->readULong(4));
    m_state->m_idToPen[id]=style;
    lF << style;
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  }))
  return false;

  if (!m_mainParser->readIndexMap(stream, "PenStyl",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &, long len) {
  auto lInput=lStream.input();
    auto &asciiFile=lStream.ascii();
    long lPos=lInput->tell();
    libmwaw::DebugStream lF;
    lF << "Pe" << id << ",";
    Canvas5StyleManagerInternal::PenStyle style;
    auto it=m_state->m_idToPen.find(id);
    if (it!=m_state->m_idToPen.end()) style=it->second;
    switch (style.m_type) {
    case 1: {
      if (len<32) {
        MWAW_DEBUG_MSG(("Canvas5StyleManager::readPenSize: find unexpected size for type 1\n"));
        lF << "###";
        break;
      }
      float widths[2];
      for (auto &w : widths) w=float(lInput->readLong(4))/65536.f;
      style.m_size=MWAWVec2f(widths[0],widths[1]);
      asciiFile.addDelimiter(lInput->tell(),'|');
      break;
    }
    case 0x706c696e: { // plin
      if (len<12) {
        MWAW_DEBUG_MSG(("Canvas5StyleManager::readPenSize: find unexpected size for plin\n"));
        lF << "###";
        break;
      }
      int N=int(lInput->readULong(4));
      lF << "plin,N=" << N << ",";
      if (N<0 || (len-16)/128<N) {
        MWAW_DEBUG_MSG(("Canvas5StyleManager::readPenSize: find unexpected value of N for plin\n"));
        lF << "###";
        break;
      }
      style.m_usePenColor=false;
      for (int i=0; i<2; ++i) {
        int val=int(lInput->readLong(1));
        if (!val) continue;
        char const *wh[]= {"equidistant", "usePenLine" };
        if (val==1) {
          lF << wh[i] << ",";
          if (i==1)
            style.m_usePenColor=true;
        }
        else
          lF << wh[i] << "=" << val << ",";
      }
      asciiFile.addDelimiter(lInput->tell(),'|');
      lInput->seek(lPos+16, librevenge::RVNG_SEEK_SET);
      libmwaw::DebugStream f2;
      std::vector<float> offsets;
      float sumOffsets=0;
      for (int i=0; i<N; ++i) {
        long actPos=lInput->tell();
        f2.str("");
        f2 << "PenStyl,Pe" << id << "[plin" << i << "]";
        Canvas5StyleManagerInternal::PenStyle::Line line;
        std::string extra;
        if (!readColor(lStream, line.m_color, extra)) {
          f2 << "##";
          lInput->seek(actPos+24, librevenge::RVNG_SEEK_SET);
        }
        f2 << extra;
        float width[2];
        for (auto &w : width) w=float(lInput->readULong(4))/65536;
        line.m_size=MWAWVec2f(width[0],width[1]);
        f2 << "w=" << line.m_size << ",";
        for (int j=0; j<46; ++j) { // 0
          int val=int(lInput->readLong(2));
          if (val) f2 << "g" << j << "=" << val << ",";
        }
        offsets.push_back(float(lInput->readULong(4))/65536);
        sumOffsets+=offsets.back();
        f2 << "decal=" << offsets.back() << ",";
        style.m_lines.push_back(line);
        asciiFile.addPos(actPos);
        asciiFile.addNote(f2.str().c_str());
        lInput->seek(actPos+128, librevenge::RVNG_SEEK_SET);
      }
      float actualOffset=sumOffsets/2;
      for (size_t i=0; i<std::min(style.m_lines.size(),offsets.size()); ++i) {
        style.m_lines[i].m_offset=actualOffset;
        actualOffset-=offsets[i];
      }
      asciiFile.addDelimiter(lInput->tell(),'|');
      break;
    }
    case 0x766e656f: { // vneo
      if (len<68) {
        MWAW_DEBUG_MSG(("Canvas5StyleManager::readPenSize: the vneo zone seems too short\n"));
        lF << "###";
        break;
      }
      float w=float(lInput->readLong(4))/65536;
      lF << "w=" << w << ",";
      style.m_size=MWAWVec2f(w,w);
      for (int i=0; i<2; ++i) {
        std::string extra;
        if (!readColor(lStream, style.m_colors[i], extra)) {
          lF << "##";
          lInput->seek(lPos+4+24*(i+1), librevenge::RVNG_SEEK_SET);
        }
        lF << "col" << i << "=[" << extra << "],";
      }
      lF << "f0=" << lInput->readLong(2) << ",";
      int val=int(lInput->readULong(2));
      if (val&0x100)
        lF << "axial,";
      val &= 0xfeff;
      if (val!=0xdd)
        lF << "fl=" << std::hex << val << std::dec << ",";
      for (int i=0; i<2; ++i) {
        val=int(lInput->readULong(4));
        if (!val) continue;
        if (i==0)
          lF << "corner=" << val << ","; // 0: none, 1: round, 2: square
        else
          lF << "join=" << val << ","; // 0: mitter, 1: round, 2:bevel
      }
      lF << "angle=" << float(lInput->readLong(4))/65536 << "rad,";
      break;
    }
    default:
      if (!len)
        break;
      MWAW_DEBUG_MSG(("Canvas5StyleManager::readPenSize: find unexpected type\n"));
      lF << "###";
    }
    lF << style;
    m_state->m_idToPen[id]=style;
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  }))
  return false;
  return m_mainParser->readUsed(stream, "PenStyl");
}

bool Canvas5StyleManager::readStrokes(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  long pos=input->tell();

  libmwaw::DebugStream f;
  libmwaw::DebugFile &ascFile = stream.ascii();
  f << "Entries(Stroke):";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  if (!m_mainParser->readUsed(stream, "Stroke"))
    return false;
  if (!m_mainParser->readExtendedHeader(stream, 20, "Stroke",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &, long) {
  auto lInput=lStream.input();
    long lPos=lInput->tell();
    libmwaw::DebugFile &asciiFile = lStream.ascii();
    libmwaw::DebugStream lF;
    lF << "St" << id << ",";
    Canvas5StyleManagerInternal::Stroke style;
    style.m_type=unsigned(lInput->readULong(4));
    style.m_penId=int(lInput->readLong(4));
    style.m_dashId=int(lInput->readLong(4));
    for (int i=0; i<2; ++i)
      style.m_arrowId[i]=int(lInput->readLong(4));
    lF << style;
    m_state->m_idToStroke[id]=style;
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  }))
  return false;

  return true;
}

////////////////////////////////////////////////////////////
// styles
////////////////////////////////////////////////////////////
bool Canvas5StyleManager::readCharStyle(Canvas5Structure::Stream &stream, int id, std::pair<MWAWFont, int> &fontId)
{
  auto input=stream.input();
  libmwaw::DebugFile &ascFile = stream.ascii();
  long pos=input->tell();
  libmwaw::DebugStream f;
  if (id<0)
    f << "Entries(CharStyl):";
  else
    f << "CharStyle-" << id << ":";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  auto &font=fontId.first;
  fontId.second=0;
  if (!input->checkPosition(pos+60)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readCharStyle: the zone is too short\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  f.str("");
  int val=int(input->readLong(2));
  if (val!=1) f << "used=" << val << ",";
  f << "h=[";
  for (int i=0; i<2; ++i)
    f << input->readLong(2) << ",";
  f << "],";
  font.setId(int(input->readULong(2)));
  val=int(input->readULong(1));
  uint32_t flags = 0;
  if (val&0x1) flags |= MWAWFont::boldBit;
  if (val&0x2) flags |= MWAWFont::italicBit;
  if (val&0x4) font.setUnderlineStyle(MWAWFont::Line::Simple);
  if (val&0x8) flags |= MWAWFont::embossBit;
  if (val&0x10) flags |= MWAWFont::shadowBit;
  if (val&0x80) font.setStrikeOutStyle(MWAWFont::Line::Simple);
  if (val&0x60) f << "fl=" << std::hex << (val&0x60) << std::dec << ",";
  val=int(input->readULong(1));
  if (val) f << "fl1=" << std::hex << val << std::dec;
  font.setSize(float(input->readULong(2)));
  val=int(input->readLong(4));
  if (val) {
    auto it=m_state->m_idToColor.find(val);
    if (it!=m_state->m_idToColor.end())
      font.setColor(it->second.m_color);
    else
      f << "#Co1" << val << ",";
  }
  for (int i=0; i<2; ++i) { //
    val=int(input->readLong(2));
    if (!val) continue;
    if (i==0) {
      f << "E" << val << ",";
      fontId.second=val;
    }
    else
      f << "f" << i << "=" << val << ",";
  }
  val=int(input->readLong(4));
  if ((val>=-6*55536 && val<0) || (val>0 && val<6*65536))
    font.setDeltaLetterSpacing(float(val)/2/65536, librevenge::RVNG_POINT);
  else if (val) {
    MWAW_DEBUG_MSG(("Canvas5Style::readCharStyle: unknown delta spacing\n"));
    f << "##delta[spacing]=" << val/65536 << ",";
  }

  int lVals[4];
  for (auto &l : lVals) l=int(input->readLong(2));
  float stretchs[]= {1,1};
  for (int i=0; i<2; ++i) {
    if (lVals[i]==lVals[i+2])
      continue;
    f << "scaling[" << (i==0 ? "hori" : "verti") << "]=" << lVals[i] << "/" << lVals[i+2] << ",";
    if (lVals[i]<=0 || lVals[i+2]<=0) {
      MWAW_DEBUG_MSG(("Canvas5Style::readCharStyle: invalid scaling\n"));
      f << "###";
    }
    else
      stretchs[i]=float(lVals[i])/float(lVals[i+2]);
  }
  if (stretchs[1]>1-1e-4f && stretchs[1]<1+1e-4f) {
    if (stretchs[0]<1-1e-4f || stretchs[0]>1+1e-4f)
      font.setWidthStreching(stretchs[0]);
  }
  else {
    font.setSize(font.size()*stretchs[1]);
    font.setWidthStreching(stretchs[0]/stretchs[1]);
  }
  val=int(input->readLong(4));
  if (val)
    font.set(MWAWFont::Script(float(val)/65536,librevenge::RVNG_POINT));
  val=int(input->readLong(2));
  if (val) f << "h0=" << val << ",";
  val=int(input->readULong(2));
  if (val&1)
    flags |= MWAWFont::smallCapsBit;
  if (val&2)
    flags |= MWAWFont::uppercaseBit;
  if (val&4)
    flags |= MWAWFont::lowercaseBit;
  if (val&8)
    flags |= MWAWFont::initialcaseBit;
  if (val&0x200)
    f << "spread,";
  if (val&0x800)
    f << "overprint,";
  val &= 0xF5F0;
  if (val) {
    MWAW_DEBUG_MSG(("Canvas5Style::readCharStyle: unknown small caps bits\n"));
    f << "##smallCaps=" << std::hex << val << std::dec << ",";
  }

  for (int i=0; i<7; ++i) {
    val=int(input->readLong((i>=2 && i<=4) ? 4 : 2));
    if (!val) continue;
    if (i==4 && fontId.second==0) { // checkme: there is two color: the para color and the font color
      auto it=m_state->m_idToColor.find(val);
      if (it!=m_state->m_idToColor.end())
        font.setColor(it->second.m_color);
      else
        f << "#Co2" << val << ",";
    }
    else
      f << "h" << i+1 << "=" << val << ",";
  }
  font.setFlags(flags);
  std::string const extra=f.str();
  f.str("");
  f << font.getDebugString(m_parserState->m_fontConverter) << "," << extra;
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  return true;
}

bool Canvas5StyleManager::readCharStyles(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  libmwaw::DebugFile &ascFile = stream.ascii();
  ascFile.addPos(input->tell());
  ascFile.addNote("Entries(CharStyl):");
  if (!m_mainParser->readExtendedHeader(stream, 0x64, "CharStyl",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &what, long) {
  std::pair<MWAWFont,int> fontId;
  if (!readCharStyle(lStream, id, fontId))
      return;
    auto lInput=lStream.input();
    long pos=lInput->tell();
    libmwaw::DebugStream f;
    libmwaw::DebugFile &asciiFile = lStream.ascii();

    pos=lInput->tell();
    f.str("");
    f << what << "-" << id << "[A]:";
    for (int i=0; i<4; ++i) {
      int val=int(lInput->readLong(2));
      if (val)
        f << "f" << i << "=" << val << ",";
    }
    asciiFile.addPos(pos);
    asciiFile.addNote(f.str().c_str());

    pos=lInput->tell();
    f.str("");
    f << what << "-" << id << "[name]:";
    std::string name;
    for (int i=0; i<32; ++i) {
      char c=char(lInput->readLong(1));
      if (c==0) break;
      name+=c;
    }
    f << name << ",";
    lInput->seek(pos+32, librevenge::RVNG_SEEK_SET);
    asciiFile.addPos(pos);
    asciiFile.addNote(f.str().c_str());
  }))
  return false;
  if (!m_mainParser->readIndexMap(stream, "CharStyl"))
    return false;
  std::vector<bool> defined;
  if (!m_mainParser->readDefined(stream, defined, "CharStyl"))
    return false;
  if (!m_mainParser->readUsed(stream, "CharStyl"))
    return false;
  return m_mainParser->readExtendedHeader(stream, 8, "CharStyl[data2]", &Canvas5Parser::defDataFunction);
}

bool Canvas5StyleManager::readStyle(Canvas5Structure::Stream &stream, int id, Canvas5StyleManager::StyleList *styles)
{
  auto input=stream.input();
  libmwaw::DebugFile &ascFile = stream.ascii();

  long pos=input->tell();
  libmwaw::DebugStream f;
  if (id<0)
    f << "Entries(Style):";
  if (!input->checkPosition(pos+128)) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::readStyle: the zone is too short\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  std::pair<MWAWParagraph,int> *paraId=nullptr;
  if (styles && id>=0) {
    if (styles->m_paragraphs.size()<=size_t(id))
      styles->m_paragraphs.resize(size_t(id+1));
    paraId=&styles->m_paragraphs[size_t(id)];
    paraId->second=0;
  }
  MWAWParagraph *para=paraId ? &paraId->first : nullptr;
  int val=int(input->readLong(4));
  if (val!=1)
    f << "used=" << val << ",";
  val=int(input->readLong(4));
  if (val) {
    if (paraId) paraId->second=val;
    f << "Tab" << val << ",";
  }
  for (int i=0; i<2; ++i) { // 0
    val=int(input->readLong(2));
    if (!val) continue;
    if (i==0) {
      switch (val) {
      case -1:
        if (para) para->m_justify=MWAWParagraph::JustificationRight;
        f << "align=right,";
        break;
      // 0: left
      case 1:
        if (para) para->m_justify=MWAWParagraph::JustificationCenter;
        f << "align=center,";
        break;
      // 2: ?
      case 4:
        if (para) para->m_justify=MWAWParagraph::JustificationFull;
        f << "align=justify,";
        break;
      default:
        MWAW_DEBUG_MSG(("Canvas5StyleManager::readStyle: find unexpected align\n"));
        f << "##align=" << val << ",";
        break;
      }
    }
    else
      f << "f" << i << "=" << val << ",";
  }
  val=int(input->readULong(4));
  if (val) {
    f << "interline=" << float(val)/65536 << "pt,";
    if (para)
      para->setInterline(double(val)/65536, librevenge::RVNG_POINT);
  }
  val=int(input->readULong(4));
  if (val && val!=0x10000) {
    f << "interline=" << float(val)/65536.f << ",";
    if (para)
      para->setInterline(double(val)/65536, librevenge::RVNG_PERCENT);
  }
  for (int i=0; i<4; ++i) {
    val=int(input->readULong(2));
    if (val)
      f << "g" << i << "=" << val << ",";
  }
  if (para) para->m_marginsUnit=librevenge::RVNG_POINT;
  for (int i=0; i<3; ++i) {
    val=int(input->readLong(4));
    if (!val) continue;
    char const *wh[]= {"first", "left", "right"};
    f << wh[i] << "[marg]=" << float(val)/65536.f << ",";
    if (para)
      para->m_margins[i]=double(val)/65536;
  }
  if (para)
    para->m_margins[0]=*(para->m_margins[0])-*(para->m_margins[1]);
  for (int i=0; i<2; ++i) {
    val=int(input->readLong(4));
    if (!val) continue;
    f << "space[" << (i==0 ? "before" : "after") << "]=" << float(val)/65536 << ",";
    if (para)
      para->m_spacings[i+1]=float(val)/65536/72;
  }
  val=int(input->readULong(4));
  if (val) f << "g8=" << val << ",";
  val=int(input->readLong(4));
  if (val) f << "wrap[object]=" << float(val)/65536 << ","; // useme: unit in point
  for (int i=0; i<2; ++i) { // g9=0|2, g11=g12=0|-1, g13=0|1, g14=0|2
    val=int(input->readULong(2));
    if (!val) continue;
    f << "g" << i+9 << "=" << val << ",";
  }
  int dropChar=int(input->readULong(2));
  int dropLine=int(input->readULong(2));
  if (dropChar>0 && dropLine>1) {
    if (para) {
      para->m_dropNumCharacters=dropChar;
      para->m_dropNumLines=dropLine;
    }
    f << "drop=" << dropChar << "[l=" << dropLine << "],";
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  pos=input->tell();
  f.str("");
  if (id>=0)
    f << "Style-E" << id << "[A]:";
  else
    f << "Style[A]:";
  val=int(input->readLong(4));
  if (val)
    f << "justify[last,width]=" << 100-float(val)/65536 << "%,";
  for (int i=0; i<6; ++i) {
    val=int(input->readLong(4));
    if (!val) continue;
    char const *wh[]= {"spacing[word,min]", "spacing[word]", "spacing[word,max]", "spacing[min]", "spacing", "spacing[max]"};
    if (i==4 && styles) {
      for (auto &fId : styles->m_fonts) {
        if (fId.second==id)
          fId.first.setDeltaLetterSpacing(1+float(val)/65336, librevenge::RVNG_PERCENT);
      }
    }
    f << wh[i] << "=" << 100+100*float(val)/65336 << "%,";
  }
  f << "hyphen=[";
  for (int i=0; i<4; ++i) {
    val=int(input->readLong(2));
    int const expected[]= {3,2,6,3}; // after word, before word, smallest word, consecutive limite
    if (val!=expected[i])
      f << val << ",";
    else
      f << "_,";
  }
  f << "],";
  for (int i=0; i<4; ++i) { // g2=0|-1, g3=0|-1|0x140
    val=i==2 ? int(input->readULong(2)) : int(input->readLong(2));
    if (i==2) {
      int flags=0;
      if ((val&2)==0)
        f << "no[hyphen],";
      if ((val&4)==0)
        f << "skip[cap],";
      if ((val&0x200)==0)
        f << "orphan,";
      if ((val&0x400)==0) {
        flags|=MWAWParagraph::NoBreakBit;
        f << "keep[alllines],";
      }
      if ((val&0x800)==0) {
        flags|=MWAWParagraph::NoBreakWithNextBit;
        f << "keep[with,next],";
      }
      if (flags&&para)
        para->m_breakStatus=flags;
      val&=0xf1f9;
      if (val)
        f << "g2=" << std::hex << val << std::dec << ",";
      continue;
    }
    if (!val) continue;
    f << "g" << i << "=" << val << ",";
  }
  for (int i=0; i<10; ++i) { // 0
    val=int(input->readLong(2));
    if (!val)
      continue;
    if (i==2) { // checkme, something is bad here
      if (val!=100)
        f << "min[line,width]=" << 100-val << ",";
    }
    else if (i==3)
      f << "para[orphan]=" << val << ",";
    else if (i==4)
      f << "para[window]=" << val << ",";
    else
      f << "h" << i << "=" << val << ",";
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  return true;
}

bool Canvas5StyleManager::readStyles(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  libmwaw::DebugFile &ascFile = stream.ascii();
  ascFile.addPos(input->tell());
  ascFile.addNote("Entries(Style):");
  if (!m_mainParser->readUsed(stream, "Style"))
    return false;

  if (!m_mainParser->readExtendedHeader(stream, 0x114, "Style", &Canvas5Parser::stringDataFunction)) // string:256 + 5xlong?
    return false;

  if (!m_mainParser->readExtendedHeader(stream, 0xa8, "Style",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &what, long) {
  if (!readStyle(lStream, id))
      return;
    auto lInput=lStream.input();
    long pos=lInput->tell();
    libmwaw::DebugStream f;
    auto &asciiFile=lStream.ascii();
    f.str("");
    f << what << "-" << id << "[B]:";
    for (int i=0; i<4; ++i) { // f1=0|1, f3=0|1
      int val=int(lInput->readLong(2));
      if (val)
        f << "f" << i << "=" << val << ",";
    }
    asciiFile.addPos(pos);
    asciiFile.addNote(f.str().c_str());

    pos=lInput->tell();
    f.str("");
    f << what << "-" << id << "[name]:";
    std::string name;
    for (int i=0; i<32; ++i) {
      char c=char(lInput->readLong(1));
      if (c==0) break;
      name+=c;
    }
    f << name << ",";
    lInput->seek(pos+32, librevenge::RVNG_SEEK_SET);
    asciiFile.addPos(pos);
    asciiFile.addNote(f.str().c_str());
  }))
  return false;
  if (!m_mainParser->readIndexMap(stream, "Style"))
    return false;
  std::vector<bool> defined;
  return m_mainParser->readDefined(stream, defined, "Style");
}
////////////////////////////////////////////////////////////
//
// Windows resource
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
//
// send data
//
////////////////////////////////////////////////////////////
bool Canvas5StyleManager::updateLineColor(int cId, MWAWGraphicStyle &style)
{
  auto it=m_state->m_idToColor.find(cId);
  if (it==m_state->m_idToColor.end()) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineColor: can not find color %d\n", cId));
    return false;
  }
  auto const &color=it->second;
  switch (color.m_type) {
  case 0: // checkme
    style.m_lineOpacity=0;
    break;
  case 1:
    style.m_lineColor=color.m_color;
    if (color.m_color.getAlpha()!=255)
      style.m_lineOpacity=float(color.m_color.getAlpha())/255;
    break;
  case 0x54585552: // texture
    style.m_lineColor=color.m_color;
    break;
  case 0x68746368: // hatch
  case 0x766b666c: // vkfl
    if (!color.m_isBad)
      style.m_lineColor=color.m_color;
    break;
  case 0x4f62466c: // ObFl
    if (color.m_gradient.hasGradient())
      color.m_gradient.getAverageColor(style.m_lineColor);
    break;
  default:
    MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineColor: can not send type=%s\n", Canvas5Parser::getString(color.m_type).c_str()));
    break;
  }
  return true;
}

bool Canvas5StyleManager::updateSurfaceColor(int cId, MWAWGraphicStyle &style)
{
  auto it=m_state->m_idToColor.find(cId);
  if (it==m_state->m_idToColor.end()) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::updateSurfaceColor: can not find color %d\n", cId));
    return false;
  }
  auto const &color=it->second;
  switch (color.m_type) {
  case 0:
    style.m_surfaceOpacity=0;
    break;
  case 1:
    style.setSurfaceColor(color.m_color, float(color.m_color.getAlpha())/255);
    break;
  case 0x54585552:
    if (color.m_texture.isEmpty()) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::updateSurfaceColor: can not find the texture %d\n", cId));
      return false;
    }
    style.setPattern(MWAWGraphicStyle::Pattern(color.m_textureDim, color.m_texture, color.m_color));
    break;
  case 0x4f62466c:
    if (!color.m_gradient.hasGradient()) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::updateSurfaceColor: can not find the gradient %d\n", cId));
      return false;
    }
    style.m_gradient=color.m_gradient;
    break;
  case 0x68746368:
    if (!color.m_hatch.hasHatch()) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::updateSurfaceColor: can not find the hatch %d\n", cId));
      return false;
    }
    style.m_hatch=color.m_hatch;
    if (!color.m_isBad)
      style.setSurfaceColor(color.m_color, float(color.m_color.getAlpha())/255);
    break;
  case 0x766b666c: { // vkfl
    if (color.m_isBad) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::updateSurfaceColor: can not find the symbol color %d\n", cId));
      return false;
    }
    style.setSurfaceColor(color.m_color, float(color.m_color.getAlpha())/255);
    static bool first=true;
    if (first) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::updateSurfaceColor: symbol color are replaced by background color\n"));
      first=false;
    }
    break;
  }
  default:
    MWAW_DEBUG_MSG(("Canvas5StyleManager::updateSurfaceColor: can not send type=%s\n", Canvas5Parser::getString(color.m_type).c_str()));
    break;
  }
  return true;
}

bool Canvas5StyleManager::updateLineStyle(int sId, MWAWGraphicStyle &style, int &numLines, int lineId, float *offset)
{
  numLines=1;
  if (offset) *offset=0;
  auto it=m_state->m_idToStroke.find(sId);
  if (it==m_state->m_idToStroke.end()) {
    MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineStyle: can not find stroke %d\n", sId));
    return false;
  }
  auto const &stroke=it->second;
  style.m_lineWidth=0;
  if (stroke.m_penId) {
    auto pIt=m_state->m_idToPen.find(stroke.m_penId);
    if (pIt==m_state->m_idToPen.end()) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineStyle: can not find pen %d\n", stroke.m_penId));
    }
    else {
      auto const &pen=pIt->second;
      switch (pen.m_type) {
      case 1:
        style.m_lineWidth=(pen.m_size[0]+pen.m_size[1])/2;
        break;
      case 0x766e656f: { // vneo
        style.m_lineWidth=(pen.m_size[0]+pen.m_size[1])/2;
        // fixme: normally a gradient, let's replace it by it barycenters color...
        style.m_lineColor=MWAWColor::barycenter(0.5, pen.m_colors[0], 0.5, pen.m_colors[1]);
        static bool first=true;
        if (first) {
          MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineStyle: replace line gradient with their average color\n"));
          first=false;
        }
        break;
      }
      case 0x706c696e: // plin
        numLines=int(pen.m_lines.size());
        if ((lineId>=0 && lineId<numLines) || (numLines==1 && lineId<0)) {
          auto const &line = pen.m_lines[size_t(lineId)];
          style.m_lineWidth=(line.m_size[0]+line.m_size[1])/2;
          style.m_lineColor=line.m_color;
          if (offset) *offset=line.m_offset;
        }
        else if (lineId>=0) {
          MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineStyle: can not find the line with id=%d\n", lineId));
        }
        break;
      default:
        MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineStyle: can not send pen with type %s\n", Canvas5Parser::getString(pen.m_type).c_str()));
      }
    }
  }
  if (stroke.m_dashId) {
    auto dIt=m_state->m_idToDash.find(stroke.m_dashId);
    if (dIt==m_state->m_idToDash.end()) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineStyle: can not find dash %d\n", stroke.m_dashId));
    }
    else
      style.m_lineDashWidth=dIt->second;
  }
  for (int i=0; i<2; ++i) {
    if (!stroke.m_arrowId[i])
      continue;
    auto dIt=m_state->m_idToArrow.find(stroke.m_arrowId[i]);
    if (dIt==m_state->m_idToArrow.end()) {
      MWAW_DEBUG_MSG(("Canvas5StyleManager::updateLineStyle: can not find arrow %d\n", stroke.m_arrowId[i]));
    }
    else
      style.m_arrows[i]=dIt->second;
  }
  return true;
}

// vim: set filetype=cpp tabstop=2 shiftwidth=2 cindent autoindent smartindent noexpandtab:
