/* -*- Mode: C++; c-default-style: "k&r"; indent-tabs-mode: nil; tab-width: 2; c-basic-offset: 2 -*- */

/* libmwaw
* Version: MPL 2.0 / LGPLv2+
*
* The contents of this file are subject to the Mozilla Public License Version
* 2.0 (the "License"); you may not use this file except in compliance with
* the License or as specified alternatively below. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
*
* Software distributed under the License is distributed on an "AS IS" basis,
* WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
* for the specific language governing rights and limitations under the
* License.
*
* Major Contributor(s):
* Copyright (C) 2002 William Lachance (wrlach@gmail.com)
* Copyright (C) 2002,2004 Marc Maurer (uwog@uwog.net)
* Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
* Copyright (C) 2006, 2007 Andrew Ziem
* Copyright (C) 2011, 2012 Alonso Laurent (alonso@loria.fr)
*
*
* All Rights Reserved.
*
* For minor contributions see the git repository.
*
* Alternatively, the contents of this file may be used under the terms of
* the GNU Lesser General Public License Version 2 or later (the "LGPLv2+"),
* in which case the provisions of the LGPLv2+ are applicable
* instead of those above.
*/

#include <cmath>
#include <cctype>
#include <iomanip>
#include <iostream>
#include <limits>
#include <algorithm>
#include <set>
#include <sstream>
#include <stack>
#include <string>
#include <utility>

#include <librevenge/librevenge.h>

#include "MWAWFontConverter.hxx"
#include "MWAWGraphicListener.hxx"
#include "MWAWGraphicShape.hxx"
#include "MWAWGraphicStyle.hxx"
#include "MWAWHeader.hxx"
#include "MWAWParagraph.hxx"
#include "MWAWPictBitmap.hxx"
#include "MWAWPictData.hxx"
#include "MWAWPrinter.hxx"
#include "MWAWPosition.hxx"
#include "MWAWRSRCParser.hxx"
#include "MWAWStringStream.hxx"

#include "Canvas5Graph.hxx"
#include "Canvas5Structure.hxx"
#include "Canvas5StyleManager.hxx"

#include "Canvas5Parser.hxx"

/** Internal: the structures of a Canvas5Parser */
namespace Canvas5ParserInternal
{
//! Internal: the slide data of a Canvas5Parser
struct Slide {
  Slide()
    : m_dim(0,0)
    , m_numLayers(0)
    , m_layers()
  {
  }

  //! the slide dimension
  MWAWVec2f m_dim;
  //! the number of layer
  int m_numLayers;
  //! the layer id
  std::vector<int> m_layers;
};

//! Internal: the layer of a Canvas5Parser
struct Layer {
  //! constructor
  Layer()
    : m_name()
    , m_numShapes()
    , m_shapesId()
    , m_type(-1)
  {
  }
  //! the layer name
  librevenge::RVNGString m_name;
  //! the number of shape
  int m_numShapes;
  //! the shape id
  std::vector<int> m_shapesId;
  //! the layer type (unknonw)
  int m_type;
};

////////////////////////////////////////
//! Internal: the state of a Canvas5Parser
struct State {
  //! constructor
  State()
    : m_isWindowsFile(false)
    , m_stream()

    , m_type(1)
    , m_fileFlags(0)
    , m_documentSetup(0)
    , m_facingPages(false)

    , m_numSlides(1)
    , m_slideIds()
    , m_idToSlide()

    , m_numLayers(1)
    , m_idToLayer()
    , m_numShapes(0)
  {
  }

  //! true if this is a windows file
  bool m_isWindowsFile;
  //! the current stream
  std::shared_ptr<Canvas5Structure::Stream> m_stream;

  //! the document type 1: graphic, 2: list of pages, 3: slides
  int m_type;
  //! the file flags
  int m_fileFlags;
  //! the document setup: 0 full page, 1: two page bottom/down, 2: four page
  int m_documentSetup;
  //! true if the document uses facing page
  bool m_facingPages;

  //! the number of slides
  int m_numSlides;
  //! the slides id
  std::vector<int> m_slideIds;
  //! the slide data
  std::map<int, Slide> m_idToSlide;

  //! the number of layer
  int m_numLayers;
  //! the layer data
  std::map<int, Layer> m_idToLayer;
  //! the number of shapes
  int m_numShapes;
};

}

////////////////////////////////////////////////////////////
// constructor/destructor, ...
////////////////////////////////////////////////////////////
Canvas5Parser::Canvas5Parser(MWAWInputStreamPtr const &input, MWAWRSRCParserPtr const &rsrcParser, MWAWHeader *header)
  : MWAWGraphicParser(input, rsrcParser, header)
  , m_state()
  , m_graphParser()
  , m_styleManager()
{
  resetGraphicListener();
  setAsciiName("main-1");

  m_state.reset(new Canvas5ParserInternal::State);

  m_styleManager.reset(new Canvas5StyleManager(*this));
  m_graphParser.reset(new Canvas5Graph(*this));

  getPageSpan().setMargins(0.1);
}

Canvas5Parser::~Canvas5Parser()
{
}

bool Canvas5Parser::isWindowsFile() const
{
  return m_state->m_isWindowsFile;
}

bool Canvas5Parser::readSpecialData(long len, unsigned type, std::string &extra)
{
  return m_graphParser->readSpecialData(len, type, extra);
}

////////////////////////////////////////////////////////////
// the parser
////////////////////////////////////////////////////////////
void Canvas5Parser::parse(librevenge::RVNGDrawingInterface *docInterface)
{
  if (!getInput().get() || !checkHeader(nullptr))  throw(libmwaw::ParseException());
  bool ok = false;
  try {
    checkHeader(nullptr);

    auto input=decode(getInput());
    if (!input)
      throw(libmwaw::ParseException());

    // create the main stream
    m_state->m_stream=std::make_shared<Canvas5Structure::Stream>(input);
    m_state->m_stream->ascii().open(asciiName());

    ok = createZones();
    if (ok)
      createDocument(docInterface);
  }
  catch (...) {
    MWAW_DEBUG_MSG(("Canvas5Parser::parse: exception catched when parsing\n"));
    ok = false;
  }

  ascii().reset();
  resetGraphicListener();
  if (!ok) throw(libmwaw::ParseException());
}

////////////////////////////////////////////////////////////
// create the document
////////////////////////////////////////////////////////////
void Canvas5Parser::createDocument(librevenge::RVNGDrawingInterface *documentInterface)
{
  if (!documentInterface) return;
  if (getGraphicListener()) {
    MWAW_DEBUG_MSG(("Canvas5Parser::createDocument: listener already exist\n"));
    return;
  }

  std::vector<MWAWPageSpan> pageList;
  MWAWPageSpan page(getPageSpan());

  bool createMasterPage=false;
  size_t numMasters=m_state->m_type==1 ? 0 : 1;
  size_t numPagesOnOnePage=1, decal=0;
  if (m_state->m_type==2) {
    switch (m_state->m_documentSetup) {
    case 0:
      if (m_state->m_facingPages) {
        /** FIXME: it is simpler to create a big page which contains the left and right page,
            but it may be better to create each page and to only keep in each page the
            used shapes, ie. to translate back the right shape and also decompose the master
            page shapes in left/right
         */
        numMasters=2;
        numPagesOnOnePage=2;
        decal=1; // first page is the left page
        page.setFormWidth(2*page.getFormWidth());
      }
      break;
    case 1:
      page.setFormLength(page.getFormLength()/2);
      break;
    case 2:
      page.setFormWidth(page.getFormWidth()/2);
      page.setFormLength(page.getFormLength()/2);
      break;
    default:
      break;
    }
  }
  std::vector<std::vector<int> > listSlides;
  for (size_t p=0, n=decal; p<m_state->m_slideIds.size(); ++p) {
    int sId=m_state->m_slideIds[p];
    auto const &it=m_state->m_idToSlide.find(sId);
    if (it==m_state->m_idToSlide.end()) {
      MWAW_DEBUG_MSG(("Canvas5Parser::createDocument: can not find the slide %d\n", sId));
      continue;
    }
    auto const &slide=it->second;
    /*
       if type==1(illustration), one slide, multiple layer
       if type==2(publication), the first slide is the master page
       if type==3(slide), the first slide is the master page
     */
    MWAWPageSpan ps(page);
    if (p>=numMasters && createMasterPage && !slide.m_layers.empty() && slide.m_layers[0]==1)
      ps.setMasterPageName(librevenge::RVNGString("Master"));
    ps.setPageSpan(1);
    if (p<numMasters) {
      if (p==0)
        createMasterPage=slide.m_layers.size()==1 && slide.m_layers[0]==1;
      continue;
    }

    size_t nPage=n++/numPagesOnOnePage;
    if (listSlides.size() >= nPage) {
      listSlides.resize(nPage+1);
      pageList.push_back(ps);
    }
    listSlides[nPage].push_back(sId);
  }

  MWAWGraphicListenerPtr listen(new MWAWGraphicListener(*getParserState(), pageList, documentInterface));
  setGraphicListener(listen);

  listen->startDocument();

  if (createMasterPage) {
    MWAWPageSpan ps(page);
    ps.setMasterPageName(librevenge::RVNGString("Master"));
    if (!listen->openMasterPage(ps)) {
      MWAW_DEBUG_MSG(("Canvas5Parser::createDocument: can not create the master page\n"));
    }
    else {
      auto lIt=m_state->m_idToLayer.find(1);
      if (lIt!=m_state->m_idToLayer.end())
        send(lIt->second);
      listen->closeMasterPage();
    }
  }

  bool first=true;
  for (auto const &lId : listSlides) {
    if (!first)
      listen->insertBreak(MWAWListener::PageBreak);
    first=false;
    for (auto id : lId) {
      auto const &it=m_state->m_idToSlide.find(id);
      if (it==m_state->m_idToSlide.end()) {
        MWAW_DEBUG_MSG(("Canvas5Parser::parse: can not find slide %d\n", id));
        continue;
      }
      send(it->second);
    }
  }
}

////////////////////////////////////////////////////////////
//
// Intermediate level
//
////////////////////////////////////////////////////////////
bool Canvas5Parser::createZones()
{
  MWAWRSRCParserPtr rsrcParser = getRSRCParser();
  if (rsrcParser) {
    Canvas5Structure::Stream stream(rsrcParser->getInput(), rsrcParser->ascii());
    auto &entryMap = rsrcParser->getEntriesMap();

    for (int w=0; w<2; ++w) {
      // also some icons: ICN#, icl8, ics#, ics8
      static char const *wh[]= {"pnot" /*0*/, "PICT" /* value in pnot */};
      auto it = entryMap.lower_bound(wh[w]);
      while (it != entryMap.end() && it->first==wh[w]) {
        auto const &entry=it++->second;
        if (!entry.valid()) continue;
        switch (w) {
        case 0:
          readPnot(stream, entry);
          break;
        case 1:
        default:
          readPicture(stream, entry);
          break;
        }
      }
    }
  }

  auto stream=m_state->m_stream;
  if (!stream || !stream->input() || !readFileHeader(*stream) || !readMainBlock(stream) || !m_graphParser->readBitmaps(*stream) || !readFileRSRCs(*stream))
    return false;

  auto input=stream->input();
  if (!input->isEnd())
    readFileDesc(*stream);

  if (input->isEnd())
    return !m_state->m_idToSlide.empty();

  MWAW_DEBUG_MSG(("Canvas5Parser::createZones: find extra data\n"));
  int n=0;
  long pos=input->tell();
  libmwaw::DebugStream f;
  auto &ascFile=stream->ascii();
  ascFile.addPos(pos);
  ascFile.addNote("Entries(Extra):###");

  while (!input->isEnd()) {
    pos=input->tell();
    f.str("");
    f << "Extra-" << ++n << ":";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+256, librevenge::RVNG_SEEK_SET);
  }
  return !m_state->m_idToSlide.empty();
}

bool Canvas5Parser::readMainBlock(std::shared_ptr<Canvas5Structure::Stream> stream)
{
  if (!stream || !stream->input()) return false;

  if (!readPreview(*stream) || !readDocumentSettings(*stream))
    return false;
  if (!m_graphParser->findShapeDataZones(stream))
    return false;
  if (!m_graphParser->readShapes(*stream, m_state->m_numShapes))
    return false;

  if (!readSlides(*stream) || !readLayers(*stream))
    return false;

  if (!m_styleManager->readColors(*stream))
    return false;

  if (!m_graphParser->readMatrices(*stream))
    return false;

  //
  // the styles
  //
  if (!m_styleManager->readStrokes(*stream) || !m_styleManager->readPenStyles(*stream) ||
      !m_styleManager->readArrows(*stream) || !m_styleManager->readDashes(*stream))
    return false;

  if (!m_styleManager->readStyles(*stream) || !m_styleManager->readCharStyles(*stream))
    return false;

  return readTextLinks(*stream);
}

bool Canvas5Parser::readFileRSRCs(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  if (!input) return false;

  long pos=input->tell();
  if (!input->checkPosition(pos+4)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs: the zone is too short\n"));
    return false;
  }

  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "Entries(RsrcList):";
  int N=int(input->readLong(4));
  f << "N=" << N << ",";
  if (N<0 || (input->size()-pos-4)/16<N || pos+4+N*16<pos+4 || !input->checkPosition(pos+4+N*16)) {
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  std::map<std::string, int> const nameToId= {
    {"DoIn",0}, {"Edit", 1}, {"MACO", 2}, {"Mgor", 3}, {"TeXT", 4},
    {"effe",5}, {"fndt", 6}, {"pceg", 7}, {"pobj", 8}, {"prnt", 9},
    {"txrl",10}, {"vinf",11}
  };
  int numRsrc=N;
  for (int i=0; i<numRsrc; ++i) {
    pos=input->tell();
    f.str("");
    if (!input->checkPosition(pos+16)) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs: can not find block %d\n", i));
      f << "RsrcList-" << i << ":###";
      ascFile.addPos(pos);
      ascFile.addNote(f.str().c_str());
      return false;
    }
    std::string what;
    for (int c=0; c<4; ++c) what+=char(input->readULong(1));
    if (what=="TEXT") what="TeXT";
    f << "Entries(Rsrc" << what << "),";
    int id=int(input->readLong(4));
    f << "id=" << id << ",";
    f << "fl=" << std::hex << input->readULong(4) << std::dec << ","; // 2XXXXXX ?
    long len=input->readLong(4);
    long endPos=pos+16+len;
    if (endPos<pos+16 || !input->checkPosition(endPos+4)) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs: can not find block %d\n", i));
      f << "###";
      ascFile.addPos(pos);
      ascFile.addNote(f.str().c_str());
      return false;
    }
    input->pushLimit(endPos);
    auto const &it=nameToId.find(what);
    int wh=it!=nameToId.end() ? it->second : -1, val;
    std::vector<bool> defined;
    switch (wh) {
    case 0: // DoIn
      if (len!=32) break;
      for (int j=0; j<4; ++j) { // f0=0|100|146-148
        val=int(input->readLong(4));
        if (val) f << "f" << j << "=" << val << ",";
      }
      f << "N=["; // 1-a, 1-3, 1-3, 1-2
      for (int j=0; j<3; ++j) f << input->readLong(4) << ",";
      f << "],";
      val=int(input->readLong(4));
      if (val==2)
        f << "docUnit=points,";
      else if (val!=1)
        f << "#docUnit=" << val << ",";
      break;
    case 1: // Edit
      if (len!=8) break;
      for (int j=0; j<2; ++j) {
        val=int(input->readLong(4));
        if (val!=(j==0 ? 0 : -1)) f << "f" << j << "=" << val << ",";
      }
      break;
    case 2: // MACO
      if (len==0)
        break;
      if (len<4) {
        MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs: can not read first MACO value\n"));
        f << "###";
        break;
      }
      val=int(input->readULong(4));
      if (val!=0x77cc) f << "f0=" << std::hex << val << std::dec << ",";
      if (input->isEnd())
        break;
      if (!readExtendedHeader(stream, 0x58, "RsrcMACO", &Canvas5Parser::defDataFunction)) {
        f << "###";
        break;
      }
      if (input->isEnd())
        break;
      if (!readIndexMap(stream, "RsrcMACO")) {
        f << "###";
        break;
      }
      if (input->isEnd())
        break;
      if (!readUsed(stream, "RsrcMACO")) {
        f << "###";
        break;
      }
      if (input->isEnd())
        break;
      ascFile.addPos(input->tell()); // 14,0
      ascFile.addNote("RsrcMACO-end:");
      input->seek(8, librevenge::RVNG_SEEK_CUR);
      break;
    case 3: { // Mgor
      if (len!=48) break;
      ascFile.addDelimiter(input->tell(),'|');
      input->seek(20, librevenge::RVNG_SEEK_CUR);
      ascFile.addDelimiter(input->tell(),'|');
      int dim[2];
      for (auto &d : dim) d=int(input->readLong(4));
      f << "windows[dim]=" << MWAWVec2i(dim[1], dim[0]) << ","; // ~700x1000
      val=int(input->readLong(4));
      if (val) f << "f0=" << val << ",";
      break;
    }
    case 4: // TEXT
      switch (id) {
      case 1: {
        f << "char[style],";
        std::pair<MWAWFont,int> fontId;
        if (!m_styleManager->readCharStyle(stream, -1, fontId))
          f << "###";
        break;
      }
      case 2:
        f << "style,";
        if (!m_styleManager->readStyle(stream, -1))
          f << "###";
        break;
      case 3:
        if (len!=2)
          break;
        val=int(input->readLong(2));
        if (val) f << "f0=" << val << ",";
        break;
      default:
        MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs[TEXT]: unknown id=%d\n", id));
        f << "###";
        break;
      }
      break;
    case 5: // effe
      if (len==0) break;
      if (!readExtendedHeader(stream, 0xc, "Rsrceffe",
      [](Canvas5Structure::Stream &lStream, int, std::string const &, long) {
      auto lInput=lStream.input();
        long lPos=lInput->tell();
        int lVal=int(lInput->readULong(4)); // Enve
        libmwaw::DebugStream lF;
        auto &asciiFile=lStream.ascii();
        if (lVal!=1)
          lF << "f0=" << getString(unsigned(lVal)) << ",";
        for (int j=0; j<2; ++j) { // f1=580
          lVal=int(lInput->readLong(4));
          if (lVal)
            lF << "f" << j+1 << "=" << lVal << ",";
        }
        asciiFile.addPos(lPos);
        asciiFile.addNote(lF.str().c_str());
      })) {
        f << "###";
        break;
      }
      if (input->isEnd())
        break;
      if (!readIndexMap(stream, "Rsrceffe")) {
        f << "###";
        break;
      }
      if (input->isEnd())
        break;
      if (!readDefined(stream, defined, "Rsrceffe")) {
        f << "###";
        break;
      }
      break;
    case 6: //fndt
      if (id==2) { // unsure what to parse
        if (len<514) {
          MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs: can not read id=2 fndt block\n"));
          f << "###";
          break;
        }
        break;
      }
      if (id==3) {
        if (len<132) {
          MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs: can not read id=3 fndt block\n"));
          f << "###";
          break;
        }
        f << "N=[";
        for (int j=0; j<3; ++j) f << input->readLong(4) << ",";
        f << "],";
        ascFile.addPos(input->tell());
        ascFile.addNote("Rsrcfndt3-A:");
        input->seek(120, librevenge::RVNG_SEEK_CUR);

        while (!input->isEnd() && input->checkPosition(input->tell()+60)) {
          ascFile.addPos(input->tell());
          ascFile.addNote("Rsrcfndt3-B:");
          input->seek(60, librevenge::RVNG_SEEK_CUR);
        }
        break;
      }
      break;
    case 7: // pceg
      if (len!=56)
        break;
      for (int j=0; j<8; ++j) {
        val=int(input->readLong(2));
        int const expected[]= {0,0,0,0x38,0,0,0x13c,0};
        if (val!=expected[j])
          f << "f" << j << "=" << val << ",";
      }
      for (int j=0; j<20; ++j) { // 0
        val=int(input->readLong(2));
        if (val)
          f << "g" << j << "=" << val << ",";
      }
      break;
    case 8: // pobj
    case 10: // txrl: very rare
      if (len%4) break;
      for (int j=0; j<len/4; ++j) {
        val=int(input->readLong(4));
        if (val!=(j==0 ? 1 : 0))
          f << "f" << j << "=" << val << ",";
      }
      break;
    case 9: // prnt
      readPrinterRsrc(stream);
      break;
    case 11: { // vinf
      if (len<4) break;
      N=int(input->readULong(4));
      f << "N=" << N << ",";
      if (52*(N+1)+4<4 || N<0 || (len-4-52)/52<N || len<52*(N+1)+4) {
        MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs[vinf]: can not find the number of view\n"));
        f << "###";
        break;
      }
      ascFile.addPos(input->tell());
      ascFile.addNote("_");
      input->seek(52, librevenge::RVNG_SEEK_CUR);
      libmwaw::DebugStream f2;
      for (int v=1; v<=N; ++v) {
        long actPos=input->tell();
        f2.str("");
        f2 << "Rsrcvinf-v:";
        std::string text;
        for (int c=0; c<36; ++c) {
          char ch=char(input->readULong(1));
          if (!ch) break;
          text+=ch;
        }
        f2 << text << ",";
        input->seek(actPos+36, librevenge::RVNG_SEEK_SET);
        f2 << "val=["; // scale then translation
        for (int d=0; d<4; ++d) f2 << float(input->readLong(4))/65536.f << ",";
        f2 << "],";
        ascFile.addPos(actPos);
        ascFile.addNote(f2.str().c_str());
        input->seek(actPos+52, librevenge::RVNG_SEEK_SET);
      }
      break;
    }
    default:
      break;
    }
    input->popLimit();
    if (input->tell()!=endPos)
      ascFile.addDelimiter(input->tell(),'|');
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(endPos, librevenge::RVNG_SEEK_SET);
  }

  pos=input->tell();
  long len=input->readLong(4);
  if (pos+4+len<pos+4 || !input->checkPosition(pos+4+len)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readFileRSRCs: can not find font block\n"));
    ascFile.addPos(pos);
    ascFile.addNote("Entries(Font):###");
    return false;
  }
  if (!m_styleManager->readFonts(stream, int(len/136)))
    return false;
  input->seek(pos+4+len, librevenge::RVNG_SEEK_SET);

  pos=input->tell();
  ascFile.addPos(pos);
  ascFile.addNote("Entries(RsrcStrings):");

  if (!readUsed(stream, "RsrcStrings"))
    return false;

  if (!readIndexMap(stream, "RsrcStrings", &Canvas5Parser::stringDataFunction))
    return false;

  return true;
}

bool Canvas5Parser::readFileDesc(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  if (!input) return false;

  long pos=input->tell();
  if (!input->checkPosition(pos+0x30c)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readFileDesc: the zone is too short\n"));
    return false;
  }

  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "Entries(FileDesc):";
  int byteOrdering=int(input->readULong(1));
  switch (byteOrdering) {
  case 1:
    input->setReadInverted(true);
    break;
  case 2:
    input->setReadInverted(false);
    break;
  default:
    MWAW_DEBUG_MSG(("Canvas5Parser::readFileDesc: unknown byte ordering\n"));
    return false;
  }
  input->seek(3, librevenge::RVNG_SEEK_CUR);
  unsigned what=unsigned(input->readULong(4));
  if (what!=0x434e5635)
    return false;
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  for (int z=0; z<3; ++z) { // z=0: always empty, z=1: full path, z=2: filename
    pos=input->tell();
    f.str("");
    f << "FileDesc" << z << ":";
    std::string text;
    for (int c=0; c<256; ++c) {
      char ch=char(input->readULong(1));
      if (!ch) break;
      text+=ch;
    }
    f << text << ",";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+256, librevenge::RVNG_SEEK_SET);
  }

  pos=input->tell();
  f.str("");
  f << "FileDesc-end:";
  what=unsigned(input->readULong(4));
  if (what!=0x434e5635)
    f << what;
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  return true;
}

bool Canvas5Parser::readFileHeader(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  if (!input) return false;
  if (!input->checkPosition(0x2a)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readFileHeader: the zone is too short\n"));
    return false;
  }
  input->seek(5, librevenge::RVNG_SEEK_SET);
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "FileHeader:";
  int val=int(input->readULong(1));
  switch (val) {
  case 0x17:
    f << "v5(win),";
    break;
  case 0x18:
    f << "v5(mac),";
    break;
  default:
    f << "vers=" << val << ",";
    break;
  }
  val=int(input->readULong(1));
  f << "byte[order]=" << val << ",";
  switch (val) {
  case 1:
    input->setReadInverted(true);
    break;
  case 2:
    input->setReadInverted(false);
    break;
  default:
    MWAW_DEBUG_MSG(("Canvas5Parser::readFileHeader: unknown byte order\n"));
    break;
  }
  val=int(input->readULong(2)); // 0 or 2
  if (val) f << "f0=" << val << ",";
  val=int(input->readULong(4));
  if (val!=0xea8da) f << "f1=" << std::hex << val << std::dec << ",";

  std::string name;
  for (int i=0; i<7; ++i)
    name+=char(input->readULong(1));
  if (name!="CANVAS5") {
    MWAW_DEBUG_MSG(("Canvas5Parser::readFileHeader: not a Canvas 5 file\n"));
    f << "name=" << name << ",";
#ifndef DEBUG
    return false;
#endif
  }
  input->seek(1, librevenge::RVNG_SEEK_CUR);
  ascFile.addPos(0);
  ascFile.addNote(f.str().c_str());

  long pos=input->tell();
  f.str("");
  f << "FileHeader-A:";
  m_state->m_type=int(input->readULong(1));
  switch (m_state->m_type) {
  case 1:
    f << "illustration,";
    break;
  case 2: // list of pages with header/footer
    f << "publi,";
    break;
  case 3:
    f << "slide,";
    break;
  default:
    MWAW_DEBUG_MSG(("Canvas5Parser::readFileHeader: unknown document type %d\n", m_state->m_type));
    f << "##type=" << m_state->m_type << ",";
    m_state->m_type=1;
#ifndef DEBUG
    return false;
#endif
    break;
  }
  input->seek(1, librevenge::RVNG_SEEK_CUR);
  int N=int(input->readULong(2)); // number block of size 400000 ?
  if (N) f << "h[sz]=" << N << "*256k,";
  val=int(input->readULong(4));
  if (val!=0x502)  // 0|502, 0 if no FileDesc,Rsrcpceg,... ?
    f << "f0=" << std::hex << val << std::dec << ",";
  for (int i=0; i<6; ++i) {
    val=int(input->readULong(2));
    if (val==0)
      continue;
    f << "f" << i << "=" << val << ",";
  }
  m_state->m_fileFlags=int(input->readULong(1));
  if (m_state->m_fileFlags==0x22)
    f << "no[preview],";
  else if (m_state->m_fileFlags!=0x21)
    f << "fl=" << std::hex << m_state->m_fileFlags << std::dec << ",";
  ascFile.addDelimiter(input->tell(),'|');
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  input->seek(pos+21, librevenge::RVNG_SEEK_SET);
  return true;
}

bool Canvas5Parser::readDocumentSettings(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  auto &ascFile=stream.ascii();
  if (!input) return false;
  long pos=input->tell();
  if (!input->checkPosition(pos+54)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readDocumentSettings: the zone is too short\n"));
    return false;
  }
  libmwaw::DebugStream f;
  f << "Entries(DocSettings):";
  f << "lengths=[";
  for (int i=0; i<5; ++i) { // 5 small number: f3 numShape?
    int val=int(input->readLong(4));
    if (i==1) m_state->m_numSlides=val;
    else if (i==3) m_state->m_numShapes=val;
    if (val)
      f << val << ",";
    else
      f << "_,";
  }
  f << "],";
  for (int i=0; i<3; ++i) { // 3 small number
    int val=int(input->readLong(2));
    if (val==1)
      continue;
    if (i==0) {
      f << "num[layers]=" << val << ",";
      m_state->m_numLayers=val;
      continue;
    }
    else if (i==2) // checkme
      f << "cur[layer]=" << val << ",";
    else
      f << "f" << i << "=" << val << ",";
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  pos=input->tell();
  f.str("");
  f << "DocSettings-A:";
  int val=int(input->readLong(2));
  if (val)
    f << "f0=" << val << ",";
  val=int(input->readLong(4)); // 72|100
  if (val!=0x480000)
    f << "f1=" << float(val)/65536 << ",";
  double dVal;
  bool isNan;
  if (!readDouble(stream, dVal, isNan)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readDocumentSettings: can not read a double\n"));
    f << "###";
  }
  else
    f << "grid[dim]=" << dVal << "pt,";
  input->seek(pos+14, librevenge::RVNG_SEEK_SET);
  ascFile.addDelimiter(input->tell(),'|');
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  input->seek(pos+28, librevenge::RVNG_SEEK_SET);

  if (!m_styleManager->readPenSize(stream))
    return false;

  pos=input->tell();
  f.str("");
  f << "DocSettings-B:";
  if (!input->checkPosition(pos+4*256+134)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readDocumentSettings: the 0 zone seems too short\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  input->seek(pos+38, librevenge::RVNG_SEEK_SET);
  ascFile.addDelimiter(input->tell(),'|');

  f << "grid[dims]=[";
  for (int i=0; i<2; ++i) { // inches and points
    long actPos=input->tell();
    if (!readDouble(stream, dVal, isNan)) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readDocumentSettings: can not read a double\n"));
      f << "###";
      input->seek(actPos+8, librevenge::RVNG_SEEK_SET);
    }
    else
      f << dVal << (i==0 ? "in" : "pt") << ",";
  }
  ascFile.addDelimiter(input->tell(),'|');
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  input->seek(pos+128, librevenge::RVNG_SEEK_SET);

  pos=input->tell();
  f.str("");
  f << "DocSettings-B1:";
  input->seek(pos+128+22, librevenge::RVNG_SEEK_SET);
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  if (!m_styleManager->readFormats(stream))
    return false;

  for (int i=0; i<6; ++i) {
    pos=input->tell();
    int const len=i==1 ? 118 : i==5 ? 58 : 98;
    f.str("");
    f << "DocSettings-C" << i << ":";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+len, librevenge::RVNG_SEEK_SET);
  }

  return true;
}

bool Canvas5Parser::readBitmap(Canvas5Structure::Stream &stream, MWAWEmbeddedObject &object, MWAWColor *avgColor)
{
  object=MWAWEmbeddedObject();
  auto input=stream.input();
  long pos=input->tell();
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "Entries(Bitmap):";
  int type0=int(input->readULong(4)); // found type0=5 in texture bw bitmap
  if (type0!=6) f << "type0=" << type0 << ",";
  if (!input->checkPosition(pos+64) || (type0!=5 && type0!=6)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readBitmap: the zone beginning seems bad\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  int type=int(input->readLong(2)); // 1-3
  switch (type) {
  case 0:
    f << "bw[indexed],"; // 1 bool by bytes
    break;
  case 1:
    f << "bw[color],"; // 1 plane
    break;
  case 2:
    f << "indexed,"; // 1 plane, color map
    break;
  case 3:
    f << "color,"; // with 3 planes
    break;
  default:
    f << "##type=" << type << ",";
    MWAW_DEBUG_MSG(("Canvas5Parser::readBitmap: unexpected type\n"));
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  int numBytes=int(input->readLong(2)); // number of byte?
  if (numBytes!=8) {
    if (numBytes==1 && type==0)
      f << "num[bytes]=1,";
    else {
      MWAW_DEBUG_MSG(("Canvas5Parser::readBitmap: oops, find a number of bytes unexpected, unimplemented\n"));
      f << "##num[bytes]=" << numBytes << ",";
    }
  }
  int dim[2];
  for (auto &d : dim) d=int(input->readULong(4));
  MWAWVec2i dimension(dim[1], dim[0]);
  f << "dim=" << dimension << ",";
  int val=int(input->readLong(2)); // 1-3
  int val1=int(input->readLong(2)); // val
  if (val!=val1)
    f << "f2=" << val << "x" << val1 << ",";
  else if (val!=1) f << "f2=" << val << ",";
  float fDim[2];
  for (auto &v : fDim) v=float(input->readULong(4))/65536.f;
  if (MWAWVec2f(fDim[0],fDim[1])!=MWAWVec2f(72,72))
    f << "fDim=" << MWAWVec2f(fDim[0],fDim[1]) << ",";
  for (int i=0; i<4; ++i) { // 0
    val=int(input->readLong(2));
    if (val)
      f << "f" << i+3 << "=" << val << ",";
  }
  for (auto &d : dim) d=int(input->readULong(4));
  MWAWVec2i dim1(dim[1], dim[0]);
  if (dimension!=dim1)
    f << "dim1=" << dim1 << ",";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  // FIXME: find correctly the data, color positions
  //   but only reconstruct correctly small bitmaps :-~
  std::shared_ptr<MWAWPictBitmapIndexed> bitmapIndexed;
  std::shared_ptr<MWAWPictBitmapColor> bitmapColor;
  switch (type) {
  case 0:
  case 2:
    bitmapIndexed.reset(new MWAWPictBitmapIndexed(dimension));
    break;
  case 1:
  case 3:
  default:
    bitmapColor.reset(new MWAWPictBitmapColor(dimension));
    break;
  }

  pos=input->tell();
  int const width=type==0 ? (dimension[0]+7)/8 : dimension[0];
  long dataLength=(type==3 ? 3 : 1)*(20+width*dimension[1]);
  if (width<=0 || dimension[1]<=0 || pos+dataLength<pos || !input->checkPosition(pos+dataLength)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readBitmap: can not find the bitmap data\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }

  long dataPos=pos;
  // first read the color map
  input->seek(pos+dataLength, librevenge::RVNG_SEEK_SET);
  pos=input->tell();
  long len=input->readLong(4);
  if (pos+4+(len?4:0)+len<pos+4 || !input->checkPosition(pos+4+(len?4:0)+len)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readBitmap: can not find the color block\n"));
    ascFile.addPos(pos);
    ascFile.addNote("Bitmap[color]:###");
    return false;
  }
  if (len==0) {
    ascFile.addPos(pos);
    ascFile.addNote("_");
  }
  else {
    input->seek(4, librevenge::RVNG_SEEK_CUR);
    unsigned long numBytesRead;
    auto *data=input->read(size_t(len), numBytesRead);
    if (!data || long(numBytesRead)!=len) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readBitmap: can not find the color block\n"));
      ascFile.addPos(pos);
      ascFile.addNote("Bitmap[color]:###");
      return false;
    }
    size_t N=size_t(len/3);
    std::vector<MWAWColor> colors(N);
    for (size_t c=0; c<N; ++c) colors[c]=MWAWColor(data[c],data[c+N],data[c+2*N]);
    if (type==2)
      bitmapIndexed->setColors(colors);
    ascFile.addPos(pos);
    ascFile.addNote("Bitmap[color]:");
    ascFile.skipZone(pos+8, pos+8+len-1);
  }
  long endPos=input->tell();
  if (type==0)
    bitmapIndexed->setColors({MWAWColor::black(), MWAWColor::white()});
  // now read the bitmap data
  input->seek(dataPos, librevenge::RVNG_SEEK_SET);
  for (int plane=0; plane<(type==3 ? 3 : 1); ++plane) {
    pos=input->tell();
    f.str("");
    f << "Bitmap-P" << plane << ":";
    for (int i=0; i<6; ++i) {
      val=int(input->readLong(2));
      int const expected[]= {0,2,0,8,0,1};
      if (val==expected[i]) continue;
      if (i==3)
        f << "num[bytes]=" << val << ",";
      else
        f << "g" << i << "=" << val << ",";
    }
    for (auto &d : dim) d=int(input->readULong(4));
    dim1=MWAWVec2i(dim[1], dim[0]);
    if (dimension!=dim1)
      f << "dim2=" << dim1 << ",";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());

    if (type==0) {
      // checkme: is the picture decomposed by block if dim[0]>128*8 or dim[1]>128 ?
      for (int y=0; y<dimension[1]; ++y) {
        int x=0;
        for (int w=0; w<width; ++w) {
          val=int(input->readULong(1));
          for (int v=0, depl=0x80; v<8; ++v, depl>>=1) {
            if (x>=dimension[0])
              break;
            bitmapIndexed->set(x++,y, (val&depl) ? 0 : 1);
          }
        }
      }
    }
    else {
      for (int nY=0; nY<(dimension[1]+127)/128; ++nY) {
        for (int nW=0; nW<(dimension[0]+127)/128; ++nW) {
          for (int y=128*nY; y<std::min(dimension[1], 128*(nY+1)); ++y) {
            for (int w=128*nW; w<std::min(dimension[0], 128*(nW+1)); ++w) {
              unsigned char c=(unsigned char)input->readULong(1);
              if (type==1)
                bitmapColor->set(w,y, MWAWColor(c,c,c));
              else if (type==2)
                bitmapIndexed->set(w,y,c);
              else {
                if (plane==0)
                  bitmapColor->set(w,y,MWAWColor(c,0,0));
                else {
                  uint32_t finalValue=bitmapColor->get(w,y).value()|(uint32_t(c)<<(16-(8*plane)));
                  bitmapColor->set(w,y,MWAWColor(finalValue));
                }
              }
            }
          }
        }
      }
    }
    ascFile.skipZone(dataPos+20, dataPos+dataLength-1);
  }
  input->seek(endPos, librevenge::RVNG_SEEK_SET);

  bool ok=false;
  if (type==0 || type==2) {
    ok=bitmapIndexed->getBinary(object);
    if (ok && avgColor) *avgColor=bitmapIndexed->getAverageColor();
  }
  else if (type==1 || type==3) {
    ok=bitmapColor->getBinary(object);
    if (ok && avgColor) *avgColor=bitmapColor->getAverageColor();
  }
#ifdef DEBUG_WITH_FILES
  if (ok && !object.m_dataList.empty()) {
    std::stringstream s;
    static int index=0;
    s << "file" << ++index << ".png";
    libmwaw::Debug::dumpFile(object.m_dataList[0], s.str().c_str());
  }
#endif
  return ok && !object.m_dataList.empty();
}

bool Canvas5Parser::readLayers(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  if (!input) return false;
  long pos=input->tell();
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "Entries(Layer):";
  long len=input->readLong(4);
  long endPos=pos+4+len;
  if (len<60*(m_state->m_numLayers+1) || endPos<pos+4 || !input->checkPosition(endPos)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readLayers: can not find the layer's header\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  ascFile.addPos(pos+4);
  ascFile.addNote("_");
  input->seek(pos+60, librevenge::RVNG_SEEK_SET);
  auto fontConverter=getFontConverter();
  int defaultFont=3;
  for (int z=0; z<m_state->m_numLayers; ++z) {
    pos=input->tell();
    f.str("");
    f << "Layer-L" << z+1 << ":";
    Canvas5ParserInternal::Layer layer;
    for (int i=0; i<2; ++i) { // ?
      int val=int(input->readLong(2));
      if (val) f << "f" << i << "=" << val << ",";
    }
    f << "ID=" << std::hex << input->readULong(4) << std::dec << ",";
    int numShapes=int(input->readLong(4));
    f << "N=" << numShapes-1 << ",";
    layer.m_numShapes=numShapes-1;
    layer.m_type=int(input->readLong(4));
    f << "type=" << layer.m_type << ",";
    int val=int(input->readULong(4));
    if ((val&4)==0) f << "no[print],";
    if (val&8) f << "bw,";
    if (val&0x10) f << "protected,";
    val &= 0xffe3;
    if (val!=1) f << "fl=" << std::hex << val << std::dec << ",";
    val=int(input->readLong(4));
    if (val) // &8: also bw?
      f << "f0=" << val << ",";
    for (int i=0; i<36; ++i) {
      char c=char(input->readULong(1));
      if (!c)
        break;

      int unicode = fontConverter->unicode(defaultFont, static_cast<unsigned char>(c));
      if (unicode>0)
        libmwaw::appendUnicode(uint32_t(unicode), layer.m_name);
      else {
        MWAW_DEBUG_MSG(("CanvasParser::readLayers: find unknown unicode for char=%d\n", int(c)));
      }
    }
    if (!layer.m_name.empty())
      f << layer.m_name.cstr() << ",";
    m_state->m_idToLayer[z+1]=layer;
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+60, librevenge::RVNG_SEEK_SET);
  }
  if (input->tell()!=endPos) { // find four bytes : junk?
    ascFile.addPos(input->tell());
    ascFile.addNote("Layer-end:");
    input->seek(endPos, librevenge::RVNG_SEEK_SET);
  }

  for (auto &it : m_state->m_idToLayer) {
    auto &layer = it.second;
    if (layer.m_type==-1) continue;
    int const nextShape=layer.m_numShapes+1;
    pos=input->tell();
    len=input->readLong(4);
    f.str("");
    f << "Layer-L" << it.first << ":";
    if (len<0 || (nextShape>1 && len<4*nextShape) || pos+4+len<pos+4 || !input->checkPosition(pos+4+len)) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readLayers: can not find some layer\n"));
      f << "###";
      ascFile.addPos(pos);
      ascFile.addNote(f.str().c_str());
      return false;
    }
    if (nextShape>1) {
      input->seek(4, librevenge::RVNG_SEEK_CUR); // junk?
      f << "id=[";
      for (int s=1; s<nextShape; ++s) {
        layer.m_shapesId.push_back(int(input->readLong(4)));
        f << "S" << layer.m_shapesId.back() << ",";
      }
      f << "],";
    }
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+4+len, librevenge::RVNG_SEEK_SET);
  }
  return true;
}

bool Canvas5Parser::readPreview(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  if (!input) return false;
  long pos=input->tell();
  bool noPreview=(m_state->m_fileFlags&3)==2;
  if (!input->checkPosition(pos+12+(noPreview ? 0 : 12))) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readPreview: the zone is too short\n"));
    return false;
  }
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "Entries(Preview):";

  int dims[3];
  for (auto &d : dims) d=int(input->readLong(4));
  f << "dim=" << MWAWVec2i(dims[1], dims[0]) << "[" << dims[2] << "],";
  int width=noPreview ? 0 : int(input->readLong(4));
  if (width) f << "w=" << width << ",";
  long endPos=pos+(noPreview ? 12 : 24)+long(width*dims[0]);
  if (noPreview || dims[0]<=0 || dims[1]<=0 || dims[2]<3 || dims[2]>4 || width<dims[1]*dims[2] ||
      endPos<=pos+24 || !input->checkPosition(endPos)) {
    if (dims[0]==0 && dims[1]==0 && input->checkPosition(endPos)) {
      ascFile.addPos(pos);
      ascFile.addNote(f.str().c_str());
      input->seek(endPos, librevenge::RVNG_SEEK_SET);
      ascFile.skipZone(input->tell(), endPos-1);
      return true;
    }
    f << "###";
    MWAW_DEBUG_MSG(("Canvas5Parser::readPreview: the dimensions seems bad\n"));
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  for (int i=0; i<2; ++i) {
    int val=int(input->readLong(4));
    int const expected[]= {3,1};
    if (val!=expected[i])
      f << "f" << i << "=" << val << ",";
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  pos=input->tell();
  MWAWPictBitmapColor pict(MWAWVec2i(dims[1], dims[0]), dims[2]==4);
  for (int y=0; y<dims[0]; ++y) {
    long actPos=input->tell();
    unsigned char cols[4]= {0,0,0,0};
    for (int w=0; w<dims[1]; ++w) {
      for (int c=0; c<dims[2]; ++c) cols[c]=(unsigned char)(input->readULong(1));
      if (dims[2]==4)
        pict.set(w, y, MWAWColor(cols[1], cols[2], cols[3], (unsigned char)(255-cols[0])));
      else
        pict.set(w, y, MWAWColor(cols[0], cols[1], cols[2]));
    }
    input->seek(actPos+width, librevenge::RVNG_SEEK_SET);
  }

  input->seek(endPos, librevenge::RVNG_SEEK_SET);
  ascFile.skipZone(pos, endPos-1);
#ifdef DEBUG_WITH_FILES
  MWAWEmbeddedObject obj;
  if (pict.getBinary(obj) && !obj.m_dataList.empty())
    libmwaw::Debug::dumpFile(obj.m_dataList[0], "file.png");
#endif

  return true;
}

bool Canvas5Parser::readPrinterRsrc(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  long pos=input->tell();
  if (!input->checkPosition(pos+16)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readPrinterRsrc: can not find the input\n"));
    return false;
  }
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "Rsrcprnt-header:";
  int val;
  for (int i=0; i<3; ++i) {
    val=int(input->readLong(4));
    if (val!=(i==0 ? 4 : 0))
      f << "f" << i << "=" << val << ",";
  }

  long len=input->readLong(4);
  long endPos=pos+16+len;
  if (endPos<pos+16+24 || !input->checkPosition(endPos)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readPrinterRsrc: can not find the input\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  pos=input->tell();
  f.str("");
  f << "Rsrcprnt-A:";
  for (int i=0; i<5; ++i) {
    val=int(input->readLong(4));
    int const expected[]= {0x4000, 0, 3, 0, 0};
    if (val!=expected[i])
      f << "f" << i << "=" << val << ",";
  }
  len=input->readLong(4);
  long end1Pos=pos+24+len;
  if (end1Pos>endPos) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readPrinterRsrc: first block seems bad\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  int N=int(len/64);
  for (int i=0; i<N; ++i) {
    pos=input->tell();
    f.str("");
    f << "Rsrcprnt-B" << i << ":";
    std::string name;
    for (int j=0; j<32; ++j) {
      char ch=char(input->readLong(1));
      if (!ch)
        break;
      name+=ch;
    }
    f << name << ",";
    input->seek(pos+32, librevenge::RVNG_SEEK_SET);
    for (int j=0; j<6; ++j) { // 0
      val=int(input->readLong(2));
      if (val!=(j==5 ? -1 : 0))
        f << "f" << j << "=" << val << ",";
    }
    f << "col=[";
    for (int j=0; j<4; ++j) f << int(input->readULong(2)>>8) << ",";
    f << "],";
    std::string what; // cmyk, rgb, sepp
    for (int j=0; j<4; ++j) what+=char(input->readULong(1));
    f << what << ",";
    for (int j=0; j<4; ++j) { // g0=2d-69, g2=3c
      val=int(input->readLong(2));
      if (val!=(j==2 ? 0x3c : 0))
        f << "g" << j << "=" << val << ",";
    }
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+64, librevenge::RVNG_SEEK_SET);
  }
  input->seek(end1Pos, librevenge::RVNG_SEEK_SET);

  pos=input->tell();
  f.str("");
  f << "Rsrcprnt-C:";
  if (pos+18>endPos) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readPrinterRsrc: second block seems bad\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  for (int i=0; i<7; ++i) {
    val=int(input->readLong(2));
    if (val!=(i==6 ? 1 : 0))
      f << "f" << i << "=" << val << ",";
  }
  len=input->readLong(4);
  if (len<0x78 || pos+len+18<pos || pos+len+18>endPos) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readPrinterRsrc: printInfo block seems bad\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  pos=input->tell();
  f.str("");
  f << "Rsrcprnt-PrintInfo:";
  libmwaw::PrinterInfo info;
  if (!info.read(input)) {
    MWAW_DEBUG_MSG(("CanvasParser::readPrinterRsrc: can not read the print info data\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  f << info;
  MWAWVec2i paperSize = info.paper().size();
  MWAWVec2i pageSize = info.page().size();
  if (pageSize.x() <= 0 || pageSize.y() <= 0 ||
      paperSize.x() <= 0 || paperSize.y() <= 0) {
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  return true;
}

bool Canvas5Parser::readTextLinks(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  if (!input || !input->checkPosition(input->tell()+1)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readTextLinks: the zone is too short\n"));
    return false;
  }
  auto &ascFile=stream.ascii();
  long pos=input->tell();
  ascFile.addPos(pos);
  ascFile.addNote("Entries(TxtLink):");
  input->seek(1, librevenge::RVNG_SEEK_CUR); // 0-3 | 67 | 72 | 99
  if (!readExtendedHeader(stream, 1, "TxtLink", &Canvas5Parser::defDataFunction))
    return false;
  if (!readIndexMap(stream, "TxtLink",
  [](Canvas5Structure::Stream &lStream, int id, std::string const &, long len) {
  auto lInput=lStream.input();
    long lPos=lInput->tell();
    auto &asciiFile=lStream.ascii();
    if (len<8) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readTextLinks: can not read the txtLink\n"));
      asciiFile.addPos(lPos);
      asciiFile.addNote("###");
      return;
    }
    libmwaw::DebugStream f;
    f << "TL" << id << ":";
    int N=int(lInput->readULong(4));
    f << "N=" << N << ",";
    if ((len-8)/4<N || 8+4*N>len) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readTextLinks: can not read the txtLink N\n"));
      asciiFile.addPos(lPos);
      asciiFile.addNote("###");
    }
    int val=int(lInput->readULong(4));
    if (val)
      f << "f0=" << val << ",";
    f << "id=[";
    for (int i=0; i<N; ++i)
      f << "TLb" << lInput->readULong(4) << ",";
    f << "],";
    asciiFile.addPos(lPos);
    asciiFile.addNote(f.str().c_str());
  }))
  return false;
  std::vector<bool> defined;
  if (!readDefined(stream, defined, "TxtLink"))
    return false;

  if (!readExtendedHeader(stream, 1, "TxtLink-B", &Canvas5Parser::defDataFunction))
    return false;
  if (!readIndexMap(stream, "TxtLink-B",
  [](Canvas5Structure::Stream &lStream, int id, std::string const &, long len) {
  auto lInput=lStream.input();
    auto &asciiFile=lStream.ascii();
    long lPos=lInput->tell();
    if (len<28) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readTextLinks: can not read the txtPlcB\n"));
      asciiFile.addPos(lPos);
      asciiFile.addNote("###");
    }
    libmwaw::DebugStream f;
    f << "TLb" << id << ":";
    f << "fl=" << std::hex << lInput->readULong(2) << std::dec << ","; // 6[01]0
    int val=int(lInput->readULong(2)); // 78|100
    if (val)
      f << "f0=" << val << ",";
    f << getString(unsigned(lInput->readULong(4))) << ","; // TexU
    val=int(lInput->readULong(4));
    if (val)
      f << "TLc" << val << ","; // checkme
    lInput->seek(4, librevenge::RVNG_SEEK_CUR);
    int N=int(lInput->readULong(4));
    f << "N=" << N << ",";
    if (N>(len-28)/4 || N<0 || len<28+4*N) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readTextLinks: can not find the list of block\n"));
      f << "###";
      N=0;
    }
    lInput->seek(8, librevenge::RVNG_SEEK_CUR); // junk, flag?
    f << "shapes=[";
    for (int i=0; i<N; ++i)
      f << "S" << lInput->readULong(4) << ",";
    f << "],";
    if (lInput->tell()!=lPos+len)
      asciiFile.addDelimiter(lInput->tell(),'|');
    asciiFile.addPos(lPos);
    asciiFile.addNote(f.str().c_str());
  }))
  return false;
  if (!readDefined(stream, defined, "TxtLink-B"))
    return false;

  if (!readExtendedHeader(stream, 1, "TxtLink-C", &Canvas5Parser::defDataFunction))
    return false;
  // find 0000000200000001000000020238fc000238fcc00000000000014440023a58100000002d0000000a010100c50238fcc00000
  if (!readIndexMap(stream, "TxtLink-C"))
    return false;
  return readDefined(stream, defined, "TxtLink-C");
}

bool Canvas5Parser::readSlides(Canvas5Structure::Stream &stream)
{
  auto input=stream.input();
  if (!input || !input->checkPosition(input->tell()+64)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readSlides: the zone is too short\n"));
    return false;
  }

  long pos=input->tell();
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();

  f << "Entries(Slide):";
  int val=int(input->readLong(4)); // 1-3
  if (val) f << "f0=" << val << ",";
  float fDim[4];
  for (auto &d : fDim) d=float(input->readULong(4))/65536.f;
  f << "dim?=" << MWAWVec2f(fDim[0], fDim[1]) << ",";
  MWAWVec2f pageDim=MWAWVec2f(fDim[2], fDim[3]);
  f << "page[dim]=" << pageDim << ",";
  val=int(input->readULong(4));
  switch (val) {
  case 0: // full page
    break;
  case 1: // the page's height is divided by 2, ie. we print on each page TOP: page 2*N+1(reverted), BOTTOM: page 2*N
    m_state->m_documentSetup=1;
    f << "setup=top/bottom,";
    break;
  case 2: // greetings pages, page height/width is divided by 2
    m_state->m_documentSetup=2;
    f << "setup=greetings,";
    break;
  default:
    MWAW_DEBUG_MSG(("Canvas5Parser::readSlides: find unknown setup type\n"));
    f << "###setup=" << val << ",";
    break;
  }
  val=int(input->readULong(4));
  if (val==2) {
    f << "pages[facing],";
    m_state->m_facingPages=true;
  }
  else if (val) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readSlides: find unknown facing pages\n"));
    f << "###pages[facing]=" << val << ",";
  }
  for (auto &d : fDim) d=float(input->readULong(4))/65536.f;
  f << "margins=" << MWAWBox2f(MWAWVec2f(fDim[0], fDim[1]),MWAWVec2f(fDim[2], fDim[3])) << ",";

  // time to set the page dimension
  if (pageDim[0]>100 && pageDim[1]>100) {
    // checkme: check the margins ordering
    if (fDim[0]>=0)
      getPageSpan().setMarginTop((fDim[0]>14 ? fDim[0]-14 : 0)/72);
    if (fDim[1]>=0)
      getPageSpan().setMarginLeft((fDim[1]>14 ? fDim[1]-14 : 0)/72);
    if (fDim[2]>=0)
      getPageSpan().setMarginBottom((fDim[2]>10 ? fDim[2]-10 : 0)/72);
    if (fDim[3]>=0)
      getPageSpan().setMarginRight((fDim[3]>10 ? fDim[3]-10 : 0)/72);
    getPageSpan().setFormLength(pageDim[1]/72);
    getPageSpan().setFormWidth(pageDim[0]/72);
  }

  for (int i=0; i<4; ++i) { // g3=0
    val=int(input->readLong(2));
    if (!val) continue;
    f << "g" << i << "=" << val << ",";
  }
  int N=int(input->readLong(4));
  if (N)
    f << "N=" << N << ",";
  f << "IDs=[";
  for (int i=0; i<2; ++i) f << std::hex << input->readULong(4) << std::dec << ",";
  f << "],";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  input->seek(pos+64, librevenge::RVNG_SEEK_SET);

  pos=input->tell();
  long len=input->readLong(4);
  if (len<4 || N<0 || len/4<N || pos+4+len<pos+4 || !input->checkPosition(pos+4+len)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readSlides: can not read the Slides length\n"));
    ascFile.addPos(pos);
    ascFile.addNote("Entries(Bad):###");
    return false;
  }
  f.str("");
  f << "Slide-id:";
  input->seek(4, librevenge::RVNG_SEEK_CUR);
  f << "id=[";
  for (int i=0; i<N; ++i) {
    m_state->m_slideIds.push_back(int(input->readLong(4)));
    f << m_state->m_slideIds.back() << ",";
  }
  f << "],";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  input->seek(pos+4+len, librevenge::RVNG_SEEK_SET);

  if (!readExtendedHeader(stream, 0xac, "Slide",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &what, long) {
  auto lInput=lStream.input();
    long lPos=lInput->tell();
    libmwaw::DebugStream lF;
    auto &asciiFile=lStream.ascii();

    if (m_state->m_idToSlide.find(id)==m_state->m_idToSlide.end())
      m_state->m_idToSlide[id]=Canvas5ParserInternal::Slide();
    auto &slide=m_state->m_idToSlide.find(id)->second;
    int lVal;
    for (int i=0; i<4; ++i) { // f1=240|2c0|6c0, f3=0|1
      lVal=int(lInput->readLong(2));
      if (lVal)
        lF << "f" << i << "=" << lVal << ",";
    }
    float fDims[2];
    for (auto &d : fDims) d=float(lInput->readLong(4))/65536.f;
    slide.m_dim = MWAWVec2f(fDims[0],fDims[1]);
    lF << "page[dim]=" << slide.m_dim << ",";
    for (int i=0; i<2; ++i) {  // f5=1-13,
      lVal=int(lInput->readLong(2));
      if (lVal)
        lF << "f" << 4+i << "=" << lVal << ",";
    }
    asciiFile.addDelimiter(lInput->tell(),'|');
    lInput->seek(8, librevenge::RVNG_SEEK_CUR);
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());

    lPos=lInput->tell();
    lF.str("");
    lF << what << "-E" << id << "[name]:";
    std::string text;
    for (int i=0; i<256; ++i) {
      char c=char(lInput->readULong(1));
      if (c==0)
        break;
      text+=c;
    }
    lF << text << ",";
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
    lInput->seek(lPos+128, librevenge::RVNG_SEEK_SET);

    lPos=lInput->tell();
    lF.str("");
    lF << what << "-E" << id << "[A]:";
    for (int i=0; i<6; ++i) { // f2=0|260, f4=0-3, f5=0-2
      lVal=int(lInput->readLong(2));
      if (!lVal) continue;
      lF << "f" << i << "=" << lVal << ",";
    }
    slide.m_numLayers=int(lInput->readLong(4));
    if (slide.m_numLayers!=1)
      lF << "num[layer]=" << slide.m_numLayers << ",";
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  }))
  return false;

  if (!readIndexMap(stream, "SlideLa",
  [this](Canvas5Structure::Stream &lStream, int id, std::string const &, long lLen) {
  auto lInput=lStream.input();
    auto &asciiFile=lStream.ascii();
    long lPos=lInput->tell();
    auto it=m_state->m_idToSlide.find(id);
    if (it==m_state->m_idToSlide.end() || lLen<8+8*it->second.m_numLayers) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readSlides: can not read the slides index %d\n", id));
      asciiFile.addPos(lPos);
      asciiFile.addNote("###");
      return;
    }
    libmwaw::DebugStream lF;
    auto &slide=it->second;
    lInput->seek(8, librevenge::RVNG_SEEK_CUR); // 0
    lF << "layers=[";
    for (int i=0; i<slide.m_numLayers; ++i) {
      lF << "[";
      slide.m_layers.push_back(int(lInput->readULong(4)));
      lF << "L" << slide.m_layers.back() << ",";
      for (int j=0; j<2; ++j) { // f2 0 | big number
        int lVal=int(lInput->readLong(2));
        if (lVal) lF << "f" << i+1 << "=" << lVal << ",";
      }
      lF << "],";
    }
    lF << "],";
    asciiFile.addPos(lPos);
    asciiFile.addNote(lF.str().c_str());
  }))
  return false;

  std::vector<bool> defined;
  return readDefined(stream, defined, "Slide");
}

////////////////////////////////////////////////////////////
// read the header
////////////////////////////////////////////////////////////
bool Canvas5Parser::checkHeader(MWAWHeader *header, bool strict)
{
  MWAWInputStreamPtr input = getInput();
  if (!input || !input->hasDataFork() || !input->checkPosition(0x100))
    return false;

  input->setReadInverted(false);
  input->seek(0, librevenge::RVNG_SEEK_SET);
  int byteOrdering=int(input->readLong(1));
  switch (byteOrdering) {
  case 1:
    m_state->m_isWindowsFile=true;
    input->setReadInverted(true);
    break;
  case 2:
    break;
  default:
    return false;
  }
  if (input->readULong(4)!=0x8000)
    return false;

  for (int i=0; i<2; ++i) {
    // try to read the ith zone header
    int compType=int(input->readULong(4));
    if (compType<0 || compType>8)
      return false;

    long len=long(input->readULong(4));
    if ((i==0 && len<0x800) || len>0x8000)
      return false;

    len=long(input->readULong(4));
    if (len+17<=17 || !input->checkPosition(17+len))
      return false;
    input->seek(len, librevenge::RVNG_SEEK_CUR);
    if (!strict)
      break;
  }
  int const vers=5;
  setVersion(vers);
  if (header)
    header->reset(MWAWDocument::MWAW_T_CANVAS, vers, MWAWDocument::MWAW_K_DRAW);

  input->seek(5, librevenge::RVNG_SEEK_SET);
  return true;
}

// ------------------------------------------------------------
// mac resource fork
// ------------------------------------------------------------

bool Canvas5Parser::readPnot(Canvas5Structure::Stream &stream, MWAWEntry const &entry)
{
  auto input=stream.input();
  if (!input || !entry.valid() || !input->checkPosition(entry.end()))
    return false;
  if (entry.length()<14) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readPnot: the zone seems too small\n"));
    return false;
  }
  entry.setParsed(true);
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "Entries(Pnot)[" << entry.id() << "]:";
  input->seek(entry.begin(),librevenge::RVNG_SEEK_SET);
  f << "ID=" << std::hex << input->readULong(4) << std::dec << ",";
  int val=int(input->readLong(2));
  if (val) f << "f0=" << val << ",";
  val=int(input->readULong(4)); // PICT
  f << getString(unsigned(val)) << ",";
  f << "id=" << input->readULong(2) << ",";
  val=int(input->readLong(2));
  if (val) f << "f1=" << val << ",";

  ascFile.addPos(entry.begin()-4);
  ascFile.addNote(f.str().c_str());
  return true;
}

bool Canvas5Parser::readPicture(Canvas5Structure::Stream &stream, MWAWEntry const &entry)
{
  auto input=stream.input();
  if (!input || !entry.valid() || !input->checkPosition(entry.end()))
    return false;
  if (entry.length()<14) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readPicture: the zone seems too small\n"));
    return false;
  }
  entry.setParsed(true);
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << "Entries(RSRCPicture)[" << entry.id() << "]:";
#ifdef DEBUG_WITH_FILES
  input->seek(entry.begin(), librevenge::RVNG_SEEK_SET);
  std::shared_ptr<MWAWPict> thePict(MWAWPictData::get(input, int(entry.length())));
  MWAWEmbeddedObject obj;
  bool ok=thePict && thePict->getBinary(obj);
  librevenge::RVNGBinaryData file;
  input->seek(entry.begin(), librevenge::RVNG_SEEK_SET);
  input->readDataBlock(entry.length(), file);
  libmwaw::DebugStream s;
  s << "PICT-" << entry.id() << ".pct";
  libmwaw::Debug::dumpFile(file, s.str().c_str());
  if (!ok)
    f << "###";
  else
    ascFile.skipZone(entry.begin(), entry.end()-1);
#endif
  ascFile.addPos(entry.begin()-4);
  ascFile.addNote(f.str().c_str());
  return true;
}

// ------------------------------------------------------------
// windows resource fork
// ------------------------------------------------------------

////////////////////////////////////////////////////////////
//
// send data
//
////////////////////////////////////////////////////////////

bool Canvas5Parser::send(Canvas5ParserInternal::Slide const &slide)
{
  auto listener=getGraphicListener();
  if (!listener) {
    MWAW_DEBUG_MSG(("Canvas5Parser::send[slide]: can not find the listener\n"));
    return false;
  }
  size_t numLayers=slide.m_layers.size();
  bool needOpenLayer=numLayers==2 ? (m_state->m_type==1 || slide.m_layers[0]!=1) : numLayers>2;
  int layerId=0;
  for (auto lId : slide.m_layers) {
    if (lId==1 && m_state->m_type>=2) continue; // this is the master page
    auto it = m_state->m_idToLayer.find(lId);
    if (it==m_state->m_idToLayer.end()) {
      MWAW_DEBUG_MSG(("Canvas5Parser::send[slide]: can not find layer %d\n", lId));
      continue;
    }
    auto const &layer=it->second;
    if (needOpenLayer) {
      if (!layer.m_name.empty())
        listener->openLayer(layer.m_name);
      else {
        std::stringstream s;
        s << "Layer" << ++layerId;
        listener->openLayer(s.str().c_str());
      }
    }
    send(layer);
    if (needOpenLayer)
      listener->closeLayer();
  }
  return true;
}

bool Canvas5Parser::send(Canvas5ParserInternal::Layer const &layer)
{
  auto listener=getGraphicListener();
  if (!listener) {
    MWAW_DEBUG_MSG(("Canvas5Parser::send[layer]: can not find the listener\n"));
    return false;
  }
  for (auto sId : layer.m_shapesId)
    m_graphParser->sendShape(sId);
  return true;
}

// ------------------------------------------------------------
// low level
// ------------------------------------------------------------
bool Canvas5Parser::readDouble(Canvas5Structure::Stream &stream, double &val, bool &isNaN) const
{
  auto input=stream.input();
  if (!input || !input->checkPosition(input->tell()+8)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readDouble: can not find the input\n"));
    return false;
  }
  if (input->readInverted())
    return input->readDoubleReverted8(val, isNaN);
  return input->readDouble8(val, isNaN);
}

bool Canvas5Parser::readString(Canvas5Structure::Stream &stream, librevenge::RVNGString &string, int maxSize, bool canBeCString)
{
  string.clear();
  auto input=stream.input();
  if (!input) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readString: can not find the input\n"));
    return false;
  }
  bool const isWindows=isWindowsFile();
  auto fontConverter=getFontConverter();
  int defaultFont=isWindows ? fontConverter->getId("CP1252") : 3;
  if (isWindows && canBeCString) {
    int n=0;
    while (!input->isEnd() && (maxSize<=0 || n<maxSize)) {
      char c=char(input->readULong(1));
      if (c==0)
        break;
      int unicode = fontConverter->unicode(defaultFont, static_cast<unsigned char>(c));
      if (unicode>0)
        libmwaw::appendUnicode(uint32_t(unicode), string);
      else {
        MWAW_DEBUG_MSG(("Canvas5Parser::readString: find unknown unicode for char=%d\n", int(c)));
      }
    }
    return true;
  }
  int sSz=int(input->readULong(1));
  if ((maxSize<=0 || sSz<maxSize) && input->checkPosition(input->tell()+sSz)) {
    for (int ch=0; ch<sSz; ++ch) {
      char c=char(input->readULong(1));
      if (c==0)
        break;
      int unicode = fontConverter->unicode(defaultFont, static_cast<unsigned char>(c));
      if (unicode>0)
        libmwaw::appendUnicode(uint32_t(unicode), string);
      else {
        MWAW_DEBUG_MSG(("Canvas5Parser::readString: find unknown unicode for char=%d\n", int(c)));
      }
    }
  }
  else {
    MWAW_DEBUG_MSG(("Canvas5Parser::readString: bad size=%d\n", sSz));
    return false;
  }
  return true;
}

bool Canvas5Parser::readDataHeader(Canvas5Structure::Stream &stream, int expectedSize, int &N)
{
  auto input=stream.input();
  if (!input)
    return false;
  long pos=input->tell();
  if (!input->checkPosition(pos+4))
    return false;
  int dSize=int(input->readULong(4));
  if (!dSize) {
    N=0;
    return true;
  }
  if (dSize<0 || dSize!=expectedSize || !input->checkPosition(pos+8))
    return false;
  N=int(input->readULong(4));
  if (N<0 || (input->size()-pos)/long(dSize)<N || pos+8+long(dSize)*N<pos+8 || !input->checkPosition(pos+8+long(dSize)*N))
    return false;
  return true;
}

bool Canvas5Parser::readExtendedHeader(Canvas5Structure::Stream &stream, int expectedValue, std::string const &what, DataFunction const &func)
{
  auto input=stream.input();
  long pos=input ? input->tell() : 0;
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << what << "-extended:";
  if (!input || !input->checkPosition(input->tell()+12) || int(input->readULong(4))!=expectedValue) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readExtendedHeader: the size seems bad\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  int f0=int(input->readLong(4));
  int N=int(input->readULong(4));
  if (N) f << "N=" << N << ",";
  if (f0) f << "f0=" << f0 << ",";
  if (N<0 || pos+8+N<pos || !input->checkPosition(pos+8+N)) {
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  if (expectedValue<=0 || (N%expectedValue)!=0) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readExtendedHeader: the data size seems bad\n"));
    f << "###";
    ascFile.addDelimiter(input->tell(),'|');
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+12+N, librevenge::RVNG_SEEK_SET);
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  if (!N) return true;
  if (expectedValue==1) {
    pos=input->tell();
    f.str("");
    f<< what << "-E0:";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    func(stream, 0, what, N);
    if (input->tell()!=pos && input->tell()!=pos+N)
      ascFile.addDelimiter(input->tell(),'|');
    input->seek(pos+N, librevenge::RVNG_SEEK_SET);
    return true;
  }

  N/=expectedValue;
  // the first value seems always a buffer (which contains junk data)
  ascFile.addPos(input->tell());
  ascFile.addNote("_");
  input->seek(expectedValue, librevenge::RVNG_SEEK_CUR);
  for (int i=1; i<N; ++i) {
    pos=input->tell();
    f.str("");
    f<< what << "-E" << i << ":";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    func(stream, i, what, expectedValue);
    if (input->tell()!=pos && input->tell()!=pos+expectedValue)
      ascFile.addDelimiter(input->tell(),'|');
    input->seek(pos+expectedValue, librevenge::RVNG_SEEK_SET);
  }
  return true;
}

bool Canvas5Parser::readUnknownHeader(Canvas5Structure::Stream &stream, MWAWEntry(&entries)[2], long len, std::string const &what)
{
  auto input=stream.input();
  long pos=input->tell();
  if (len==0)
    return true;
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << what << "-A:";
  long endPos=pos+len;
  if (len<180 || !input->checkPosition(endPos)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readUnknownHeader: the zone seems too short\n"));
    f << "###sz";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  int val;
  val=int(input->readLong(4));
  if (val!=256)
    f << "f0=" << val << ",";
  auto tBegin=long(input->readULong(4));
  auto tLen=long(input->readULong(4));
  f << "pos=" << tBegin << "<->" << tBegin+tLen << ",";
  if (tBegin+tLen<0 || tLen<36 || tBegin<180 || tBegin+tLen>len) {
    f << "###";
    MWAW_DEBUG_MSG(("Canvas5Parser::readUnknownHeader: can not read an effect length\n"));
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  if (tBegin+tLen<len) {
    ascFile.addPos(pos+tBegin+tLen);
    ascFile.addNote((what+"-A[end]:").c_str());
  }

  float dim[4];
  for (auto &d : dim) d=float(input->readLong(4))/65536;
  f << "dim=" << MWAWBox2f(MWAWVec2f(dim[0],dim[1]),MWAWVec2f(dim[2],dim[3])) << ",";
  for (int i=0; i<2; ++i) {
    val=int(input->readLong(4));
    if (val!=1-i)
      f << "f" << i+1 << "=" << val << ",";
  }
  for (auto &d : dim) d=float(input->readLong(4))/65536;
  f << "dim2=" << MWAWBox2f(MWAWVec2f(dim[1],dim[0]),MWAWVec2f(dim[3],dim[2])) << ",";
  for (int st=0; st<2; ++st) {
    f << "mat" << st << "=[";
    for (int j=0; j<9; ++j)
      f << float(input->readLong(4))/65536 << ",";
    f << "],";
  }
  for (int j=0; j<3; ++j) { // g1=54|6c
    val=int(input->readLong(4));
    if (val!=-1)
      f << "g" << j << "=" << val << ",";
  }
  int dDim[2]; // g1+48, g1+24
  for (auto &d : dDim) d=int(input->readLong(4));
  f << "dim3=" << MWAWVec2i(dDim[1],dDim[0]) << ",";
  for (int j=0; j<5; ++j) {
    val=int(input->readLong(4));
    int const expected[]= {-1,-1,1,0,0};
    if (val!=expected[j])
      f << "g" << j+3 << "=" << val << ",";
  }
  f << "entries=[";
  for (auto &entry : entries) {
    entry.setBegin(input->readLong(4)+pos);
    entry.setLength(input->readLong(4));
    f << std::hex << entry.begin() << ":" << entry.end() << std::dec << ",";
    if (entry.begin()<pos || entry.end()>pos+tBegin+tLen) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readUnknownHeader: unexpected subs size for an effect\n"));
      f << "###";
      entry.setLength(0);
    }
  }
  f << ",";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  input->seek(endPos, librevenge::RVNG_SEEK_SET);
  return true;
}

void Canvas5Parser::stringDataFunction(Canvas5Structure::Stream &stream, int, std::string const &, long len)
{
  auto input=stream.input();
  auto &ascFile=stream.ascii();
  long pos=input->tell();
  libmwaw::DebugStream f;
  for (int i=0; i<int(len); ++i) {
    char c=char(input->readULong(1));
    if (c==0)
      break;
    f << c;
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
}

bool Canvas5Parser::readIndexMap(Canvas5Structure::Stream &stream, std::string const &what, DataFunction const &func)
{
  auto input=stream.input();
  if (!input) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readIndexMap: can not find the input\n"));
    return false;
  }
  long pos=input->tell();
  libmwaw::DebugStream f;
  auto &ascFile = stream.ascii();
  f << what << "[id]:";

  int N;
  if (!readDataHeader(stream, 12,N)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readIndexMap: can not read the header N\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  f << "N=" << N << ",";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  std::map<int, MWAWEntry> idToEntry;
  for (int i=1; i<=N; ++i) {
    pos=input->tell();
    f.str("");
    f << what << "-" << i << "[id]:";
    MWAWEntry entry;
    long len=long(input->readULong(4));
    int id=int(input->readULong(4));
    if (id==0) {
      ascFile.addPos(pos);
      ascFile.addNote("_");
      input->seek(pos+12, librevenge::RVNG_SEEK_SET);
      continue;
    }
    if (id!=1) f << "id=" << id << ",";
    entry.setBegin(long(input->readULong(4)));
    entry.setLength(len);
    entry.setId(id);
    f << std::hex << entry.begin() << "<->" << entry.end() << std::dec << ",";
    if (entry.valid())
      idToEntry[i]=entry;
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+12, librevenge::RVNG_SEEK_SET);
  }

  pos=input->tell();
  if (!input->checkPosition(pos+4)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readIndexMap: can not find the number of map\n"));
    return false;
  }
  f.str("");
  f << what << "[data]:";
  N=int(input->readULong(4));
  f << "num=" << N << ",";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  for (int z=1; z<=N; ++z) {
    pos=input->tell();
    f.str("");
    f << what << "[data-" << z << "]:";
    int len=int(input->readULong(4));
    f << "len=" << len << ",";
    long endPos=pos+4+len;
    if (len<0 || !input->checkPosition(pos+4+len)) {
      MWAW_DEBUG_MSG(("Canvas5Parser::readIndexMap: can not find the length of the map data %d\n", z));
      f << "###";
      ascFile.addPos(pos);
      ascFile.addNote(f.str().c_str());
      return false;
    }

    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    if (len==0) continue;
    pos+=4;

    for (auto const &it : idToEntry) {
      auto const &entry=it.second;
      if (entry.id()!=z) continue;
      if (pos+entry.end()>endPos) {
        MWAW_DEBUG_MSG(("Canvas5Parser::readIndexMap: can not find data %d\n", it.first));
        continue;
      }
      ascFile.addPos(pos+entry.end());
      ascFile.addNote("_");
      f.str("");
      f << what << "-Dt" << it.first << ":";
      ascFile.addPos(pos+entry.begin());
      ascFile.addNote(f.str().c_str());
      input->seek(pos+entry.begin(), librevenge::RVNG_SEEK_SET);
      func(stream, it.first, what, entry.length());
    }
    input->seek(endPos, librevenge::RVNG_SEEK_SET);
  }

  return true;
}

bool Canvas5Parser::readDefined(Canvas5Structure::Stream &stream, std::vector<bool> &defined, std::string const &what)
{
  auto input=stream.input();
  long pos=input ? input->tell() : 0;
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << what << "[def,N]:";
  if (!input || !input->checkPosition(pos+16)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readDefined: can not find the input\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  f << "N=["; //
  for (int i=0; i<2; ++i) f << input->readLong(4) << ",";
  f << "],";
  int val=int(input->readLong(4)); // 0
  if (val) f << "f0=" << val << ",";
  if (input->readLong(4)!=4) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readDefined: bad header\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  pos=input->tell();
  f.str("");
  f << what << "[def]:";
  int N;
  if (!readDataHeader(stream, 4,N)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readDefined: can not read N\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  f << "N=" << N << ",";
  f << "def=[";
  defined.clear();
  for (int i=0; i<N; ++i) {
    defined.push_back(input->readLong(4)!=0); // 0 or -1
    f << (defined.back() ? "*" : "_") << ",";
  }
  f << "],";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  input->seek(pos+8+4*N, librevenge::RVNG_SEEK_SET);
  return true;
}

bool Canvas5Parser::readUsed(Canvas5Structure::Stream &stream, std::string const &what)
{
  auto input=stream.input();
  long pos=input ? input->tell() : 0;
  libmwaw::DebugStream f;
  auto &ascFile=stream.ascii();
  f << what << "[used,N]:";
  if (!input || !input->checkPosition(pos+20) || input->readULong(4)!=4) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readUsed: can not find the input\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  f << "N=["; // used, id
  for (int i=0; i<2; ++i) f << input->readLong(4) << ",";
  f << "],";
  for (int i=0; i<2; ++i) {
    int val=int(input->readLong(4));
    if (val!=(i==1 ? 8 : 0)) f << "f" << i << "=" << val << ",";
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  pos=input->tell();
  f.str("");
  f << what << "[used]:";
  int N;
  if (!readDataHeader(stream,8,N)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::readUsed: can not read N\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }
  f << "N=" << N << ",";
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  for (int i=0; i<N; ++i) {
    pos=input->tell();
    f.str("");
    f << what << "-U" << i+1 << ":";
    int val=int(input->readLong(4));
    if (val!=-1) f << "f0=" << val << ",";
    val=int(input->readLong(4));
    if (val==0) {
      ascFile.addPos(pos);
      ascFile.addNote("_");
      continue;
    }
    if (val!=1) f << "used=" << val << ",";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
  }
  return true;
}

// ------------------------------------------------------------
// decode stream
// ------------------------------------------------------------
MWAWInputStreamPtr Canvas5Parser::decode(MWAWInputStreamPtr input)
{
  MWAWInputStreamPtr res;
  if (!input)
    return res;

  long pos=5;
  if (!input->checkPosition(pos+12)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::decode: the input seems too short\n"));
    return res;
  }

  input->seek(0, librevenge::RVNG_SEEK_SET);
  unsigned long read;
  const unsigned char *dt = input->read((unsigned long)(pos), read);
  if (!dt || read != (unsigned long)(pos)) {
    MWAW_DEBUG_MSG(("Canvas5Parser::decode: can not read some data\n"));
    return res;
  }

  auto stream=std::make_shared<MWAWStringStream>(dt, unsigned(pos));
  while (!input->isEnd()) {
    pos=input->tell();
    if (!input->checkPosition(pos+12))
      break;
    int type=int(input->readULong(4));
    unsigned long lengths[2];
    for (auto &l : lengths) l=input->readULong(4);
    long endPos=pos+12+long(lengths[1]);
    if (type<0 || type>8 || lengths[0]+8<lengths[1] || long(lengths[1])<0 ||
        endPos<pos+12 || !input->checkPosition(endPos)) {
      input->seek(pos, librevenge::RVNG_SEEK_SET);
      break;
    }
    if (!Canvas5Structure::decodeZone(input, endPos, type, lengths[0], stream)) {
      MWAW_DEBUG_MSG(("Canvas5Parser::decode: problem with type=%d at position=%lx\n", type, (unsigned long)pos));
      input->seek(pos, librevenge::RVNG_SEEK_SET);
      return res;
    }
    input->seek(endPos, librevenge::RVNG_SEEK_SET);
#ifdef DEBUG
    if (lengths[0]!=0x8000) {
      stream->seek(0, librevenge::RVNG_SEEK_END);
      std::cout << "\t" << std::hex << stream->tell() << std::dec << "\n";
    }
#endif
  }
  if (!input->isEnd()) { // last zone is not compressed
    MWAW_DEBUG_MSG(("Canvas5Parser::decode: stop at pos=%lx->%lx\n", (unsigned long) input->tell(),
                    (unsigned long) stream->tell()));
    unsigned long remain=(unsigned long)(input->size()-input->tell());
    dt = input->read(remain, read);
    if (!dt || read != remain) {
      MWAW_DEBUG_MSG(("Canvas5Parser::decode: can not read some data\n"));
      return res;
    }
    stream->append(dt, unsigned(remain));
  }

  res.reset(new MWAWInputStream(stream, false));
  res->seek(5, librevenge::RVNG_SEEK_SET);
  res->setReadInverted(input->readInverted());
  return res;
}

std::string Canvas5Parser::getString(unsigned val)
{
  if (val<20) return std::to_string(val);
  std::string res;
  for (int dec=24; dec>=0; dec-=8) {
    char c=char((val>>dec)&0xff);
    if (!std::isprint(c))
      return std::to_string(val);
    res+=c;
  }
  return res;
}

// vim: set filetype=cpp tabstop=2 shiftwidth=2 cindent autoindent smartindent noexpandtab:
