/*
 * Copyright 2024 Aon Cyber Solutions
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#pragma once

#include "basic.h"
#include "automata.h"
#include "parsenode.h"

#include <vector>

typedef std::vector<NFA::VertexDescriptor> InListT;
typedef std::vector<std::pair<NFA::VertexDescriptor, uint32_t>> OutListT;

static const uint32_t NOSKIP = std::numeric_limits<uint32_t>::max();

struct Fragment {
  Fragment(): Skippable(NOSKIP) {}

  Fragment(NFA::VertexDescriptor in, const ParseNode& n):
    InList(1, in), N(n), Skippable(NOSKIP) {}

  /*
   * InList is the list of vertices in this fragment which have incoming
   * edges from outside the fragment. OutList is the is the list of vertices
   * in this fragment which have edges leaving the fragment.
   */
  InListT InList;
  OutListT OutList;
  ParseNode N;

  uint32_t Skippable;

  void initFull(NFA::VertexDescriptor in, const ParseNode& n) {
    N = n;
    Skippable = NOSKIP;
    InList.clear();
    InList.push_back(in);
    OutList.clear();
    OutList.emplace_back(in, 0);
  }

  void reset(const ParseNode& n) {
    N = n;
    Skippable = NOSKIP;
    InList.clear();
    OutList.clear();
  }

  void assign(Fragment& f) {
    InList.swap(f.InList);
    OutList.swap(f.OutList);
    N = f.N;
    Skippable = f.Skippable;
  }
};
