/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Zaliva <lord@crocodile.org>

  $Id: fwbd.c,v 1.16 2002/08/29 00:34:09 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#ifndef __MINGW32__
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <resolv.h>
#include <netdb.h>
#else
#include <winsock2.h>
#endif

#include "fwbd/fwbd.h"

#undef DONT_CHECK_CERTIFICATES

#ifdef HAVE_LIBSSL
static int do_connect(struct fwbd_connection *con)
{
    int sd;
    struct hostent *host;
    struct sockaddr_in addr;

    if((host = gethostbyname(con->host)) == NULL)
    {
        con->lasterr=strdup(strerror(errno));
        return -1;
    }
    
    sd = socket(PF_INET, SOCK_STREAM, 0);
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(con->port);
    addr.sin_addr.s_addr = *(long*)(host->h_addr);
    if(connect(sd, (struct sockaddr*)&addr, sizeof(addr)) != 0)
    {
        close(sd);
        con->lasterr=strdup(strerror(errno));
        return -1;
    }

    return sd;
}
#endif

#ifdef HAVE_LIBSSL
static char *last_ssl_err()
{
    char err_buf[1024];
    *err_buf='\0';
    ERR_error_string_n(ERR_get_error(), err_buf, sizeof(err_buf));
    return strdup(err_buf);
}
#endif

/* --------------------- public functions below -------------- */

void fwbd_init()
{
#ifdef HAVE_LIBSSL
    SSL_library_init();
    OpenSSL_add_all_algorithms();
    SSL_load_error_strings();
#endif
}

struct fwbd_connection * fwdb_new_connection()
{
#ifndef HAVE_LIBSSL
    return NULL;
#else
    
    struct fwbd_connection *con = malloc(sizeof(struct fwbd_connection));
    con->server    = -1   ;
    con->ctx       = NULL ;
    con->ssl       = NULL ;
    con->connected = 0    ;
    con->host      = NULL ;
    con->port      = 0    ;
    con->lasterr   = NULL ;
    return con;
#endif
}

void fwdb_free_connection(struct fwbd_connection *c)
{
    free(c);
}


#ifdef HAVE_LIBSSL
static int load_certificates(struct fwbd_connection *con, EVP_PKEY *pkey, X509 *cert)
{
    if(SSL_CTX_use_certificate(con->ctx, cert) <= 0)
    {
        con->lasterr=strdup("Error while loading certifcate file");
        return 1;
    }
    
    if(SSL_CTX_use_PrivateKey(con->ctx, pkey) <= 0 )
    {
        con->lasterr=strdup("Error while loading key file");
        return 2;
    }
    
    if(!SSL_CTX_check_private_key(con->ctx) )
    {
        con->lasterr=strdup("Private key does not match the public certificate");
        return 3;
    }
    
    return 0;
}
#endif

#ifdef HAVE_LIBSSL
static int fwbd_verify_callback(int preverify_ok, X509_STORE_CTX *ctx)
{
    return 1;
}
#endif

int fwbd_connect(const char *host, int port, struct fwbd_connection *con, EVP_PKEY *pkey, X509 *cert)
{
#ifndef HAVE_LIBSSL
    return -9;
#else

    con->lasterr = NULL;
    
    con->ctx = SSL_CTX_new(TLSv1_client_method());
    if(!con->ctx)
    {
        con->lasterr = last_ssl_err();
        return -1;
    }
    
#ifdef DONT_CHECK_CERTIFICATES
    SSL_CTX_set_verify(con->ctx, SSL_VERIFY_NONE, NULL);
#else
    SSL_CTX_set_verify(con->ctx, SSL_VERIFY_PEER|SSL_VERIFY_FAIL_IF_NO_PEER_CERT|SSL_VERIFY_CLIENT_ONCE, fwbd_verify_callback);
    SSL_CTX_set_verify_depth(con->ctx, 0); /* The depth count is 0: peer certificate */
#endif
    
#ifndef DONT_CHECK_CERTIFICATES
    if(cert && pkey)
    {
        if(load_certificates(con, pkey, cert)!=0)
            return -1;
    }
#endif
    
    con->host = strdup(host);
    con->port = port;
    
    con->server = do_connect(con);
    if(con->server==-1)
    {
        return -1;
    }
    
    con->ssl    = SSL_new(con->ctx);
    SSL_set_fd(con->ssl, con->server);
    if(SSL_connect(con->ssl) == -1)
    {
        con->lasterr = last_ssl_err();
        fwbd_disconnect(con);
        return -1;
    }

    con->connected = 1;
    
    return 0;
#endif
    
}

int fwbd_disconnect(struct fwbd_connection *con)
{
#ifndef HAVE_LIBSSL
    return -9;
#else

    close(con->server);
    SSL_CTX_free(con->ctx);
    free(con->host);
    con->connected = 0;
    return 0;
#endif
}

int fwbd_read_protocol_line(struct fwbd_connection *con, char *buf, int len)
{
#ifndef HAVE_LIBSSL
    return -9;
#else
    int l=0;
    while(len!=l)
    {
        int n;
        
        n = SSL_read(con->ssl, buf, 1);
        if(n<=0)
        {
            con->lasterr = last_ssl_err();
            return -1; /* err */
        }
        if(*buf=='\n')
        {
            *buf='\0';
            return l;
        }
        l++;
        buf++;
    }
    con->lasterr = strdup("Protocol reponse line too long.");
    return -2;
#endif
}

int fwbd_send_protocol_line(struct fwbd_connection *con, const char *s)
{
#ifndef HAVE_LIBSSL
    return -9;
#else
    int n=strlen(s);
    while(n)
    {
        int x = SSL_write(con->ssl, s, n);
        if(x<0)
        {
            con->lasterr = last_ssl_err();
            return -1;
        }
        n-=x;
        s+=x;
    }
    
    if(SSL_write(con->ssl, "\n", 1)!=1)
    {
        con->lasterr = last_ssl_err();
        return -2;
    }
    
    return 0;
#endif
}

char *fwbd_send_command(struct fwbd_connection *con, const char *cmd)
{
#ifndef HAVE_LIBSSL
    return NULL;
#else
    char *rsp;
    
    if(fwbd_send_protocol_line(con, cmd)!=0)
        return NULL;
    
    rsp = malloc(FWBD_MAX_PROTOCOL_LINE_LEN);
    if(fwbd_read_protocol_line(con, rsp, FWBD_MAX_PROTOCOL_LINE_LEN)<0)
    {
        free(rsp);
        return NULL;
    }
    
    return rsp;
#endif
}

char *fwbd_key2str(EVP_PKEY *skey)
{
#ifndef HAVE_LIBSSL
    return NULL;
#else
    char   *res;
    char   *ptr;
    BIO    *mem;
    size_t  l;

    mem = BIO_new(BIO_s_mem());
    if(!PEM_write_bio_PUBKEY(mem, skey))
    {
        BIO_set_close(mem, BIO_NOCLOSE);
        BIO_free(mem);
        return NULL;
    }
    
    l = BIO_get_mem_data(mem, &ptr);
    res=malloc(l+1);
    memcpy(res,ptr,l);
    res[l]='\0';
    
    BIO_set_close(mem, BIO_NOCLOSE);
    BIO_free(mem);

    return res;
#endif
}

#undef DONT_CHECK_CERTIFICATES
