/*
Studio: a simple GUI for the libfive CAD kernel
Copyright (C) 2017  Matt Keeter

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/
#pragma once

#include <QOpenGLWidget>
#include <QOpenGLFramebufferObject>

#include "studio/arrow.hpp"
#include "studio/axes.hpp"
#include "studio/background.hpp"
#include "studio/bbox.hpp"
#include "studio/busy.hpp"
#include "studio/camera.hpp"
#include "studio/shape.hpp"
#include "studio/settings.hpp"

#include "libfive/eval/eval_jacobian.hpp"

namespace Studio {

class View : public QOpenGLWidget, QOpenGLFunctions
{
    Q_OBJECT
public:
    View(QWidget* parent=nullptr);

    /*
     *  Requests that every active Shape cancels its render operation
     *  This is used when quitting to clean up threads quickly
     */
    void cancelShapes();

    /*
     *  In the destructor, free the OpenGL data associated with
     *  all children Shapes (because there could be shapes that
     *  aren't explicitly stored in the object, which will be freed
     *  by the QObject destructor, which runs after the OpenGL context
     *  is gone).
     */
    ~View();

public slots:
    void setShapes(QList<Shape*> shapes);
    void showAxes(bool a);
    void showBBox(bool b);

    void toOrthographic() { camera.toOrthographic();  }
    void toPerspective()  { camera.toPerspective();   }
    void toTurnZ() { camera.toTurnZ();  }
    void toTurnY()  { camera.toTurnY();   }
    void setLowRotSensitivity()  { camera.setRotationSensitivity(240); }
    void setMedRotSensitivity()  { camera.setRotationSensitivity(360); }
    void setHighRotSensitivity() { camera.setRotationSensitivity(720); }
    void setZoomCursorCentric() { zoom_cursor_centric = true; }
    void setZoomSceneCentric() { zoom_cursor_centric = false; }
    void zoomTo() { camera.zoomTo(settings.min, settings.max); }

    void toDCMeshing();
    void toIsoMeshing();
    void toHybridMeshing();

    /*
     *  Emits shapesReady if all the shapes being drawn
     *  are at their final (highest) resolution
     */
    void checkMeshes() const;

    /*
     *  Called when the script changes settings
     *  If first is true, then the camera zooms to the new bounds.
     */
    void onSettingsFromScript(Settings s, bool first);

signals:
    /*
     *  Called to kick off a render and start the busy spinner running
     */
    void startRender(Settings s);

    /*
     *  Emitted when all shapes are done rendering at their highest resolution
     */
    void meshesReady(QList<const libfive::Mesh*> shapes) const;

    /*
     *  Indicates when a drag operation begins and ends
     */
    void dragStart();
    void dragEnd();

    /*
     *  Emitted when a drag operation has changed variables
     */
    void varsDragged(QMap<libfive::Tree::Id, float> vs);

protected slots:
    void update() { QOpenGLWidget::update(); }
    void redrawPicker();

protected:
    void initializeGL() override;
    void paintGL() override;
    void resizeGL(int width, int height) override;

    void setAlgorithm(libfive::BRepAlgorithm alg);

    /*
     *  Converts from mouse event coordinates to model coordinates
     *  using the z depth from the pick buffer
     */
    QVector3D toModelPos(QPoint pt) const;

    /*
     *  Converts from mouse event coordinates to model coordinates
     *  using a user-provided z depth
     */
    QVector3D toModelPos(QPoint pt, float z) const;

    /*  Background items to render  */
    Arrow arrow;
    Axes axes;
    Background background;
    BBox bbox;
    Busy busy;

    void mouseMoveEvent(QMouseEvent *event) override;
    void mousePressEvent(QMouseEvent *event) override;
    void mouseReleaseEvent(QMouseEvent *event) override;
    void wheelEvent(QWheelEvent *event) override;

    /*
     *  Updates hover_target based on mouse cursor position
     */
    void checkHoverTarget(QPoint pos);

    /*
     *  Ensures that the pick buffer is synced, forcing a pick render
     *  if the timer is currently running.
     */
    void syncPicker();

    Camera camera;

    struct {
        enum { RELEASED, DRAG_ROT, DRAG_PAN, DRAG_EVAL } state = RELEASED;
        QPoint pos;
    } mouse;

    QList<Shape*> shapes;
    bool settings_enabled=true;
    Settings settings;
    libfive::BRepAlgorithm alg = libfive::BRepAlgorithm::DUAL_CONTOURING;

    bool show_axes=true;
    bool show_bbox=false;

    bool zoom_cursor_centric = true;

    /*  Framebuffer to render pick data  */
    QScopedPointer<QOpenGLFramebufferObject> pick_fbo;
    QTimer pick_timer;
    QImage pick_img;
    QVector<float> pick_depth;

    /*  Data to handle direct modification of shapes */
    QVector3D drag_start;
    QVector3D drag_dir;
    std::pair<std::unique_ptr<libfive::JacobianEvaluator>,
              std::shared_ptr<libfive::Tape>> drag_eval;
    Shape* drag_target=nullptr;
    bool drag_valid=false;
    Shape* hover_target=nullptr;

    QVector3D cursor_pos;
    bool cursor_pos_valid=false;

    /*  Set to true on the first draw, if the OpenGL version is new enough */
    bool gl_checked=false;
};

}   // namespace Studio
