#include "gen_amd64_regalloc_if.h"
#include "irverify_t.h"
#include "fourcc.h"


ir_op *op_amd64_Add = NULL;
ir_op *op_amd64_Store = NULL;
ir_op *op_amd64_FrameAddr = NULL;
ir_op *op_amd64_Jcc = NULL;
ir_op *op_amd64_Immediate = NULL;
ir_op *op_amd64_Push = NULL;
ir_op *op_amd64_Neg = NULL;
ir_op *op_amd64_Cmp = NULL;
ir_op *op_amd64_SymConst = NULL;
ir_op *op_amd64_Load = NULL;
ir_op *op_amd64_Sub = NULL;
ir_op *op_amd64_Jmp = NULL;
ir_op *op_amd64_Mul = NULL;
ir_op *op_amd64_Conv = NULL;

ir_op *get_op_amd64_Add(void)         { return op_amd64_Add; }
int    is_amd64_Add(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Add; }

ir_op *get_op_amd64_Store(void)         { return op_amd64_Store; }
int    is_amd64_Store(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Store; }

ir_op *get_op_amd64_FrameAddr(void)         { return op_amd64_FrameAddr; }
int    is_amd64_FrameAddr(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_FrameAddr; }

ir_op *get_op_amd64_Jcc(void)         { return op_amd64_Jcc; }
int    is_amd64_Jcc(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Jcc; }

ir_op *get_op_amd64_Immediate(void)         { return op_amd64_Immediate; }
int    is_amd64_Immediate(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Immediate; }

ir_op *get_op_amd64_Push(void)         { return op_amd64_Push; }
int    is_amd64_Push(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Push; }

ir_op *get_op_amd64_Neg(void)         { return op_amd64_Neg; }
int    is_amd64_Neg(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Neg; }

ir_op *get_op_amd64_Cmp(void)         { return op_amd64_Cmp; }
int    is_amd64_Cmp(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Cmp; }

ir_op *get_op_amd64_SymConst(void)         { return op_amd64_SymConst; }
int    is_amd64_SymConst(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_SymConst; }

ir_op *get_op_amd64_Load(void)         { return op_amd64_Load; }
int    is_amd64_Load(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Load; }

ir_op *get_op_amd64_Sub(void)         { return op_amd64_Sub; }
int    is_amd64_Sub(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Sub; }

ir_op *get_op_amd64_Jmp(void)         { return op_amd64_Jmp; }
int    is_amd64_Jmp(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Jmp; }

ir_op *get_op_amd64_Mul(void)         { return op_amd64_Mul; }
int    is_amd64_Mul(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Mul; }

ir_op *get_op_amd64_Conv(void)         { return op_amd64_Conv; }
int    is_amd64_Conv(const ir_node *n) { return get_amd64_irn_opcode(n) == iro_amd64_Conv; }



static int amd64_opcode_start = -1;
static int amd64_opcode_end   = -1;


/** A tag for the amd64 opcodes. Note that the address is used as a tag value, NOT the FOURCC code. */
#define amd64_op_tag FOURCC('A', 'M', 'D', '6')

/** Return the opcode number of the first amd64 opcode. */
int get_amd64_opcode_first(void) {
	return amd64_opcode_start;
}

/** Return the opcode number of the last amd64 opcode + 1. */
int get_amd64_opcode_last(void) {
	return amd64_opcode_end;
}

/** Return 1 if the given opcode is a amd64 machine op, 0 otherwise */
int is_amd64_op(const ir_op *op) {
	return get_op_tag(op) == amd64_op_tag;
}

/** Return 1 if the given node is a amd64 machine node, 0 otherwise */
int is_amd64_irn(const ir_node *node) {
	return is_amd64_op(get_irn_op(node));
}

int get_amd64_irn_opcode(const ir_node *node) {
	if (is_amd64_irn(node))
		return get_irn_opcode(node) - amd64_opcode_start;
	return -1;
}

#ifdef BIT
#undef BIT
#endif
#define BIT(x)  (1 << (x))

static const unsigned amd64_limit_flags_eflags[] = { BIT(REG_FLAGS_EFLAGS) };
static const unsigned amd64_limit_gp_rsp[] = { BIT(REG_GP_RSP), 0 };
static const unsigned amd64_limit_gp_rax[] = { BIT(REG_GP_RAX), 0 };
static const unsigned amd64_limit_gp_rax_rdx[] = { BIT(REG_GP_RAX) | BIT(REG_GP_RDX), 0 };

static const arch_register_req_t amd64_requirements_gp_gp = {
	arch_register_req_type_normal,
	& amd64_reg_classes[CLASS_amd64_gp],
	NULL,        /* limit bitset */
	0,           /* same pos */
	0,           /* different pos */
	1            /* width */
};


static const arch_register_req_t amd64_requirements__none = {
	arch_register_req_type_none,
	NULL,                         /* regclass */
	NULL,                         /* limit bitset */
	0,                            /* same pos */
	0,                            /* different pos */
	0                             /* width */
};


static const arch_register_req_t amd64_requirements_flags_eflags = {
	arch_register_req_type_limited,
	& amd64_reg_classes[CLASS_amd64_flags],
	amd64_limit_flags_eflags,
	0,        /* same pos */
	0,       /* different pos */
	1             /* width */
};


static const arch_register_req_t amd64_requirements_gp_rsp_I_S = {
	arch_register_req_type_ignore | arch_register_req_type_produces_sp | arch_register_req_type_limited,
	& amd64_reg_classes[CLASS_amd64_gp],
	amd64_limit_gp_rsp,
	0,        /* same pos */
	0,       /* different pos */
	1             /* width */
};


static const arch_register_req_t amd64_requirements_gp_rsp = {
	arch_register_req_type_limited,
	& amd64_reg_classes[CLASS_amd64_gp],
	amd64_limit_gp_rsp,
	0,        /* same pos */
	0,       /* different pos */
	1             /* width */
};


static const arch_register_req_t amd64_requirements_gp_in_r1 = {
	arch_register_req_type_should_be_same,
	& amd64_reg_classes[CLASS_amd64_gp],
	NULL,
	1,        /* same pos */
	0,       /* different pos */
	1             /* width */
};


static const arch_register_req_t amd64_requirements_flags_flags = {
	arch_register_req_type_normal,
	& amd64_reg_classes[CLASS_amd64_flags],
	NULL,        /* limit bitset */
	0,           /* same pos */
	0,           /* different pos */
	1            /* width */
};


static const arch_register_req_t amd64_requirements_gp_rax = {
	arch_register_req_type_limited,
	& amd64_reg_classes[CLASS_amd64_gp],
	amd64_limit_gp_rax,
	0,        /* same pos */
	0,       /* different pos */
	1             /* width */
};


static const arch_register_req_t amd64_requirements_gp_rax_rdx = {
	arch_register_req_type_limited,
	& amd64_reg_classes[CLASS_amd64_gp],
	amd64_limit_gp_rax_rdx,
	0,        /* same pos */
	0,       /* different pos */
	1             /* width */
};



/**
 * construct Add node
 */
ir_node *new_bd_amd64_Add(dbg_info *dbgi, ir_node *block, ir_node *left, ir_node *right)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Add;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 2;
	ir_node        *in[2];
	int             n_res   = 1;
	ir_mode        *mode    = mode_Lu;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
		& amd64_requirements_gp_gp,
	};

	/* construct in array */
	in[0] = left;
	in[1] = right;

	/* flags */
	irn_flags_ |= arch_irn_flags_rematerializable;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
		arch_add_irn_flags(res, arch_irn_flags_modify_flags);

	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_gp;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Store node
 */
ir_node *new_bd_amd64_Store(dbg_info *dbgi, ir_node *block, ir_node *ptr, ir_node *val, ir_node *mem, ir_entity *entity)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Store;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 3;
	ir_node        *in[3];
	int             n_res   = 1;
	ir_mode        *mode    = mode_M;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
		& amd64_requirements_gp_gp,
		& amd64_requirements__none,
	};

	/* construct in array */
	in[0] = ptr;
	in[1] = val;
	in[2] = mem;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);	init_amd64_SymConst_attributes(res, entity);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements__none;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct FrameAddr node
 */
ir_node *new_bd_amd64_FrameAddr(dbg_info *dbgi, ir_node *block, ir_node *base, ir_entity *entity)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_FrameAddr;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 1;
	ir_node        *in[1];
	int             n_res   = 1;
	ir_mode        *mode    = mode_Lu;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
	};

	/* construct in array */
	in[0] = base;

	/* flags */
	irn_flags_ |= arch_irn_flags_rematerializable;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);	init_amd64_SymConst_attributes(res, entity);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_gp;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Jcc node
 */
ir_node *new_bd_amd64_Jcc(dbg_info *dbgi, ir_node *block, ir_node *eflags, ir_relation relation)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Jcc;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 1;
	ir_node        *in[1];
	int             n_res   = 2;
	ir_mode        *mode    = mode_T;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_flags_eflags,
	};
	amd64_attr_t *attr;

	/* construct in array */
	in[0] = eflags;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements__none;
	info->out_infos[1].req = &amd64_requirements__none;


	attr = (amd64_attr_t*)get_irn_generic_attr(res);
	(void) attr; /* avoid potential warning */
	attr->ext.relation = relation;
	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Immediate node
 */
ir_node *new_bd_amd64_Immediate(dbg_info *dbgi, ir_node *block, unsigned imm_value)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Immediate;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 0;
	ir_node       **in      = NULL;
	int             n_res   = 1;
	ir_mode        *mode    = mode_Lu;
	static const arch_register_req_t **in_reqs = NULL;
	amd64_attr_t *attr;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_gp;


	attr = (amd64_attr_t*)get_irn_generic_attr(res);
	(void) attr; /* avoid potential warning */
	attr->ext.imm_value = imm_value;
	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Push node
 */
ir_node *new_bd_amd64_Push(dbg_info *dbgi, ir_node *block, ir_node *base, ir_node *index, ir_node *mem, ir_node *val, ir_node *stack)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Push;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 5;
	ir_node        *in[5];
	int             n_res   = 2;
	ir_mode        *mode    = mode_T;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
		& amd64_requirements_gp_gp,
		& amd64_requirements__none,
		& amd64_requirements_gp_gp,
		& amd64_requirements_gp_rsp,
	};

	/* construct in array */
	in[0] = base;
	in[1] = index;
	in[2] = mem;
	in[3] = val;
	in[4] = stack;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_rsp_I_S;
	info->out_infos[1].req = &amd64_requirements__none;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Neg node
 */
ir_node *new_bd_amd64_Neg(dbg_info *dbgi, ir_node *block, ir_node *val)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Neg;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 1;
	ir_node        *in[1];
	int             n_res   = 2;
	ir_mode        *mode    = mode_Lu;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
	};

	/* construct in array */
	in[0] = val;

	/* flags */
	irn_flags_ |= arch_irn_flags_rematerializable;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_in_r1;
	info->out_infos[1].req = &amd64_requirements_flags_flags;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Cmp node
 */
ir_node *new_bd_amd64_Cmp(dbg_info *dbgi, ir_node *block, ir_node *left, ir_node *right, int ins_permuted, int cmp_unsigned)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Cmp;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 2;
	ir_node        *in[2];
	int             n_res   = 1;
	ir_mode        *mode    = mode_Iu;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
		& amd64_requirements_gp_gp,
	};
	amd64_attr_t *attr;

	/* construct in array */
	in[0] = left;
	in[1] = right;

	/* flags */
	irn_flags_ |= arch_irn_flags_rematerializable;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
		arch_add_irn_flags(res, arch_irn_flags_modify_flags);

	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_flags_flags;


	attr = (amd64_attr_t*)get_irn_generic_attr(res);
	(void) attr; /* avoid potential warning */
	attr->data.ins_permuted   = ins_permuted;
	attr->data.cmp_unsigned = cmp_unsigned;

	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct SymConst node
 */
ir_node *new_bd_amd64_SymConst(dbg_info *dbgi, ir_node *block, ir_entity *entity)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_SymConst;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 0;
	ir_node       **in      = NULL;
	int             n_res   = 1;
	ir_mode        *mode    = mode_Lu;
	static const arch_register_req_t **in_reqs = NULL;

	/* flags */
	irn_flags_ |= arch_irn_flags_rematerializable;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);	init_amd64_SymConst_attributes(res, entity);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_gp;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Load node
 */
ir_node *new_bd_amd64_Load(dbg_info *dbgi, ir_node *block, ir_node *ptr, ir_node *mem, ir_entity *entity)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Load;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 2;
	ir_node        *in[2];
	int             n_res   = 2;
	ir_mode        *mode    = mode_T;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
		& amd64_requirements__none,
	};

	/* construct in array */
	in[0] = ptr;
	in[1] = mem;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);	init_amd64_SymConst_attributes(res, entity);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_gp;
	info->out_infos[1].req = &amd64_requirements__none;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Sub node
 */
ir_node *new_bd_amd64_Sub(dbg_info *dbgi, ir_node *block, ir_node *left, ir_node *right)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Sub;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 2;
	ir_node        *in[2];
	int             n_res   = 1;
	ir_mode        *mode    = mode_Lu;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
		& amd64_requirements_gp_gp,
	};

	/* construct in array */
	in[0] = left;
	in[1] = right;

	/* flags */
	irn_flags_ |= arch_irn_flags_rematerializable;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
		arch_add_irn_flags(res, arch_irn_flags_modify_flags);

	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_gp;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Jmp node
 */
ir_node *new_bd_amd64_Jmp(dbg_info *dbgi, ir_node *block)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Jmp;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 0;
	ir_node       **in      = NULL;
	int             n_res   = 1;
	ir_mode        *mode    = mode_X;
	static const arch_register_req_t **in_reqs = NULL;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements__none;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Mul node
 */
ir_node *new_bd_amd64_Mul(dbg_info *dbgi, ir_node *block, ir_node *left, ir_node *right)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Mul;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 2;
	ir_node        *in[2];
	int             n_res   = 1;
	ir_mode        *mode    = mode_Lu;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_rax,
		& amd64_requirements_gp_gp,
	};

	/* construct in array */
	in[0] = left;
	in[1] = right;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_rax_rdx;


	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}

/**
 * construct Conv node
 */
ir_node *new_bd_amd64_Conv(dbg_info *dbgi, ir_node *block, ir_node *val, ir_mode *smaller_mode)
{
	ir_graph         *irg        = get_irn_irg(block);
	ir_op            *op         = op_amd64_Conv;
	arch_irn_flags_t  irn_flags_ = arch_irn_flags_none;
	ir_node          *res;
	backend_info_t   *info;
	int             arity   = 1;
	ir_node        *in[1];
	int             n_res   = 1;
	ir_mode        *mode    = mode_Lu;
	static const arch_register_req_t *in_reqs[] =
	{
		& amd64_requirements_gp_gp,
	};
	amd64_attr_t *attr;

	/* construct in array */
	in[0] = val;

	/* create node */
	assert(op != NULL);
	res = new_ir_node(dbgi, irg, block, op, mode, arity, in);

	/* init node attributes */
		init_amd64_attributes(res, irn_flags_, in_reqs, n_res);
	
	info = be_get_info(res);
	(void) info; /* avoid potential warning */
	info->out_infos[0].req = &amd64_requirements_gp_gp;


	attr = (amd64_attr_t*)get_irn_generic_attr(res);
	(void) attr; /* avoid potential warning */
	attr->ls_mode = smaller_mode;
	/* optimize node */
	res = optimize_node(res);
	irn_verify_irg(res, irg);

	return res;
}



/**
 * Creates the amd64 specific Firm machine operations
 * needed for the assembler irgs.
 */
void amd64_create_opcodes(const arch_irn_ops_t *be_ops)
{
	ir_op *op;
	int    cur_opcode = get_next_ir_opcodes(iro_amd64_last);

	amd64_opcode_start = cur_opcode;
	op = new_ir_op(cur_opcode + iro_amd64_Add, "amd64_Add", op_pin_state_exc_pinned, irop_flag_commutative, oparity_binary, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Add = op;
	op = new_ir_op(cur_opcode + iro_amd64_Store, "amd64_Store", op_pin_state_exc_pinned, irop_flag_uses_memory, oparity_trinary, 0, sizeof(amd64_SymConst_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr_SymConst;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Store = op;
	op = new_ir_op(cur_opcode + iro_amd64_FrameAddr, "amd64_FrameAddr", op_pin_state_floats, irop_flag_constlike, oparity_unary, 0, sizeof(amd64_SymConst_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr_SymConst;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_FrameAddr = op;
	op = new_ir_op(cur_opcode + iro_amd64_Jcc, "amd64_Jcc", op_pin_state_pinned, irop_flag_cfopcode|irop_flag_forking, oparity_unary, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Jcc = op;
	op = new_ir_op(cur_opcode + iro_amd64_Immediate, "amd64_Immediate", op_pin_state_floats, irop_flag_constlike, oparity_zero, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Immediate = op;
	op = new_ir_op(cur_opcode + iro_amd64_Push, "amd64_Push", op_pin_state_exc_pinned, irop_flag_none, oparity_any, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Push = op;
	op = new_ir_op(cur_opcode + iro_amd64_Neg, "amd64_Neg", op_pin_state_floats, irop_flag_none, oparity_unary, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Neg = op;
	op = new_ir_op(cur_opcode + iro_amd64_Cmp, "amd64_Cmp", op_pin_state_exc_pinned, irop_flag_none, oparity_binary, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Cmp = op;
	op = new_ir_op(cur_opcode + iro_amd64_SymConst, "amd64_SymConst", op_pin_state_floats, irop_flag_constlike, oparity_zero, 0, sizeof(amd64_SymConst_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr_SymConst;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_SymConst = op;
	op = new_ir_op(cur_opcode + iro_amd64_Load, "amd64_Load", op_pin_state_exc_pinned, irop_flag_uses_memory, oparity_binary, 0, sizeof(amd64_SymConst_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr_SymConst;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Load = op;
	op = new_ir_op(cur_opcode + iro_amd64_Sub, "amd64_Sub", op_pin_state_exc_pinned, irop_flag_none, oparity_binary, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Sub = op;
	op = new_ir_op(cur_opcode + iro_amd64_Jmp, "amd64_Jmp", op_pin_state_pinned, irop_flag_cfopcode, oparity_zero, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Jmp = op;
	op = new_ir_op(cur_opcode + iro_amd64_Mul, "amd64_Mul", op_pin_state_exc_pinned, irop_flag_none, oparity_binary, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Mul = op;
	op = new_ir_op(cur_opcode + iro_amd64_Conv, "amd64_Conv", op_pin_state_exc_pinned, irop_flag_none, oparity_unary, 0, sizeof(amd64_attr_t));
	op->ops.be_ops        = be_ops;
	op->ops.dump_node     = amd64_dump_node;
	op->ops.node_cmp_attr = cmp_amd64_attr;
	op->ops.copy_attr     = amd64_copy_attr;
	set_op_tag(op, amd64_op_tag);
	op_amd64_Conv = op;

	amd64_opcode_end = cur_opcode + iro_amd64_last;
}

void amd64_free_opcodes(void)
{
	free_ir_op(op_amd64_Add); op_amd64_Add = NULL;
	free_ir_op(op_amd64_Store); op_amd64_Store = NULL;
	free_ir_op(op_amd64_FrameAddr); op_amd64_FrameAddr = NULL;
	free_ir_op(op_amd64_Jcc); op_amd64_Jcc = NULL;
	free_ir_op(op_amd64_Immediate); op_amd64_Immediate = NULL;
	free_ir_op(op_amd64_Push); op_amd64_Push = NULL;
	free_ir_op(op_amd64_Neg); op_amd64_Neg = NULL;
	free_ir_op(op_amd64_Cmp); op_amd64_Cmp = NULL;
	free_ir_op(op_amd64_SymConst); op_amd64_SymConst = NULL;
	free_ir_op(op_amd64_Load); op_amd64_Load = NULL;
	free_ir_op(op_amd64_Sub); op_amd64_Sub = NULL;
	free_ir_op(op_amd64_Jmp); op_amd64_Jmp = NULL;
	free_ir_op(op_amd64_Mul); op_amd64_Mul = NULL;
	free_ir_op(op_amd64_Conv); op_amd64_Conv = NULL;
}
