/*
 * Copyright (c) 2000-2008 Apple Inc. All rights reserved.
 *
 * @APPLE_OSREFERENCE_LICENSE_HEADER_START@
 * 
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. The rights granted to you under the License
 * may not be used to create, or enable the creation or redistribution of,
 * unlawful or unlicensed copies of an Apple operating system, or to
 * circumvent, violate, or enable the circumvention or violation of, any
 * terms of an Apple operating system software license agreement.
 * 
 * Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 * 
 * @APPLE_OSREFERENCE_LICENSE_HEADER_END@
 */
/*
 * Copyright (c) 1990, 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from the Stanford/CMU enet packet filter,
 * (net/enet.c) distributed as part of 4.3BSD, and code contributed
 * to Berkeley by Steven McCanne and Van Jacobson both of Lawrence
 * Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      @(#)bpf.h	8.1 (Berkeley) 6/10/93
 *	@(#)bpf.h	1.34 (LBL)     6/16/96
 *
 * $FreeBSD: src/sys/net/bpf.h,v 1.21.2.3 2001/08/01 00:23:13 fenner Exp $
 */
/*
 * NOTICE: This file was modified by SPARTA, Inc. in 2006 to introduce
 * support for mandatory and extensible security protections.  This notice
 * is included in support of clause 2.2 (b) of the Apple Public License,
 * Version 2.0.
 */

#ifndef _NET_BPF_H_
#define _NET_BPF_H_
#include <sys/appleapiopts.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/cdefs.h>

#ifdef KERNEL
#include <sys/kernel_types.h>
#endif

/* BSD style release date */
#define	BPF_RELEASE 199606

typedef	int32_t	  bpf_int32;
typedef	u_int32_t bpf_u_int32;

/*
 * Alignment macros.  BPF_WORDALIGN rounds up to the next
 * even multiple of BPF_ALIGNMENT.
 */
#define BPF_ALIGNMENT sizeof(int32_t)
#define BPF_WORDALIGN(x) (((x)+(BPF_ALIGNMENT-1))&~(BPF_ALIGNMENT-1))

#define BPF_MAXINSNS 512
#define BPF_MAXBUFSIZE 0x80000
#define BPF_MINBUFSIZE 32

/*
 *  Structure for BIOCSETF.
 */
struct bpf_program {
	u_int bf_len;
	struct bpf_insn *bf_insns;
};

#ifdef KERNEL_PRIVATE
/* LP64 version of bpf_program.  all pointers 
 * grow when we're dealing with a 64-bit process.
 * WARNING - keep in sync with bpf_program
 */
struct bpf_program64 {
	u_int		bf_len;
	user64_addr_t	bf_insns __attribute__((aligned(8)));
};

struct bpf_program32 {
	u_int		bf_len;
	user32_addr_t	bf_insns;
};
#endif /* KERNEL_PRIVATE */

/*
 * Struct returned by BIOCGSTATS.
 */
struct bpf_stat {
	u_int bs_recv;		/* number of packets received */
	u_int bs_drop;		/* number of packets dropped */
};

/*
 * Struct return by BIOCVERSION.  This represents the version number of
 * the filter language described by the instruction encodings below.
 * bpf understands a program iff kernel_major == filter_major &&
 * kernel_minor >= filter_minor, that is, if the value returned by the
 * running kernel has the same major number and a minor number equal
 * equal to or less than the filter being downloaded.  Otherwise, the
 * results are undefined, meaning an error may be returned or packets
 * may be accepted haphazardly.
 * It has nothing to do with the source code version.
 */
struct bpf_version {
	u_short bv_major;
	u_short bv_minor;
};
#if defined(__LP64__)
#define __need_struct_timeval32
#include <sys/_structs.h>
#define BPF_TIMEVAL timeval32
#else
#define BPF_TIMEVAL timeval
#endif /* __LP64__ */
/* Current version number of filter architecture. */
#define BPF_MAJOR_VERSION 1
#define BPF_MINOR_VERSION 1

#define	BIOCGBLEN	_IOR('B',102, u_int)
#define	BIOCSBLEN	_IOWR('B',102, u_int)
#define	BIOCSETF	_IOW('B',103, struct bpf_program)
#ifdef KERNEL_PRIVATE
#define	BIOCSETF64	_IOW('B',103, struct bpf_program64)
#define	BIOCSETF32	_IOW('B',103, struct bpf_program32)
#endif /* KERNEL_PRIVATE */
#define	BIOCFLUSH	_IO('B',104)
#define BIOCPROMISC	_IO('B',105)
#define	BIOCGDLT	_IOR('B',106, u_int)
#define BIOCGETIF	_IOR('B',107, struct ifreq)
#define BIOCSETIF	_IOW('B',108, struct ifreq)
#define BIOCSRTIMEOUT	_IOW('B',109, struct BPF_TIMEVAL)
#define BIOCGRTIMEOUT	_IOR('B',110, struct BPF_TIMEVAL)
#define BIOCGSTATS	_IOR('B',111, struct bpf_stat)
#define BIOCIMMEDIATE	_IOW('B',112, u_int)
#define BIOCVERSION	_IOR('B',113, struct bpf_version)
#define BIOCGRSIG	_IOR('B',114, u_int)
#define BIOCSRSIG	_IOW('B',115, u_int)
#define BIOCGHDRCMPLT	_IOR('B',116, u_int)
#define BIOCSHDRCMPLT	_IOW('B',117, u_int)
#define BIOCGSEESENT	_IOR('B',118, u_int)
#define BIOCSSEESENT	_IOW('B',119, u_int)
#define BIOCSDLT        _IOW('B',120, u_int)
#define BIOCGDLTLIST    _IOWR('B',121, struct bpf_dltlist)

/*
 * Structure prepended to each packet.
 */
struct bpf_hdr {
	struct BPF_TIMEVAL bh_tstamp;	/* time stamp */
	bpf_u_int32	bh_caplen;	/* length of captured portion */
	bpf_u_int32	bh_datalen;	/* original length of packet */
	u_short		bh_hdrlen;	/* length of bpf header (this struct
					   plus alignment padding) */
};
/*
 * Because the structure above is not a multiple of 4 bytes, some compilers
 * will insist on inserting padding; hence, sizeof(struct bpf_hdr) won't work.
 * Only the kernel needs to know about it; applications use bh_hdrlen.
 */
#ifdef KERNEL
#define	SIZEOF_BPF_HDR	(sizeof(struct bpf_hdr) <= 20 ? 18 : \
    sizeof(struct bpf_hdr))
#endif

/*
 * Data-link level type codes.
 */
#define DLT_NULL	0	/* no link-layer encapsulation */
#define DLT_EN10MB	1	/* Ethernet (10Mb) */
#define DLT_EN3MB	2	/* Experimental Ethernet (3Mb) */
#define DLT_AX25	3	/* Amateur Radio AX.25 */
#define DLT_PRONET	4	/* Proteon ProNET Token Ring */
#define DLT_CHAOS	5	/* Chaos */
#define DLT_IEEE802	6	/* IEEE 802 Networks */
#define DLT_ARCNET	7	/* ARCNET */
#define DLT_SLIP	8	/* Serial Line IP */
#define DLT_PPP		9	/* Point-to-point Protocol */
#define DLT_FDDI	10	/* FDDI */
#define DLT_ATM_RFC1483	11	/* LLC/SNAP encapsulated atm */
#define DLT_RAW		12	/* raw IP */

/*
 * These are values from BSD/OS's "bpf.h".
 * These are not the same as the values from the traditional libpcap
 * "bpf.h"; however, these values shouldn't be generated by any
 * OS other than BSD/OS, so the correct values to use here are the
 * BSD/OS values.
 *
 * Platforms that have already assigned these values to other
 * DLT_ codes, however, should give these codes the values
 * from that platform, so that programs that use these codes will
 * continue to compile - even though they won't correctly read
 * files of these types.
 */
#define DLT_SLIP_BSDOS	15	/* BSD/OS Serial Line IP */
#define DLT_PPP_BSDOS	16	/* BSD/OS Point-to-point Protocol */

#define	DLT_PFSYNC	18	/* Packet filter state syncing */
#define DLT_ATM_CLIP	19	/* Linux Classical-IP over ATM */

/*
 * This value is defined by NetBSD; other platforms should refrain from
 * using it for other purposes, so that NetBSD savefiles with a link
 * type of 50 can be read as this type on all platforms.
 */
#define DLT_PPP_SERIAL	50	/* PPP over serial with HDLC encapsulation */

/*
 * This value was defined by libpcap 0.5; platforms that have defined
 * it with a different value should define it here with that value -
 * a link type of 104 in a save file will be mapped to DLT_C_HDLC,
 * whatever value that happens to be, so programs will correctly
 * handle files with that link type regardless of the value of
 * DLT_C_HDLC.
 *
 * The name DLT_C_HDLC was used by BSD/OS; we use that name for source
 * compatibility with programs written for BSD/OS.
 *
 * libpcap 0.5 defined it as DLT_CHDLC; we define DLT_CHDLC as well,
 * for source compatibility with programs written for libpcap 0.5.
 */
#define DLT_C_HDLC	104	/* Cisco HDLC */
#define DLT_CHDLC	DLT_C_HDLC

/*
 * Reserved for future use.
 * Do not pick other numerical value for these unless you have also
 * picked up the tcpdump.org top-of-CVS-tree version of "savefile.c",
 * which will arrange that capture files for these DLT_ types have
 * the same "network" value on all platforms, regardless of what
 * value is chosen for their DLT_ type (thus allowing captures made
 * on one platform to be read on other platforms, even if the two
 * platforms don't use the same numerical values for all DLT_ types).
 */
#define DLT_IEEE802_11	105	/* IEEE 802.11 wireless */

/*
 * Values between 106 and 107 are used in capture file headers as
 * link-layer types corresponding to DLT_ types that might differ
 * between platforms; don't use those values for new DLT_ new types.
 */

/*
 * OpenBSD DLT_LOOP, for loopback devices; it's like DLT_NULL, except
 * that the AF_ type in the link-layer header is in network byte order.
 *
 * OpenBSD defines it as 12, but that collides with DLT_RAW, so we
 * define it as 108 here.  If OpenBSD picks up this file, it should
 * define DLT_LOOP as 12 in its version, as per the comment above -
 * and should not use 108 for any purpose.
 */
#define DLT_LOOP	108

/*
 * Values between 109 and 112 are used in capture file headers as
 * link-layer types corresponding to DLT_ types that might differ
 * between platforms; don't use those values for new DLT_ new types.
 */

/*
 * This is for Linux cooked sockets.
 */
#define DLT_LINUX_SLL	113

/*
 * For use in capture-file headers as a link-layer type corresponding
 * to OpenBSD PF (Packet Filter) log.
 */
#define	DLT_PFLOG	117

/*
 * BSD header for 802.11 plus a number of bits of link-layer information
 * including radio information.
 */
#ifndef DLT_IEEE802_11_RADIO
#define DLT_IEEE802_11_RADIO	127
#endif

/*
 * Apple IP-over-IEEE 1394, as per a request from Dieter Siegmund
 * <dieter@apple.com>.  The header that's presented is an Ethernet-like
 * header:
 *
 *	#define FIREWIRE_EUI64_LEN	8
 *	struct firewire_header {
 *		u_char  firewire_dhost[FIREWIRE_EUI64_LEN];
 *		u_char  firewire_shost[FIREWIRE_EUI64_LEN];
 *		u_short firewire_type;
 *	};
 *
 * with "firewire_type" being an Ethernet type value, rather than,
 * for example, raw GASP frames being handed up.
 */
#define DLT_APPLE_IP_OVER_IEEE1394	138

/*
 * For future use with 802.11 captures - defined by AbsoluteValue
 * Systems to store a number of bits of link-layer information
 * including radio information:
 *
 *	http://www.shaftnet.org/~pizza/software/capturefrm.txt
 *
 * but it might be used by some non-AVS drivers now or in the
 * future.
 */
#define DLT_IEEE802_11_RADIO_AVS 163	/* 802.11 plus AVS radio header */


/*
 * The instruction encodings.
 */
/* instruction classes */
#define BPF_CLASS(code) ((code) & 0x07)
#define		BPF_LD		0x00
#define		BPF_LDX		0x01
#define		BPF_ST		0x02
#define		BPF_STX		0x03
#define		BPF_ALU		0x04
#define		BPF_JMP		0x05
#define		BPF_RET		0x06
#define		BPF_MISC	0x07

/* ld/ldx fields */
#define BPF_SIZE(code)	((code) & 0x18)
#define		BPF_W		0x00
#define		BPF_H		0x08
#define		BPF_B		0x10
#define BPF_MODE(code)	((code) & 0xe0)
#define		BPF_IMM 	0x00
#define		BPF_ABS		0x20
#define		BPF_IND		0x40
#define		BPF_MEM		0x60
#define		BPF_LEN		0x80
#define		BPF_MSH		0xa0

/* alu/jmp fields */
#define BPF_OP(code)	((code) & 0xf0)
#define		BPF_ADD		0x00
#define		BPF_SUB		0x10
#define		BPF_MUL		0x20
#define		BPF_DIV		0x30
#define		BPF_OR		0x40
#define		BPF_AND		0x50
#define		BPF_LSH		0x60
#define		BPF_RSH		0x70
#define		BPF_NEG		0x80
#define		BPF_JA		0x00
#define		BPF_JEQ		0x10
#define		BPF_JGT		0x20
#define		BPF_JGE		0x30
#define		BPF_JSET	0x40
#define BPF_SRC(code)	((code) & 0x08)
#define		BPF_K		0x00
#define		BPF_X		0x08

/* ret - BPF_K and BPF_X also apply */
#define BPF_RVAL(code)	((code) & 0x18)
#define		BPF_A		0x10

/* misc */
#define BPF_MISCOP(code) ((code) & 0xf8)
#define		BPF_TAX		0x00
#define		BPF_TXA		0x80

/*
 * The instruction data structure.
 */
struct bpf_insn {
	u_short		code;
	u_char		jt;
	u_char		jf;
	bpf_u_int32	k;
};

/*
 * Macros for insn array initializers.
 */
#define BPF_STMT(code, k) { (u_short)(code), 0, 0, k }
#define BPF_JUMP(code, k, jt, jf) { (u_short)(code), jt, jf, k }

#pragma pack(4)

/*
 * Structure to retrieve available DLTs for the interface.
 */
struct bpf_dltlist {
	u_int32_t		bfl_len;	/* number of bfd_list array */
	union {
		u_int32_t	*bflu_list;	/* array of DLTs */
		u_int64_t	bflu_pad;
	} bfl_u;
};
#define bfl_list bfl_u.bflu_list

#pragma pack()

#ifdef KERNEL_PRIVATE
/* Forward declerations */
struct ifnet;
struct mbuf;

extern int	bpf_validate(const struct bpf_insn *, int);
extern void	bpfdetach(struct ifnet *);
extern void	bpfilterattach(int);
extern u_int	bpf_filter(const struct bpf_insn *, u_char *, u_int, u_int);
#endif /* KERNEL_PRIVATE */

#ifdef KERNEL
#ifndef BPF_TAP_MODE_T
#define BPF_TAP_MODE_T
/*!
	@enum BPF tap mode
	@abstract Constants defining interface families.
	@constant BPF_MODE_DISABLED Disable bpf.
	@constant BPF_MODE_INPUT Enable input only.
	@constant BPF_MODE_OUTPUT Enable output only.
	@constant BPF_MODE_INPUT_OUTPUT Enable input and output.
*/

enum {
	BPF_MODE_DISABLED	= 0,
	BPF_MODE_INPUT		= 1,
	BPF_MODE_OUTPUT		= 2,
	BPF_MODE_INPUT_OUTPUT	= 3
};
/*!
	@typedef bpf_tap_mode
	@abstract Mode for tapping. BPF_MODE_DISABLED/BPF_MODE_INPUT_OUTPUT etc.
*/
typedef u_int32_t bpf_tap_mode;
#endif /* !BPF_TAP_MODE_T */

/*!
	@typedef bpf_send_func
	@discussion bpf_send_func is called when a bpf file descriptor is
		used to send a raw packet on the interface. The mbuf and data
		link type are specified. The callback is responsible for
		releasing the mbuf whether or not it returns an error.
	@param interface The interface the packet is being sent on.
	@param dlt The data link type the bpf device is attached to.
	@param packet The packet to be sent.
 */
typedef errno_t (*bpf_send_func)(ifnet_t interface, u_int32_t data_link_type,
								 mbuf_t packet);

/*!
	@typedef bpf_tap_func
	@discussion bpf_tap_func is called when the tap state of the
		interface changes. This happens when a bpf device attaches to an
		interface or detaches from an interface. The tap mode will join
		together (bit or) the modes of all bpf devices using that
		interface for that dlt. If you return an error from this
		function, the bpf device attach attempt that triggered the tap
		will fail. If this function was called bacuse the tap state was
		decreasing (tap in or out is stopping), the error will be
		ignored.
	@param interface The interface being tapped.
	@param dlt The data link type being tapped.
	@param direction The direction of the tap.
 */
typedef errno_t (*bpf_tap_func)(ifnet_t interface, u_int32_t data_link_type,
								bpf_tap_mode direction);

/*!
	@function bpfattach
	@discussion Registers an interface with BPF. This allows bpf devices
		to attach to your interface to capture packets. Your interface
		will be unregistered automatically when your interface is
		detached.
	@param interface The interface to register with BPF.
	@param data_link_type The data link type of the interface. See the
		DLT_* defines in bpf.h.
	@param header_length The length, in bytes, of the data link header.
 */
extern void  bpfattach(ifnet_t interface, u_int data_link_type,
    u_int header_length);

/*!
	@function bpf_attach
	@discussion Registers an interface with BPF. This allows bpf devices
		to attach to your interface to capture and transmit packets.
		Your interface will be unregistered automatically when your
		interface is detached. You may register multiple times with
		different data link types. An 802.11 interface would use this to
		allow clients to pick whether they want just an ethernet style
		frame or the 802.11 wireless headers as well. The first dlt you
		register will be considered the default. Any bpf device attaches
		that do not specify a data link type will use the default.
	@param interface The interface to register with BPF.
	@param data_link_type The data link type of the interface. See the
		DLT_* defines in bpf.h.
	@param header_length The length, in bytes, of the data link header.
 */
extern errno_t  bpf_attach(ifnet_t interface, u_int32_t data_link_type,
    u_int32_t header_length, bpf_send_func send, bpf_tap_func tap);

/*!
	@function bpf_tap_in
	@discussion Call this function when your interface receives a
		packet. This function will check if any bpf devices need a
		a copy of the packet.
	@param interface The interface the packet was received on.
	@param dlt The data link type of the packet.
	@param packet The packet received.
	@param header An optional pointer to a header that will be prepended.
	@param headerlen If the header was specified, the length of the header.
 */
extern void bpf_tap_in(ifnet_t interface, u_int32_t dlt, mbuf_t packet,
    void *header, size_t header_len);

/*!
	@function bpf_tap_out
	@discussion Call this function when your interface trasmits a
		packet. This function will check if any bpf devices need a
		a copy of the packet.
	@param interface The interface the packet was or will be transmitted on.
	@param dlt The data link type of the packet.
	@param packet The packet received.
	@param header An optional pointer to a header that will be prepended.
	@param headerlen If the header was specified, the length of the header.
 */
extern void bpf_tap_out(ifnet_t interface, u_int32_t dlt, mbuf_t packet,
    void *header, size_t header_len);

#endif /* KERNEL */

/*
 * Number of scratch memory words (for BPF_LD|BPF_MEM and BPF_ST).
 */
#define BPF_MEMWORDS 16

#endif /* _NET_BPF_H_ */
