#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "clastfm.h"

typedef struct _LFMTest LFMTest;

struct _LFMTest {
	int (*test_func)(LFMTest *);
	const char *api_name;
	LASTFM_SESSION *s;
};

int compare_LASTFM_ARTIST_INFO(const char *api,LASTFM_ARTIST_INFO *expect, LASTFM_ARTIST_INFO *r){
	int fail = 0;
	int i,j;
	
	if(expect->name && strcmp(r->name,expect->name)){
		printf("[FAIL] %s - Expected name = \"%s\" got \"%s\"\n",
			api,expect->name,r->name);
		fail++;
	}

	if(expect->similar && r->similar == NULL){
		printf("[FAIL] %s - Similar = NULL\n",api);
		fail++;
	}else{
		for(i=0;expect->similar[i];i++);
		for(j=0;r->similar[j];j++);

		if(i != j){
			printf("[FAIL] %s - Expected %i similars, got %i\n",api,i,j);
			fail++;
		}else{
			for(i=0;expect->similar[i];i++){
				if(strcmp(r->similar[i],expect->similar[i])){
					printf("[FAIL] %s - Expected similar[%i] = \"%s\" got \"%s\"\n",
						api,i,expect->similar[i],r->similar[i]);
					fail++;
				}
			}
		}
	}

	if(expect->playcount && r->playcount == 0){
		printf("[FAIL] %s - Playcount = 0\n",api);
		fail++;
	}
	
	if(expect->image_url && r->image_url == NULL){
		printf("[FAIL] %s - Image URL = NULL\n",api);
		fail++;
	}

	if(expect->image && r->image == NULL){
		printf("[FAIL] %s - Image = NULL\n",api);
		fail++;
	}


	if(expect->summary){
		if(r->summary == NULL){
			printf("[FAIL] %s - Summary = NULL\n",api);
			fail++;
		}else if (strcmp(expect->summary,r->summary)){
			printf("[FAIL] %s - Expected sumary =\n\"%s\"\nGot\n\"%s\"\n",
				api,expect->summary,r->summary);
			fail++;
		}
	}

	return fail;
}


int compare_LASTFM_TRACK_INFO(const char *api,LASTFM_TRACK_INFO *expect, LASTFM_TRACK_INFO *r){
	int fail = 0;
	if(expect->name && strcmp(r->name,expect->name)){
		printf("[FAIL] %s - Expected name = \"%s\" got \"%s\"\n",
			api,expect->name,r->name);
		fail++;
	}

	if(expect->artist && strcmp(r->artist,expect->artist)){
		printf("[FAIL] %s - Expected artist = \"%s\" got \"%s\"\n",
			api,expect->artist,r->artist);
		fail++;
	}

	if(expect->album && strcmp(r->album,expect->album)){
		printf("[FAIL] %s - Expected album = \"%s\" got \"%s\"\n",
			api,expect->album,r->album);
		fail++;
	}

	if(expect->playcount && r->playcount == 0){
		printf("[FAIL] %s - Playcount = 0\n",api);
		fail++;
	}
	
	if(expect->time && r->time != expect->time){
		printf("[FAIL] %s - Expected time = %lu, got %lu\n",
			api,expect->time,r->time);
		fail++;
	}

	if(expect->match && expect->match != r->match){
		printf("[FAIL] %s - Expected match = \"%f\" got \"%f\"\n",
			api,expect->match,r->match);
		fail++;
	}

	return fail;
}

int compare_LASTFM_ALBUM_INFO(const char *api, LASTFM_ALBUM_INFO *expect, LASTFM_ALBUM_INFO *r){
	int fail = 0;
	
	if(expect->name && strcmp(r->name,expect->name)){
		printf("[FAIL] %s - Expected name = \"%s\" got \"%s\"\n",
			api,expect->name,r->name);
		fail++;
	}
	if(expect->artist && strcmp(r->artist,expect->artist)){
		printf("[FAIL] %s - Expected artist = \"%s\" got \"%s\"\n",
			api,expect->artist,r->artist);
		fail++;
	}
	if(expect->releasedate && strcmp(r->releasedate,expect->releasedate)){
		printf("[FAIL] %s - Expected releasedate = \"%s\" got \"%s\"\n",
			api,expect->releasedate,r->releasedate);
		fail++;
	}

	if(expect->playcount && r->playcount == 0){
		printf("[FAIL] %s - Playcount = 0\n",api);
		fail++;
	}
	
	if(expect->image_url && r->image_url == NULL){
		printf("[FAIL] %s - Image URL = NULL\n",api);
		fail++;
	}

	if(expect->image && r->image == NULL){
		printf("[FAIL] %s - Image = NULL\n",api);
		fail++;
	}

	if(expect->summary && r->summary == NULL){
		printf("[FAIL] %s - Summary = NULL\n",api);
		fail++;
	}

	return fail;
}



int test_track_get_similar(LFMTest *test){
	int rv,i;
	int fail = 0;
	LFMList *results= NULL,*li;
	LASTFM_TRACK_INFO *r;
	LASTFM_TRACK_INFO *t[] = 
{
  &(LASTFM_TRACK_INFO){"Strong Enough","Cher",NULL,1,0,1.0},
  &(LASTFM_TRACK_INFO){"All Or Nothing","Cher",NULL,1,0,0.961879},
  &(LASTFM_TRACK_INFO){"Vogue","Madonna",NULL,1,0,0.291088},
  &(LASTFM_TRACK_INFO){"Hung Up","Madonna",NULL,1,0,0.286534},
  NULL
};

	rv = LASTFM_track_get_similar(test->s,"believe","cher",4,&results);

	if(results == NULL){
		printf("[FAIL] %s - No result\n",test->api_name);
		return 1;
	}

	if(rv != LASTFM_STATUS_OK){
		printf("[FAIL] %s - Expected rv = %i, got %i\n",
			test->api_name,LASTFM_STATUS_OK,rv);	
		fail++;
	}
	for(i=0,li=results;t[i];i++,li=li->next){
		if(li == NULL) {
			printf("[FAIL] %s - Expected 4+ results, got %i\n",
				test->api_name,i);
			fail++;
			break;
		}
		r = li->data;
		fail += compare_LASTFM_TRACK_INFO(test->api_name,t[i],r);
	}

	LASTFM_free_track_info_list(results);
	return fail;
}

int test_artist_get_info(LFMTest *test){
	LASTFM_ARTIST_INFO *r=NULL;
	int fail = 0;
	char *similar[] = { "Sonny & Cher","Madonna", "Cyndi Lauper",
				"Céline Dion","Kylie Minogue" , NULL};
	LASTFM_ARTIST_INFO t = { "Cher",
		"<![CDATA[Cher (born Cherilyn Sarkisian on May 20, 1946, later adopted by Gilbert LaPierre.) is an American singer, actress, songwriter, author and entertainer. Among her many career accomplishments in music, television and film, she has won an Academy Award, a Grammy Award, an Emmy Award and three Golden Globe Awards among others.  Referred to as the Goddess of Pop, Cher first rose to prominence in 1965 as one half of the pop/rock duo Sonny & Cher.]]>",1,(unsigned char *)1,1,(char *)1, similar };

	r = LASTFM_artist_get_info(test->s,"cher",NULL);
	if(r == NULL){
		printf("[FAIL] %s - No result\n",test->api_name);
		return 1;
	}

	fail = compare_LASTFM_ARTIST_INFO(test->api_name,&t,r);

	LASTFM_free_artist_info(r);
	return fail;
}

int test_album_get_info(LFMTest *test){
	LASTFM_ALBUM_INFO *r = NULL;
	LASTFM_ALBUM_INFO t = 
		{ "Moon Safari","Air",(char *)1, "    15 May 2006, 00:00",1,
			(unsigned char *)1,0,(char *)1};

	int fail = 0;
	
	r = LASTFM_album_get_info(test->s,"air","moon safari");
	if(r == NULL){
		printf("[FAIL] %s - No result\n",test->api_name);
		return 1;
	}
	fail = compare_LASTFM_ALBUM_INFO(test->api_name,&t,r);

	LASTFM_free_album_info(r);
	return fail;
}

int test_album_search(LFMTest *test){
	LASTFM_ALBUM_INFO *a = NULL;
	LFMList *list = NULL;
	LFMList *li;
	int fail = 0;
	int i,rpages;
	LASTFM_SESSION *s = test->s;
	const char *api = test->api_name;
	LASTFM_ALBUM_INFO *t[] =
	   {
	&(LASTFM_ALBUM_INFO){"Believe","Disturbed",NULL,NULL,0,NULL,0,(char *)1},
	&(LASTFM_ALBUM_INFO){"Make Believe","Weezer",NULL,NULL,0,NULL,0,(char *)1},
	&(LASTFM_ALBUM_INFO){"Don't Believe the Truth","Oasis",NULL,NULL,0,NULL,0,(char *)1},
	&(LASTFM_ALBUM_INFO){"If You Can Believe Your Eyes And Ears","The Mamas & The Papas",NULL,NULL,0,NULL,0,(char *)1},
	NULL
	   };

	/* 0 = unset limit, 1 = page number */
	rpages = LASTFM_album_search(s,"believe",LASTFM_IMAGE_URL_OG,0,1,&list);
	if(list == NULL){
		printf("[FAIL] %s - No result\n",api);
		return 1;
	}
	if(rpages < 54){
		printf("[FAIL] %s - Expected %i+ pages, got %i\n",api,54,rpages);
		fail++;
	}else{
		for(i=0, li = list;t[i];i++,li= li->next){
			if(li == NULL) {
				printf("[FAIL] %s - Expected %i+ results, got %i\n",api,4,i);
				fail++;
				break;
			}
			a = li->data;
			fail += compare_LASTFM_ALBUM_INFO(api,t[i],a);
		}
	}

	LASTFM_free_album_info_list(list);
	return fail;
}

int test_user_get_recent_tracks(LFMTest *test){
	int rv,i;
	int fail = 0;
	LFMList *results= NULL,*li;
	LASTFM_TRACK_INFO *r;
	LASTFM_TRACK_INFO *t[] = 
{
  &(LASTFM_TRACK_INFO){"Looking Through The Eye Of A Pig","Cypress Hill","IV",0,1283895716,0},
  &(LASTFM_TRACK_INFO){"Break 'Em Off Some","Cypress Hill","Black Sunday",0,1283895551,0},
  &(LASTFM_TRACK_INFO){"Hand On The Glock","Cypress Hill","Black Sunday",0,1283895338,0},
  &(LASTFM_TRACK_INFO){"A To The K","Cypress Hill","Black Sunday",0,1283895131,0},
  NULL
};

	rv = LASTFM_user_get_recent_tracks(test->s,"rj",1283895000,1283895962,0,&results);

	if(rv != 0){
		printf("[FAIL] %s - Expected remaining pages = 0, got %i\n",
			test->api_name,rv);
		fail++;
	}

	if(results == NULL){
		printf("[FAIL] %s - No result\n",test->api_name);
		return 1;
	}

	for(i=0,li=results;t[i];i++,li=li->next){
		if(li == NULL) {
			printf("[FAIL] %s - Expected 4 results, got %i\n",
				test->api_name,i);
			fail++;
			break;
		}
		r = li->data;
		fail += compare_LASTFM_TRACK_INFO(test->api_name,t[i],r);
	}

	if(li){
		printf("[FAIL] %s - Expected 4 results, got more\n",
			test->api_name);
		fail++;
	}

	LASTFM_free_track_info_list(results);
	return fail;
}


int main(int argc, char **argv)
{
	LASTFM_SESSION *s;
	int i;

	if(argc != 3){
		printf("%s [API_KEY] [API_SECRET]\n",argv[0]);
		return 1;
	}

	s = LASTFM_init(argv[1],argv[2]);
//	LASTFM_set_verbose(1);

	LFMTest *TESTS[] = {
		&(LFMTest){ test_artist_get_info, "artist.getInfo",s },
		&(LFMTest){ test_album_get_info, "album.getInfo",s },
		&(LFMTest){ test_album_search, "album.search",s },
		&(LFMTest){ test_track_get_similar, "track.getSimilar",s },
		&(LFMTest){ test_user_get_recent_tracks, "user.getRecentTracks",s },
		NULL
	};

	for(i=0;TESTS[i];i++){
		if(TESTS[i]->test_func(TESTS[i]) == 0){
			printf("[PASS] %s\n",TESTS[i]->api_name);
		}
	}

	LASTFM_dinit(s);
	return 0;
}
