/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2006  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include <iostream>
#include <stdexcept>

#include "Common.h"
#include "LineBuffer.h"

int main(int argc, char ** argv)
{
	StartTest("creating an empty line buffer");
	{
		LineBuffer Buffer;
	}
	EndTest();
	
	StartTest("empty line buffer is empty");
	{
		LineBuffer Buffer;
		
		EqualTest(Buffer.IsEmpty(), true);
		EqualTest(Buffer.PopAll(), std::string(""));
	}
	EndTest();
	
	StartTest("empty line buffer throws exception");
	{
		LineBuffer Buffer;
		
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
		EqualTest(Buffer.IsEmpty(), true);
	}
	EndTest();
	
	StartTest("only CRLF in buffer");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("\r\n");
		EqualTest(Buffer.IsEmpty(), false);
		EqualTest(Buffer.PopLine(), std::string(""));
	}
	EndTest();
	
	StartTest("pushing in line buffer");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		EqualTest(Buffer.PopAll(), std::string("Hallo"));
		EqualTest(Buffer.IsEmpty(), true);
		EqualTest(Buffer.PopAll(), std::string(""));
		
	}
	EndTest();
	
	StartTest("concatenating in line buffer");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!");
		EqualTest(Buffer.PopAll(), std::string("Hallo Test!"));
		EqualTest(Buffer.PopAll(), std::string(""));
		EqualTest(Buffer.IsEmpty(), true);
		
	}
	EndTest();
	
	StartTest("no line in line buffer");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!");
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
	}
	EndTest();
	
	StartTest("finished line in line buffer");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!\n");
		EqualTest(Buffer.PopLine(), std::string("Hallo Test!"));
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
		EqualTest(Buffer.PopAll(), std::string(""));
	}
	EndTest();
	
	StartTest("finished and unfinished lines in line buffer");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!\n Hallo");
		EqualTest(Buffer.PopLine(), std::string("Hallo Test!"));
		EqualTest(Buffer.PopAll(), std::string(" Hallo"));
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
		EqualTest(Buffer.PopAll(), std::string(""));
	}
	EndTest();
	
	StartTest("empty lines in line buffer");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!\n\n\n Hallo\n");
		EqualTest(Buffer.PopLine(), std::string("Hallo Test!"));
		EqualTest(Buffer.PopLine(), std::string(""));
		EqualTest(Buffer.PopLine(), std::string(""));
		EqualTest(Buffer.PopLine(), std::string(" Hallo"));
		EqualTest(Buffer.IsEmpty(), true);
	}
	EndTest();
	
	StartTest("finished line in line buffer (Win)");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!\r\n");
		EqualTest(Buffer.PopLine(), std::string("Hallo Test!"));
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
		EqualTest(Buffer.PopAll(), std::string(""));
	}
	EndTest();
	
	StartTest("finished lines in line buffer (Win)");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!\r\n With text.\r\n");
		EqualTest(Buffer.PopLine(), std::string("Hallo Test!"));
		EqualTest(Buffer.PopLine(), std::string(" With text."));
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
		EqualTest(Buffer.PopAll(), std::string(""));
	}
	EndTest();
	
	StartTest("empty test with different endline characters");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Empty");
		EqualTest(Buffer.IsEmpty(), true);
		Buffer.PushString("\r");
		EqualTest(Buffer.IsEmpty(), false);
		Buffer.PopLine();
		Buffer.PushString("Empty\n");
		EqualTest(Buffer.IsEmpty(), false);
		Buffer.PopLine();
		Buffer.PushString("Empty\r\n");
		EqualTest(Buffer.IsEmpty(), false);
		Buffer.PopLine();
	}
	EndTest();
	
	StartTest("finished line in line buffer (Mac)");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!\r");
		EqualTest(Buffer.PopLine(), std::string("Hallo Test!"));
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
		EqualTest(Buffer.PopAll(), std::string(""));
	}
	EndTest();
	
	StartTest("finished lines in line buffer (Mac)");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("Hallo");
		Buffer.PushString(" ");
		Buffer.PushString("Test!\r With text.\r");
		EqualTest(Buffer.PopLine(), std::string("Hallo Test!"));
		EqualTest(Buffer.PopLine(), std::string(" With text."));
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
		EqualTest(Buffer.PopAll(), std::string(""));
	}
	EndTest();
	
	StartTest("finished line in line buffer (mixed)");
	{
		LineBuffer Buffer;
		
		Buffer.PushString("LF\nCR\rCRLF\r\nSplitted CRLF\r");
		EqualTest(Buffer.PopLine(), std::string("LF"));
		EqualTest(Buffer.PopLine(), std::string("CR"));
		EqualTest(Buffer.PopLine(), std::string("CRLF"));
		EqualTest(Buffer.PopLine(), std::string("Splitted CRLF"));
		Buffer.PushString("\nTest");
		ExceptionTest(Buffer.PopLine(), std::underflow_error);
		Buffer.PushString("CR\r");
		EqualTest(Buffer.PopLine(), std::string("TestCR"));
		Buffer.PushString("LF\n");
		EqualTest(Buffer.PopLine(), std::string("LF"));
		EqualTest(Buffer.PopAll(), std::string(""));
	}
	EndTest();
	
	return iEndTests();
}
