/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2006  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include "Common.h"
#include "../Include/BufferWriter.h"
#include "../Include/DataChannel.h"
#include "../Include/DataRepository.h"
#include "../Include/Item.h"

const char g_pcFirstItemContent[] = "First Item Content";

class Task
{
public:
	Task(const Arxx::URI & URI, Arxx::Buffer & Buffer, Arxx::FetchStatus & FetchStatus) :
		m_URI(URI),
		m_Buffer(Buffer),
		m_FetchStatus(FetchStatus)
	{
	}
	
	bool bIsFinished(void) const
	{
		return m_FetchStatus == Arxx::FETCHED;
	}
	
	void vDo(void)
	{
		switch(m_FetchStatus)
		{
		case Arxx::FETCHING:
			{
				m_FetchStatus = Arxx::CONNECTING;
				
				return;
			}
		case Arxx::CONNECTING:
			{
				m_FetchStatus = Arxx::REQUESTING;
				
				return;
			}
		case Arxx::REQUESTING:
			{
				m_FetchStatus = Arxx::TRANSFERING;
				
				return;
			}
		case Arxx::TRANSFERING:
			{
				if(m_Buffer.stGetLength() < 4)
				{
					Arxx::BufferWriter BufferWriter(m_Buffer);
					
					BufferWriter << 'a';
				}
				else
				{
					m_FetchStatus = Arxx::DISCONNECTING;
				}
				
				return;
			}
		case Arxx::DISCONNECTING:
			{
				m_FetchStatus = Arxx::FETCHED;
				
				return;
			}
		case Arxx::FETCHED:
			{
				return;
			}
		default:
			{
				throw std::runtime_error("Unexpected status value.");
			}
		}
	}
private:
	const Arxx::URI & m_URI;
	Arxx::Buffer & m_Buffer;
	Arxx::FetchStatus & m_FetchStatus;
};

Task * pTask(0);

class MemoryDataChannel : public Arxx::DataChannel
{
public:
	MemoryDataChannel(void) :
		Arxx::DataChannel("memory:")
	{
	}
	
	virtual bool bFetchData(const Arxx::URI & URI, Arxx::Buffer & Buffer, Arxx::FetchStatus & FetchStatus)
	{
		if(URI.sGetPath() == "blocking")
		{
			Arxx::BufferWriter Writer(Buffer);
			
			FetchStatus = Arxx::TRANSFERING;
			Writer << g_pcFirstItemContent;
			FetchStatus = Arxx::FETCHED;
			
			return true;
		}
		else if(URI.sGetPath() == "task")
		{
			if(pTask == 0)
			{
				pTask = new Task(URI, Buffer, FetchStatus);
				
				return true;
			}
			else
			{
				return false;
			}
		}
		else
		{
			throw std::runtime_error("Unknown fetch method.");
		}
	}
};

int main(int argc, char ** argv)
{
	MemoryDataChannel MemoryDataChannel;
	
	Arxx::Repository.bRegisterDataChannel(&MemoryDataChannel);
	
	StartTest("synchronous in-memory resolution of external data");
	{
		Arxx::Item Item;
		
		Item.vSetExternal("memory:blocking");
		EqualTest(Item.bFetch(), true);
		EqualTest(Item.bIsFetched(), true);
		EqualTest(Item.bIsFetching(), false);
		EqualTest(Item.GetFetchStatus(), Arxx::FETCHED);
		EqualTest(Item.stGetLength(), static_cast< Arxx::Buffer::size_type >(strlen(g_pcFirstItemContent) + 1));
		BufferTest(Item, 0, strlen(g_pcFirstItemContent), g_pcFirstItemContent);
	}
	EndTest();
	
	StartTest("pseudo-asynchronous in-memory resolution of external data (1)");
	{
		Arxx::Item Item;
		
		Item.vSetExternal("memory:task");
		EqualTest(Item.bFetch(), true);
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::FETCHING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::CONNECTING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::REQUESTING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::TRANSFERING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::TRANSFERING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::TRANSFERING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::TRANSFERING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::TRANSFERING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.GetFetchStatus(), Arxx::DISCONNECTING);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), true);
		EqualTest(Item.bIsFetching(), false);
		EqualTest(Item.GetFetchStatus(), Arxx::FETCHED);
		pTask->vDo();
		EqualTest(Item.bIsFetched(), true);
		EqualTest(Item.bIsFetching(), false);
		EqualTest(Item.GetFetchStatus(), Arxx::FETCHED);
		EqualTest(Item.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(Item, 0, 4, "aaaa");
		delete pTask;
		pTask = 0;
	}
	EndTest();
	
	StartTest("pseudo-asynchronous in-memory fetching of external data (2)");
	{
		Arxx::Item Item;
		
		Item.vSetExternal("memory:task");
		EqualTest(Item.bFetch(), true);
		EqualTest(Item.bIsFetching(), true);
		EqualTest(Item.bFetch(), false);
		while(Item.bIsFetching() == true)
		{
			pTask->vDo();
		}
		EqualTest(Item.bIsFetched(), true);
		EqualTest(Item.bIsFetching(), false);
		EqualTest(Item.GetFetchStatus(), Arxx::FETCHED);
		EqualTest(Item.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(Item, 0, 4, "aaaa");
		delete pTask;
		pTask = 0;
	}
	EndTest();
	
	return iEndTests();
}
