/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include "Common.h"
#include "../Include/BufferReader.h"
#include "../Include/BufferWriter.h"
#include "../Include/DataChannel.h"
#include "../Include/DataRepository.h"
#include "../Include/ItemFactory.h"
#include "../Include/Merge.h"
#include "../Include/Reference.h"

#include <fstream>
#include <sstream>

std::ostream & operator<<(std::ostream & OStream, const Arxx::Archive::iterator & iItem)
{
	const Arxx::Item & Item(*iItem);
	
	OStream << Item.sGetName() << '[' << Item.u4GetUniqueID() << ']';
	
	return OStream;
}

class CustomItemFactory : public Arxx::ItemFactory
{
public:
	CustomItemFactory(void) :
		m_stCount(0)
	{
	}
	
	virtual Arxx::Item * pNewItem(Arxx::u4byte u4UniqueID = Arxx::g_u4InvalidID)
	{
		++m_stCount;
		
		return new Arxx::Item(*this, u4UniqueID);
	}
	
	virtual Arxx::Item * pNewItem(Arxx::Archive & Archive, Arxx::u4byte u4UniqueID)
	{
		++m_stCount;
		
		return new Arxx::Item(*this, Archive, u4UniqueID);
	}
	
	virtual void vDeleteItem(Arxx::Item * pItem)
	{
		delete pItem;
		--m_stCount;
	}
	
	size_t stGetCount(void)
	{
		return m_stCount;
	}
private:
	size_t m_stCount;
};

class DropDuplicateID : public Arxx::Rule
{
public:
	DropDuplicateID(void) :
		Arxx::Rule(Arxx::DROP)
	{
	}
	
	virtual bool bMatch(Arxx::Context & Context) const
	{
		return Context.GetTargetArchive().pGetItem(Context.GetCurrentItem().u4GetUniqueID()) != 0;
	}
} Dropper;

class Add : public Arxx::Rule
{
public:
	Add(void) :
		Arxx::Rule(Arxx::ADD)
	{
	}
	
	virtual bool bMatch(Arxx::Context & Context) const
	{
		return true;
	}
} Adder;

class LocalFileChannel : public Arxx::DataChannel
{
public:
	LocalFileChannel(const Arxx::URI & URI) :
		Arxx::DataChannel(URI)
	{
		if(URI.sGetScheme() != "file")
		{
			throw std::runtime_error("Created LocalFileChannel with scheme \"" + URI.sGetScheme() + "\" but \"file\" was expected.");
		}
	}
	
	virtual bool bFetchData(const Arxx::URI & URI, Arxx::Buffer & Buffer, Arxx::FetchStatus & FetchStatus)
	{
		if(URI.sGetScheme() != "file")
		{
			std::cerr << " *** LocalFileChannel::bFetchData called with scheme \"" << URI.sGetScheme() << "\" but \"file\" was expected." << std::endl;
			
			return false;
		}
		if((URI.sGetAuthority() != "") && (URI.sGetAuthority() != "localhost") && (URI.sGetAuthority() != "127.0.0.1"))
		{
			std::cerr << " *** LocalFileChannel::bFetchData called with authority \"" << URI.sGetAuthority() << "\" but \"\", \"localhost\" or \"127.0.0.1\" was expected." << std::endl;
			
			return false;
		}
		
		std::ifstream IStream(URI.sGetPath().c_str());
		
		if(IStream == false)
		{
			std::cerr << " *** LocalFileChannel::bFetchData: Could not open file with URI \"" << URI << "\"." << std::endl;
			
			return false;
		}
		Buffer.vSetLength(0);
		
		Arxx::BufferWriter BufferWriter(Buffer);
		
		FetchStatus = Arxx::TRANSFERING;
		BufferWriter << std::make_pair(static_cast< Arxx::Buffer::size_type >(0xFFFFFFFF), reinterpret_cast< std::istream * >(&IStream));
		IStream.close();
		FetchStatus = Arxx::FETCHED;
		
		return true;
	}
};

int main(int argc, char ** argv)
{
	StartTest("creating an empty buffer");
	{
		Arxx::Buffer Buffer;
		
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
	}
	EndTest();
	
	StartTest("creating and bloating and IO positioning");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vSetLength(10);
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(10));
	}
	EndTest();
	
	StartTest("inserting into buffer");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< Arxx::Buffer::const_pointer >("1234567890"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(10));
		BufferTest(Buffer, 0, 10, "1234567890");
		Buffer.vInsert(4, 5, reinterpret_cast< const Arxx::Buffer::value_type * >("98765"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(15));
		BufferTest(Buffer, 0, 15, "123498765567890");
	}
	EndTest();
	
	StartTest("setting the marker position on BufferReader");
	{
		Arxx::Buffer Buffer;
		Arxx::BufferReader BufferReader(Buffer);
		
		EqualTest(BufferReader.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		Buffer.vInsert(0, 10, reinterpret_cast< Arxx::Buffer::const_pointer >("1234567890"));
		EqualTest(BufferReader.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		BufferReader.vSetPosition(5);
		EqualTest(BufferReader.stGetPosition(), static_cast< Arxx::Buffer::size_type >(5));
	}
	EndTest();
	
	StartTest("input data into buffer via BufferWriter");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 20, reinterpret_cast< const Arxx::Buffer::value_type * >("11223344556677889900"));
		
		Arxx::BufferWriter BufferWriter(Buffer, 20);
		
		BufferWriter.vWrite(8, reinterpret_cast< const Arxx::Buffer::value_type * >("44336655"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(28));
		EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(28));
		BufferTest(Buffer, 0, 28, "1122334455667788990044336655");
	}
	EndTest();
	
	StartTest("input data into buffer via BufferWriter");
	{
		Arxx::Buffer Buffer;
		Arxx::BufferWriter BufferWriter(Buffer);
		
		Buffer.vInsert(0, 20, reinterpret_cast< const Arxx::Buffer::value_type * >("11223344556677889900"));
		BufferWriter.vWrite(8, reinterpret_cast< const Arxx::Buffer::value_type * >("44336655"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(28));
		EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(8));
		BufferTest(Buffer, 0, 28, "4433665511223344556677889900");
	}
	EndTest();
	
	StartTest("markers in a buffer");
	{
		Arxx::Buffer Buffer;
		Arxx::Buffer::Marker Marker1(Buffer, 0, Arxx::Buffer::Marker::RIGHT);
		Arxx::Buffer::Marker Marker2(Buffer, 23, Arxx::Buffer::Marker::LEFT);
		
		EqualTest(Marker1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(Marker2.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		
		{
			Arxx::Buffer::Marker Marker3(Buffer, 45, Arxx::Buffer::Marker::RIGHT);
			
			EqualTest(Marker3.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		}
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		EqualTest(Marker1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
		EqualTest(Marker2.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
	}
	EndTest();
	
	StartTest("marker before insert position");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		
		Arxx::Buffer::Marker Marker(Buffer, 5, Arxx::Buffer::Marker::LEFT);
		
		Buffer.vInsert(10, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("6677889900"));
		EqualTest(Marker.stGetPosition(), static_cast< Arxx::Buffer::size_type >(5));
	}
	EndTest();
	
	StartTest("marker after insert position");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		
		Arxx::Buffer::Marker Marker(Buffer, 8, Arxx::Buffer::Marker::LEFT);
		
		Buffer.vInsert(5, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("6677889900"));
		EqualTest(Marker.stGetPosition(), static_cast< Arxx::Buffer::size_type >(18));
	}
	EndTest();
	
	StartTest("left aligned marker at insert position");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		
		Arxx::Buffer::Marker Marker(Buffer, 5, Arxx::Buffer::Marker::LEFT);
		
		Buffer.vInsert(5, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("6677889900"));
		EqualTest(Marker.stGetPosition(), static_cast< Arxx::Buffer::size_type >(5));
	}
	EndTest();
	
	StartTest("right aligned marker at insert position");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		
		Arxx::Buffer::Marker Marker(Buffer, 5, Arxx::Buffer::Marker::RIGHT);
		
		Buffer.vInsert(5, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("6677889900"));
		EqualTest(Marker.stGetPosition(), static_cast< Arxx::Buffer::size_type >(15));
	}
	EndTest();
	
	StartTest("right aligned marker at insert position 0");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		
		Arxx::Buffer::Marker Marker(Buffer, 0, Arxx::Buffer::Marker::RIGHT);
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("6677889900"));
		EqualTest(Marker.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
	}
	EndTest();
	
	StartTest("left aligned marker at insert position in sub buffer");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		
		Arxx::Buffer SubBuffer(Buffer, 3, 4);
		Arxx::Buffer::Marker SubBufferMarker(SubBuffer, 2, Arxx::Buffer::Marker::LEFT);
		
		SubBuffer.vInsert(2, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("6677889900"));
		EqualTest(SubBufferMarker.stGetPosition(), static_cast< Arxx::Buffer::size_type >(2));
	}
	EndTest();
	
	StartTest("right aligned marker at insert position in sub buffer");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		
		Arxx::Buffer SubBuffer(Buffer, 3, 4);
		Arxx::Buffer::Marker SubBufferMarker(SubBuffer, 2, Arxx::Buffer::Marker::RIGHT);
		
		SubBuffer.vInsert(2, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("6677889900"));
		EqualTest(SubBufferMarker.stGetPosition(), static_cast< Arxx::Buffer::size_type >(12));
	}
	EndTest();
	
	StartTest("right aligned marker at insert position 0 in sub buffer");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("1122334455"));
		
		Arxx::Buffer SubBuffer(Buffer, 3, 4);
		Arxx::Buffer::Marker SubBufferMarker(SubBuffer, Arxx::Buffer::Marker::BEGIN, Arxx::Buffer::Marker::RIGHT);
		
		SubBuffer.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("6677889900"));
		EqualTest(SubBufferMarker.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
	}
	EndTest();
	
	StartTest("creating a sub buffer and testing initial values");
	{
		Arxx::Buffer Buffer;
		Arxx::Buffer SubBuffer(Buffer, 0, 0);
		
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
	}
	EndTest();
	
	StartTest("bloating the sup buffer")
	{
		Arxx::Buffer Buffer;
		Arxx::Buffer SubBuffer(Buffer, 0, 0);
		
		Buffer.vSetLength(30);
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(30));
		EqualTest(SubBuffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
	}
	EndTest();
	
	StartTest("bloating the sub buffer");
	{
		Arxx::Buffer Buffer;
		Arxx::Buffer SubBuffer(Buffer, 0, 0);
		
		SubBuffer.vSetLength(30);
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(30));
		EqualTest(SubBuffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(30));
	}
	EndTest();
	
	StartTest("copy constructing a bufferwriter");
	{
		Arxx::Buffer Buffer;
		Arxx::BufferWriter Writer1(Buffer);
		
		EqualTest(Writer1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		Writer1.vWrite(10, reinterpret_cast< Arxx::Buffer::const_pointer >("abcdefghij"));
		EqualTest(Writer1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
		{
			Arxx::BufferWriter Writer2(Writer1);
			
			EqualTest(Writer1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
			EqualTest(Writer2.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
			Writer2.vWrite(6, reinterpret_cast< Arxx::Buffer::const_pointer >("klmnop"));
			EqualTest(Writer1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
			EqualTest(Writer2.stGetPosition(), static_cast< Arxx::Buffer::size_type >(16));
		}
		EqualTest(Writer1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
	}
	EndTest();
	
	StartTest("copy constructing a bufferwriter without the original");
	{
		Arxx::Buffer Buffer;
		Arxx::BufferWriter * Writer1(new Arxx::BufferWriter(Buffer));
		
		EqualTest(Writer1->stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		Writer1->vWrite(10, reinterpret_cast< Arxx::Buffer::const_pointer >("abcdefghij"));
		EqualTest(Writer1->stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
		
		Arxx::BufferWriter * Writer2(new Arxx::BufferWriter(*Writer1));
		
		EqualTest(Writer1->stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
		EqualTest(Writer2->stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
		Writer2->vWrite(6, reinterpret_cast< Arxx::Buffer::const_pointer >("klmnop"));
		EqualTest(Writer1->stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
		EqualTest(Writer2->stGetPosition(), static_cast< Arxx::Buffer::size_type >(16));
		delete Writer1;
		Writer2->vWrite(6, reinterpret_cast< Arxx::Buffer::const_pointer >("qrstuv"));
		EqualTest(Writer2->stGetPosition(), static_cast< Arxx::Buffer::size_type >(22));
		delete Writer2;
	}
	EndTest();
	
	StartTest("copy constructing a bufferreader");
	{
		Arxx::Buffer Buffer;
		Arxx::BufferWriter Writer(Buffer);
		
		Writer << 10.2f << 20.4f << 10.0f;
		EqualTest(Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(3 * sizeof(float)));
		
		Arxx::BufferReader Reader1(Buffer);
		float Value1;
		
		Reader1 >> Value1;
		EqualTest(Value1, 10.2f);
		EqualTest(Reader1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(sizeof(float)));
		{
			Arxx::BufferReader Reader2(Reader1);
			float Value2;
			
			EqualTest(Reader1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(sizeof(float)));
			EqualTest(Reader2.stGetPosition(), static_cast< Arxx::Buffer::size_type >(sizeof(float)));
			Reader2 >> Value2;
			EqualTest(Value2, 20.4f);
			EqualTest(Reader1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(sizeof(float)));
			EqualTest(Reader2.stGetPosition(), static_cast< Arxx::Buffer::size_type >(2 * sizeof(float)));
		}
		EqualTest(Reader1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(sizeof(float)));
		Reader1 >> Value1;
		EqualTest(Value1, 20.4f);
		EqualTest(Reader1.stGetPosition(), static_cast< Arxx::Buffer::size_type >(2 * sizeof(float)));
	}
	EndTest();
	
	StartTest("copy constructing a bufferreader and using it with destroyed original");
	{
		Arxx::Buffer Buffer;
		Arxx::BufferWriter Writer(Buffer);
		
		Writer << 10.2f << 20.4f << 10.5f;
		EqualTest(Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(3 * sizeof(float)));
		
		Arxx::BufferReader * Reader1(new Arxx::BufferReader(Buffer));
		float Value1;
		
		(*Reader1) >> Value1;
		EqualTest(Value1, 10.2f);
		EqualTest(Reader1->stGetPosition(), static_cast< Arxx::Buffer::size_type >(sizeof(float)));
		
		Arxx::BufferReader * Reader2(new Arxx::BufferReader(*Reader1));
		float Value2;
		
		EqualTest(Reader2->stGetPosition(), static_cast< Arxx::Buffer::size_type >(sizeof(float)));
		(*Reader1) >> Value1;
		EqualTest(Value1, 20.4f);
		EqualTest(Reader1->stGetPosition(), static_cast< Arxx::Buffer::size_type >(2 * sizeof(float)));
		delete Reader1;
		Reader1 = 0;
		(*Reader2) >> Value2;
		EqualTest(Reader2->stGetPosition(), static_cast< Arxx::Buffer::size_type >(2 * sizeof(float)));
		EqualTest(Value2, 20.4f);
		EqualTest(Reader2->stGetPosition(), static_cast< Arxx::Buffer::size_type >(2 * sizeof(float)));
		(*Reader2) >> Value2;
		EqualTest(Value2, 10.5f);
		EqualTest(Reader2->stGetPosition(), static_cast< Arxx::Buffer::size_type >(3 * sizeof(float)));
		delete Reader2;
		Reader2 = 0;
	}
	EndTest();
	
	StartTest("writing to a sub buffer writer at default end position");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 26, reinterpret_cast< const Arxx::Buffer::value_type * >("abcdefghijklmnopqrstuvwxyz"));
		BufferTest(Buffer, 0, 26, "abcdefghijklmnopqrstuvwxyz");
		
		Arxx::Buffer SubBuffer(Buffer, 6, 14);
		Arxx::BufferWriter SubBufferWriter(SubBuffer);
		
		SubBufferWriter.vWrite(10, reinterpret_cast< const Arxx::Buffer::value_type * >("0000000000"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(36));
		EqualTest(SubBuffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(24));
		EqualTest(SubBufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(24));
		BufferTest(Buffer, 0, 36, "abcdefghijklmnopqrst0000000000uvwxyz");
		BufferTest(SubBuffer, 0, 24, "ghijklmnopqrst0000000000");
	}
	EndTest();
	
	StartTest("writing to a sub buffer writer");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 26, reinterpret_cast< const Arxx::Buffer::value_type * >("abcdefghijklmnopqrstuvwxyz"));
		BufferTest(Buffer, 0, 26, "abcdefghijklmnopqrstuvwxyz");
		
		Arxx::Buffer SubBuffer(Buffer, 6, 14);
		Arxx::BufferWriter SubBufferWriter(SubBuffer, 7);
		
		SubBufferWriter.vWrite(10, reinterpret_cast< const Arxx::Buffer::value_type * >("0000000000"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(36));
		EqualTest(SubBuffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(24));
		EqualTest(SubBufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(17));
		BufferTest(Buffer, 0, 36, "abcdefghijklm0000000000nopqrstuvwxyz");
		BufferTest(SubBuffer, 0, 24, "ghijklm0000000000nopqrst");
	}
	EndTest();
	
	StartTest("writing to a sub buffer writer at position 0");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 26, reinterpret_cast< const Arxx::Buffer::value_type * >("abcdefghijklmnopqrstuvwxyz"));
		BufferTest(Buffer, 0, 26, "abcdefghijklmnopqrstuvwxyz");
		
		Arxx::Buffer SubBuffer(Buffer, 6, 14);
		Arxx::BufferWriter SubBufferWriter(SubBuffer, Arxx::Buffer::Marker::BEGIN);
		
		SubBufferWriter.vWrite(10, reinterpret_cast< const Arxx::Buffer::value_type * >("0000000000"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(36));
		EqualTest(SubBuffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(24));
		EqualTest(SubBufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
		BufferTest(Buffer, 0, 36, "abcdef0000000000ghijklmnopqrstuvwxyz");
		BufferTest(SubBuffer, 0, 24, "0000000000ghijklmnopqrst");
	}
	EndTest();
	
	StartTest("creating a sub buffer in a working sup buffer");
	{
		Arxx::Buffer Buffer;
		Arxx::BufferWriter BufferWriter(Buffer);
		
		BufferWriter.vWrite(26, reinterpret_cast< const Arxx::Buffer::value_type * >("abcdefghijklmnopqrstuvwxyz"));
		
		Arxx::Buffer SubBuffer(Buffer, 5, 10);
		
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(26));
		EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(26));
		BufferTest(Buffer, 0, 26, "abcdefghijklmnopqrstuvwxyz");
		EqualTest(SubBuffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(10));
		BufferTest(SubBuffer, 0, 10, "fghijklmno");
		BufferWriter.vSetPosition(20);
		
		Arxx::BufferWriter SubBufferWriter(SubBuffer, Arxx::Buffer::Marker::BEGIN);
		
		SubBufferWriter.vWrite(10, reinterpret_cast< const Arxx::Buffer::value_type * >("onmlkjihgf"));
		EqualTest(SubBuffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(20));
		EqualTest(SubBufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(10));
		BufferTest(SubBuffer, 0, 20, "onmlkjihgffghijklmno");
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(36));
		EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(30));
		BufferTest(Buffer, 0, 36, "abcdeonmlkjihgffghijklmnopqrstuvwxyz");
	}
	EndTest();
	
	StartTest("deleting a sub buffer from a working sup buffer");
	{
		Arxx::Buffer Buffer;
		Arxx::BufferWriter BufferWriter(Buffer);
		
		BufferWriter.vWrite(26, reinterpret_cast< const Arxx::Buffer::value_type * >("abcdefghijklmnopqrstuvwxyz"));
		
		{
			Arxx::Buffer SubBuffer(Buffer, 5, 10);
			Arxx::BufferWriter SubBufferWriter(SubBuffer, Arxx::Buffer::Marker::BEGIN);
			
			SubBufferWriter.vWrite(10, reinterpret_cast< const Arxx::Buffer::value_type * >("onmlkjihgf"));
		}
		
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(36));
		EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(36));
		BufferTest(Buffer, 0, 36, "abcdeonmlkjihgffghijklmnopqrstuvwxyz");
	}
	EndTest();
	
	StartTest("two sub buffers on the same position inserting");
	{
		{
			Arxx::Buffer Buffer;
			Arxx::Buffer SubBuffer1(Buffer, 0, 0);
			Arxx::Buffer SubBuffer2(Buffer, 0, 0);
			Arxx::BufferWriter SubBuffer1Writer(SubBuffer1);
			Arxx::BufferWriter SubBuffer2Writer(SubBuffer2);
			
			SubBuffer1Writer.vWrite(4, reinterpret_cast< const Arxx::Buffer::value_type * >("aaaa"));
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			BufferTest(Buffer, 0, 4, "aaaa");
			EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer1Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
			EqualTest(SubBuffer2Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
			SubBuffer2Writer.vWrite(4, reinterpret_cast< const Arxx::Buffer::value_type * >("bbbb"));
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(8));
			BufferTest(Buffer, 0, 8, "aaaabbbb");
			EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer1Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer2Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
		}
		{
			Arxx::Buffer Buffer;
			Arxx::Buffer SubBuffer1(Buffer, 0, 0);
			Arxx::Buffer SubBuffer2(Buffer, 0, 0);
			Arxx::BufferWriter SubBuffer1Writer(SubBuffer1);
			Arxx::BufferWriter SubBuffer2Writer(SubBuffer2);
			
			SubBuffer2Writer.vWrite(4, reinterpret_cast< const Arxx::Buffer::value_type * >("bbbb"));
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			BufferTest(Buffer, 0, 4, "bbbb");
			EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
			EqualTest(SubBuffer1Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
			EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer2Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
			SubBuffer1Writer.vWrite(4, reinterpret_cast< const Arxx::Buffer::value_type * >("aaaa"));
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(8));
			BufferTest(Buffer, 0, 8, "aaaabbbb");
			EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer1Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(SubBuffer2Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
		}
	}
	EndTest();
	
	StartTest("same with four buffers and mixed insertion/input");
	{
		Arxx::Buffer Buffer;
		Arxx::Buffer SubBuffer1(Buffer, 0, 0);
		Arxx::Buffer SubBuffer2(Buffer, 0, 0);
		Arxx::Buffer SubBuffer3(Buffer, 0, 0);
		Arxx::Buffer SubBuffer4(Buffer, 0, 0);
		Arxx::BufferWriter SubBuffer4Writer(SubBuffer4);
		Arxx::BufferWriter SubBuffer3Writer(SubBuffer3);
		
		SubBuffer4Writer.vWrite(4, reinterpret_cast< const Arxx::Buffer::value_type * >("4444"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(Buffer, 0, 4, "4444");
		EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer3.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer3Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer4.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		EqualTest(SubBuffer4Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(SubBuffer4, 0, 4, "4444");
		SubBuffer2.vInsert(0, 2, reinterpret_cast< const Arxx::Buffer::value_type * >("22"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(6));
		BufferTest(Buffer, 0, 6, "224444");
		EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(2));
		BufferTest(SubBuffer2, 0, 2, "22");
		EqualTest(SubBuffer3.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer3Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer4.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		EqualTest(SubBuffer4Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(SubBuffer4, 0, 4, "4444");
		SubBuffer3Writer.vWrite(3, reinterpret_cast< const Arxx::Buffer::value_type * >("333"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(9));
		BufferTest(Buffer, 0, 9, "223334444");
		EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(2));
		BufferTest(SubBuffer2, 0, 2, "22");
		EqualTest(SubBuffer3.stGetLength(), static_cast< Arxx::Buffer::size_type >(3));
		EqualTest(SubBuffer3Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(3));
		BufferTest(SubBuffer3, 0, 3, "333");
		EqualTest(SubBuffer4.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		EqualTest(SubBuffer4Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(SubBuffer4, 0, 4, "4444");
		SubBuffer1.vInsert(0, 10, reinterpret_cast< const Arxx::Buffer::value_type * >("0101010101"));
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(19));
		BufferTest(Buffer, 0, 19, "0101010101223334444");
		EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(10));
		BufferTest(SubBuffer1, 0, 10, "0101010101");
		EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(2));
		BufferTest(SubBuffer2, 0, 2, "22");
		EqualTest(SubBuffer3.stGetLength(), static_cast< Arxx::Buffer::size_type >(3));
		EqualTest(SubBuffer3Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(3));
		BufferTest(SubBuffer3, 0, 3, "333");
		EqualTest(SubBuffer4.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		EqualTest(SubBuffer4Writer.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(SubBuffer4, 0, 4, "4444");
	}
	EndTest();
	
	StartTest("deleting from sup with deletion overlapping subs");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 32, reinterpret_cast< const Arxx::Buffer::value_type * >("aaaabbbbccccddddeeeeffffgggghhhh"));
		
		Arxx::Buffer SubBuffer1(Buffer, 4, 10);
		Arxx::Buffer SubBuffer2(Buffer, 18, 10);
		
		BufferTest(SubBuffer1, 0, 10, "bbbbccccdd");
		BufferTest(SubBuffer2, 0, 10, "eeffffgggg");
		Buffer.vDelete(10, 12);
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(20));
		BufferTest(Buffer, 0, 20, "aaaabbbbccffgggghhhh");
		EqualTest(SubBuffer1.stGetLength(), static_cast< Arxx::Buffer::size_type >(6));
		BufferTest(SubBuffer1, 0, 6, "bbbbcc");
		EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(6));
		BufferTest(SubBuffer2, 0, 6, "ffgggg");
	}
	EndTest();
	
	StartTest("deleting from sup with deletion overlapping two overlapping subs");
	{
		Arxx::Buffer Buffer;
		
		Buffer.vInsert(0, 20, reinterpret_cast< const Arxx::Buffer::value_type * >("aaaabbbbccccddddeeee"));
		
		Arxx::Buffer SubBuffer1(Buffer, 2, 10);
		Arxx::Buffer SubBuffer2(Buffer, 8, 10);
		
		BufferTest(SubBuffer1, 0, 10, "aabbbbcccc");
		BufferTest(SubBuffer2, 0, 10, "ccccddddee");
		Buffer.vDelete(6, 8);
		EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(12));
		BufferTest(Buffer, 0, 12, "aaaabbddeeee");
		EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(SubBuffer1, 0, 4, "aabb");
		EqualTest(SubBuffer2.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
		BufferTest(SubBuffer2, 0, 4, "ddee");
	}
	EndTest();
	
	StartTest("writing a string to a buffer");
	{
		{
			Arxx::Buffer Buffer;
			Arxx::BufferWriter BufferWriter(Buffer);
			
			BufferWriter << std::string("a string");
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(9));
			EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(9));
		}
		{
			Arxx::Buffer Buffer;
			Arxx::BufferWriter BufferWriter(Buffer);
			
			BufferWriter << "a string";
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(9));
			EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(9));
		}
	}
	EndTest();
	
	StartTest("writing and reading Arxx::u1byte values");
	{
		{
			Arxx::Buffer Buffer;
			Arxx::BufferWriter BufferWriter(Buffer);
			Arxx::u1byte WriteValue(146);
			
			BufferWriter << WriteValue;
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(1));
			EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(1));
			
			Arxx::BufferReader BufferReader(Buffer);
			Arxx::u1byte ReadValue(0);
			
			BufferReader >> ReadValue;
			EqualTest(BufferReader.stGetPosition(), static_cast< Arxx::Buffer::size_type >(1));
			EqualTest(WriteValue, ReadValue);
		}
	}
	EndTest();
	
	StartTest("writing and reading Arxx::u4byte values");
	{
		{
			Arxx::Buffer Buffer;
			Arxx::BufferWriter BufferWriter(Buffer);
			Arxx::u4byte WriteValue(93);
			
			BufferWriter << WriteValue;
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
			
			Arxx::BufferReader BufferReader(Buffer);
			Arxx::u4byte ReadValue(0);
			
			BufferReader >> ReadValue;
			EqualTest(BufferReader.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(WriteValue, ReadValue);
		}
	}
	EndTest();
	
	StartTest("writing and reading Arxx::u8byte values");
	{
		{
			Arxx::Buffer Buffer;
			Arxx::BufferWriter BufferWriter(Buffer);
			Arxx::u8byte WriteValue(6543210987ll);
			
			BufferWriter << WriteValue;
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(8));
			EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(8));
			
			Arxx::BufferReader BufferReader(Buffer);
			Arxx::u8byte ReadValue(0);
			
			BufferReader >> ReadValue;
			EqualTest(BufferReader.stGetPosition(), static_cast< Arxx::Buffer::size_type >(8));
			EqualTest(WriteValue, ReadValue);
		}
	}
	EndTest();
	
	StartTest("writing and reading float values");
	{
		{
			Arxx::Buffer Buffer;
			Arxx::BufferWriter BufferWriter(Buffer);
			float fWriteValue(983.21f);
			
			BufferWriter << fWriteValue;
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(4));
			EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(4));
			
			Arxx::BufferReader BufferReader(Buffer);
			float fReadValue(0.0f);
			
			BufferReader >> fReadValue;
			EqualTest(fWriteValue, fReadValue);
		}
	}
	EndTest();
	
	StartTest("writing and reading bool values");
	{
		{
			Arxx::Buffer Buffer;
			Arxx::BufferWriter BufferWriter(Buffer);
			bool bWriteValue(true);
			
			BufferWriter << bWriteValue;
			EqualTest(Buffer.stGetLength(), static_cast< Arxx::Buffer::size_type >(1));
			EqualTest(BufferWriter.stGetPosition(), static_cast< Arxx::Buffer::size_type >(1));
			
			Arxx::BufferReader BufferReader(Buffer);
			bool bReadValue(false);
			
			BufferReader >> bReadValue;
			EqualTest(bWriteValue, bReadValue);
		}
	}
	EndTest();
	
	StartTest("testing URI parsing");
	{
		{
			Arxx::URI URI;
			
			EqualTest(URI.bIsValid(), false);
		}
		{
			Arxx::URI URI("");
			
			EqualTest(URI.bIsValid(), true);
		}
		{
			Arxx::URI URI("file://");
			
			EqualTest(URI.bIsValid(), true);
		}
		{
			Arxx::URI URI("file://");
			
			EqualTest(URI.sGetScheme(), std::string("file"));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string(""));
			EqualTest(URI.sGetQuery(), std::string(""));
			EqualTest(URI.sGetFragment(), std::string(""));
		}
		{
			Arxx::URI URI("http://www.ics.uci.edu/pub/ietf/uri/#Related");
			
			EqualTest(URI.sGetScheme(), std::string("http"));
			EqualTest(URI.sGetAuthority(), std::string("www.ics.uci.edu"));
			EqualTest(URI.sGetPath(), std::string("/pub/ietf/uri/"));
			EqualTest(URI.sGetQuery(), std::string(""));
			EqualTest(URI.sGetFragment(), std::string("Related"));
		}
		{
			Arxx::URI URI("#FragmentOnly");
			
			EqualTest(URI.sGetScheme(), std::string(""));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string(""));
			EqualTest(URI.sGetQuery(), std::string(""));
			EqualTest(URI.sGetFragment(), std::string("FragmentOnly"));
		}
		{
			Arxx::URI URI("?QueryOnly");
			
			EqualTest(URI.sGetScheme(), std::string(""));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string(""));
			EqualTest(URI.sGetQuery(), std::string("QueryOnly"));
			EqualTest(URI.sGetFragment(), std::string(""));
		}
		{
			Arxx::URI URI("?Query#Fragment");
			
			EqualTest(URI.sGetScheme(), std::string(""));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string(""));
			EqualTest(URI.sGetQuery(), std::string("Query"));
			EqualTest(URI.sGetFragment(), std::string("Fragment"));
		}
		{
			Arxx::URI URI("?Query#Fragment/with:specials?");
			
			EqualTest(URI.sGetScheme(), std::string(""));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string(""));
			EqualTest(URI.sGetQuery(), std::string("Query"));
			EqualTest(URI.sGetFragment(), std::string("Fragment/with:specials?"));
		}
		{
			Arxx::URI URI("ftp://?Query#Fragment/with:specials?");
			
			EqualTest(URI.sGetScheme(), std::string("ftp"));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string(""));
			EqualTest(URI.sGetQuery(), std::string("Query"));
			EqualTest(URI.sGetFragment(), std::string("Fragment/with:specials?"));
		}
		{
			Arxx::URI URI("file:///etc/hostname");
			
			EqualTest(URI.sGetScheme(), std::string("file"));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string("/etc/hostname"));
			EqualTest(URI.sGetQuery(), std::string(""));
			EqualTest(URI.sGetFragment(), std::string(""));
		}
		{
			Arxx::URI URI("file:///some/test.arx?Offset=23");
			
			EqualTest(URI.sGetScheme(), std::string("file"));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string("/some/test.arx"));
			EqualTest(URI.sGetQuery(), std::string("Offset=23"));
			EqualTest(URI.sGetParameter("Offset"), std::string("23"));
			EqualTest(URI.sGetFragment(), std::string(""));
		}
		{
			Arxx::URI URI("arx://arx.worldforge.org/ember?id=23&type=header#name");
			
			EqualTest(URI.sGetScheme(), std::string("arx"));
			EqualTest(URI.sGetAuthority(), std::string("arx.worldforge.org"));
			EqualTest(URI.sGetPath(), std::string("/ember"));
			EqualTest(URI.sGetQuery(), std::string("id=23&type=header"));
			EqualTest(URI.sGetParameter("id"), std::string("23"));
			EqualTest(URI.sGetParameter("type"), std::string("header"));
			EqualTest(URI.sGetFragment(), std::string("name"));
		}
		{
			Arxx::URI URI("http://water-wheel.dyndns.org/Sections/Starschiffchen/Data.arx");
			
			URI.vSetScheme("arx");
			URI.vSetPath("/starschiffchen/Data");
			URI.vSetFragment("root-item");
			EqualTest(URI.sGetURI(), std::string("arx://water-wheel.dyndns.org/starschiffchen/Data#root-item"));
		}
		{
			Arxx::URI URI("http://water-wheel.dyndns.org/Sections/Starschiffchen/Data.arx");
			
			URI.vSetParameter("id", "45");
			URI.vSetParameter("type", "data");
			EqualTest(URI.sGetURI(), std::string("http://water-wheel.dyndns.org/Sections/Starschiffchen/Data.arx?id=45&type=data"));
		}
		{
			Arxx::URI URI("http://water-wheel.dyndns.org/Sections/Starschiffchen/Data.arx?value=hello");
			
			URI.vSetParameter("id", "8");
			URI.vSetParameter("type", "header");
			EqualTest(URI.sGetQuery(), std::string("id=8&type=header&value=hello"));
		}
		{
			Arxx::URI URI;
			
			EqualTest(URI.bIsValid(), false);
			URI.vSetURI("file:///etc/inittab");
			EqualTest(URI.bIsValid(), true);
			URI.vInvalidate();
			EqualTest(URI.bIsValid(), false);
		}
		{
			Arxx::URI URI;
			
			URI.vSetURI("test/testing.arx");
			EqualTest(URI.sGetScheme(), std::string(""));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string("test/testing.arx"));
			EqualTest(URI.sGetQuery(), std::string(""));
			EqualTest(URI.sGetFragment(), std::string(""));
		}
		{
			Arxx::URI URI;
			
			URI.vSetURI("test#testing.arx");
			EqualTest(URI.sGetScheme(), std::string(""));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string("test"));
			EqualTest(URI.sGetQuery(), std::string(""));
			EqualTest(URI.sGetFragment(), std::string("testing.arx"));
		}
		{
			Arxx::URI URI;
			
			URI.vSetURI("test?testing=arx");
			EqualTest(URI.sGetScheme(), std::string(""));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string("test"));
			EqualTest(URI.sGetQuery(), std::string("testing=arx"));
			EqualTest(URI.sGetParameter("testing"), std::string("arx"));
			EqualTest(URI.sGetFragment(), std::string(""));
		}
		{
			Arxx::URI URI("file:some/relative/file.arx#fragment");
			
			EqualTest(URI.sGetScheme(), std::string("file"));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string("some/relative/file.arx"));
			EqualTest(URI.sGetQuery(), std::string(""));
			EqualTest(URI.sGetFragment(), std::string("fragment"));
			URI.vSetParameter("type", "header");
			URI.vSetFragment("name");
			EqualTest(URI.sGetURI(), std::string("file:some/relative/file.arx?type=header#name"));
		}
		{
			Arxx::URI URI("file:RelativePath.arx?Some=Query&Some=Arguments");
			
			EqualTest(URI.sGetScheme(), std::string("file"));
			EqualTest(URI.sGetAuthority(), std::string(""));
			EqualTest(URI.sGetPath(), std::string("RelativePath.arx"));
		}
	}
	EndTest();
	
	StartTest("testing URI matching of the data repository");
	{
		LocalFileChannel SpecificFile("file:///hello/good/old/world.arx");
		
		Arxx::Repository.bRegisterDataChannel(&SpecificFile);
		
		EqualTest(Arxx::Repository.bHasDataChannel("file:///etc/hostname"), false);
		EqualTest(Arxx::Repository.bHasDataChannel("file:///ztc/hostname"), false);
		
		LocalFileChannel AbsoluteFiles("file:///");
		Arxx::Repository.bRegisterDataChannel(&AbsoluteFiles);
		
		EqualTest(Arxx::Repository.pGetDataChannel("file:///some/file?with=query"), const_cast< const Arxx::DataChannel * >(dynamic_cast< Arxx::DataChannel * >(&AbsoluteFiles)));
		EqualTest(Arxx::Repository.pGetDataChannel("file:///some/folder/test.dat"), const_cast< const Arxx::DataChannel * >(dynamic_cast< Arxx::DataChannel * >(&AbsoluteFiles)));
		EqualTest(Arxx::Repository.pGetDataChannel("file:///test.dat"), const_cast< const Arxx::DataChannel * >(dynamic_cast< Arxx::DataChannel * >(&AbsoluteFiles)));
		EqualTest(Arxx::Repository.pGetDataChannel("file:///hello/good/old/world.cs"), const_cast< const Arxx::DataChannel * >(dynamic_cast< Arxx::DataChannel * >(&AbsoluteFiles)));
		EqualTest(Arxx::Repository.pGetDataChannel("file:///hello/good/old/world.arx"), const_cast< const Arxx::DataChannel * >(dynamic_cast< Arxx::DataChannel * >(&SpecificFile)));
		EqualTest(Arxx::Repository.pGetDataChannel("file:///hello/good/old/world.arx?Argument=Value"), const_cast< const Arxx::DataChannel * >(dynamic_cast< Arxx::DataChannel * >(&SpecificFile)));
		
		LocalFileChannel SpecificFile2("file:///hello/good/old/world/with/a/much/too/long/name/but/only/for/testing/purposes.arx");
		LocalFileChannel SpecificFile3("file:///home/user/world.arx");
		
		Arxx::Repository.bRegisterDataChannel(&SpecificFile2);
		Arxx::Repository.bRegisterDataChannel(&SpecificFile3);
		EqualTest(Arxx::Repository.pGetDataChannel("file:///home/user/world.arx?Argument=Value"), const_cast< const Arxx::DataChannel * >(dynamic_cast< Arxx::DataChannel * >(&SpecificFile3)));
		Arxx::Repository.bUnregisterDataChannel(&SpecificFile);
		Arxx::Repository.bUnregisterDataChannel(&AbsoluteFiles);
		Arxx::Repository.bUnregisterDataChannel(&SpecificFile2);
		Arxx::Repository.bUnregisterDataChannel(&SpecificFile3);
		
		LocalFileChannel RelativePath("file:RelativePath.arx");
		
		Arxx::Repository.bRegisterDataChannel(&RelativePath);
		EqualTest(Arxx::Repository.pGetDataChannel("file:RelativePath.arx?Some=Query&Some=Arguments"), const_cast< const Arxx::DataChannel * >(dynamic_cast< Arxx::DataChannel * >(&RelativePath)));
		Arxx::Repository.bUnregisterDataChannel(&RelativePath);
	}
	EndTest();
	
///////////////////////////////////////////////////////////////////////////////////////////////////
//	ATTENTION: registering fallback LocalFileChannel                                             //
///////////////////////////////////////////////////////////////////////////////////////////////////
	LocalFileChannel FallbackChannel("file:///");
	
	Arxx::Repository.bRegisterDataChannel(&FallbackChannel);
	
	// just creating an archive and testing initial values.
	StartTest("initial archive values");
	{
		Arxx::Archive Archive;
		
		EqualTest(Archive.begin(), Archive.end());
		EqualTest(Archive.pGetRootItem(), static_cast< Arxx::Item * >(0));
		EqualTest(Archive.pGetItem(0x03460), static_cast< Arxx::Item * >(0));
		UnequalTest(Archive.pGetItemFactory(), static_cast< Arxx::ItemFactory * >(0));
	}
	EndTest();
	
	// just creating an item and testing initial values.
	StartTest("initial item values");
	{
		Arxx::Item Item;
		
		EqualTest(Item.u4GetType(), static_cast< Arxx::u4byte >(0xFFFFFFFF));
		EqualTest(Item.u4GetSubType(), static_cast< Arxx::u4byte >(0xFFFFFFFF));
		EqualTest(Item.u4GetVersion(), static_cast< Arxx::u4byte >(0x00000000));
		EqualTest(Item.sGetName(), std::string(""));
		EqualTest(Item.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(Item.GetCompression(), Arxx::Data::NONE);
		EqualTest(Item.u4GetDecompressedLength(), static_cast< Arxx::u4byte >(0));
		EqualTest(Item.u4GetCompressedLength(), static_cast< Arxx::u4byte >(0));
		EqualTest(Item.bIsCompressed(), false);
		EqualTest(Item.pGetArchive(), reinterpret_cast< Arxx::Archive * >(0));
		EqualTest(Item.pGetItemFactory(), reinterpret_cast< Arxx::ItemFactory * >(0));
		EqualTest(Item.bIsExternal(), false);
		EqualTest(Item.bIsInternal(), true);
		EqualTest(Item.GetURI().bIsValid(), false);
		EqualTest(Item.bIsFetched(), true);
	}
	EndTest();
	
	StartTest("creating ItemReferences");
	{
		Arxx::Item Item1(0x23);
		Arxx::Reference Reference1(Item1);
		
		EqualTest(Reference1.pGetItem(), &Item1);
		EqualTest(Reference1.u4GetUniqueID(), Item1.u4GetUniqueID());
		EqualTest(Reference1.u4GetReferenceCount(), static_cast < Arxx::u4byte >(1));
		
		{
			Arxx::Reference Reference2(Reference1);
			
			EqualTest(Reference2.pGetItem(), &Item1);
			EqualTest(Reference2.u4GetUniqueID(), Item1.u4GetUniqueID());
			EqualTest(Reference2.u4GetReferenceCount(), static_cast < Arxx::u4byte >(2));
			
			Arxx::Reference Reference3(Reference1);
			
			EqualTest(Reference2.u4GetReferenceCount(), static_cast < Arxx::u4byte >(3));
		}
		EqualTest(Reference1.u4GetReferenceCount(), static_cast < Arxx::u4byte >(1));
	}
	EndTest();
	
	StartTest("creating more ItemReferences");
	{
		{
			Arxx::Reference Reference1(0x345);
			
			EqualTest(Reference1.u4GetUniqueID(), static_cast< Arxx::u4byte >(0x345));
			
			Arxx::Item Item(0x456);
			Arxx::Reference Reference2(Item);
			
			EqualTest(Reference1.u4GetUniqueID(), static_cast< Arxx::u4byte >(0x345));
		}
	}
	EndTest();
	
	StartTest("re-registering an item")
	{
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive);
		Arxx::Item Item2;
		
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(1));
		TraceException(Archive.vUnregisterItem(Item1));
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(0));
		TraceException(Archive.vRegisterItem(Item1));
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(1));
		TraceException(Archive.vRegisterItem(Item2));
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(2));
		TraceException(Archive.vUnregisterItem(Item2));
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(1));
		TraceException(Archive.vRegisterItem(Item2));
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(2));
	}
	EndTest();
	
	StartTest("changing UIDs of items")
	{
		Arxx::Archive Archive;
		Arxx::Item Item(Archive, 0x34);
		
		Item.vSetUniqueID(0x56);
		EqualTest(Item.u4GetUniqueID(), static_cast< Arxx::u4byte >(0x56));
	}
	EndTest();
	
	StartTest("one Reference with external binding")
	{
		Arxx::Archive Archive;
		Arxx::Reference Reference(Archive.GetReference(0x123));
		Arxx::Item Item(Archive, 0x123);
	}
	EndTest();
	
	StartTest("ItemReferences in Libraries")
	{
		StartBlock("Static Frame");
		Arxx::Archive Archive;
		Arxx::Reference Reference1(Archive.GetReference(0x56));
		
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(1));
		EqualTest(Archive.size(), static_cast< Arxx::Archive::size_type >(0));
		EqualTest(Reference1.u4GetReferenceCount(), static_cast< Arxx::u4byte >(2));
		EqualTest(Reference1.pGetItem(), static_cast< Arxx::Item * >(0));
		EqualTest(Reference1.u4GetUniqueID(), static_cast< Arxx::u4byte >(0x56));
		
		Arxx::Item Item0(Archive, 0x98);
		
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(2));
		EqualTest(Archive.size(), static_cast< Arxx::Archive::size_type >(1));
		
		Arxx::Item Item1(Archive, 0x56);
		
		EqualTest(Reference1.pGetItem(), static_cast< Arxx::Item * >(&Item1));
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(2));
		EqualTest(Archive.size(), static_cast< Arxx::Archive::size_type >(2));
		EndBlock();
	}
	EndTest();
	
	StartTest("archives hold unresolved references only if not alone")
	{
		Arxx::Archive Archive;
		
		{
			Arxx::Reference Reference1(Archive.GetReference(0x98));
			
			// Reference1 and Archive both hold a reference
			EqualTest(Reference1.u4GetReferenceCount(), static_cast< Arxx::u4byte >(2));
			EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(1));
		}
		// since Reference1 is destroyed it releases a reference
		// since the reference is unresolved it is deleted from the Archive as well
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(0));
	}
	EndTest();
	
	StartTest("unregistering an item unresolves other references");
	{
		Arxx::Archive Archive;
		Arxx::Reference Reference1(Archive.GetReference(0x456));
		
		{
			Arxx::Item Item1(Archive, 0x456);
			
			EqualTest(Reference1.pGetItem(), &Item1);
		}
		EqualTest(Reference1.pGetItem(), static_cast< Arxx::Item * >(0));
	}
	EndTest();
	
	StartTest("closing an archive with a resolved reference")
	{
		StartBlock("StaticFrame");
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive, 0x123);
		
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(1));
		EqualTest(Archive.size(), static_cast< Arxx::Archive::size_type >(1));
		Archive.vClose();
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(0));
		EqualTest(Archive.size(), static_cast< Arxx::Archive::size_type >(0));
		
		Arxx::Item Item2(Archive, 0x123);
		
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(1));
		EqualTest(Archive.size(), static_cast< Arxx::Archive::size_type >(1));
		
		Arxx::Reference Reference(Archive.GetReference(0x123));
		
		EqualTest(Reference.u4GetReferenceCount(), static_cast< Arxx::u4byte >(2));
		EqualTest(Archive.u4GetNumberOfReferences(), static_cast< Arxx::u4byte >(1));
		EqualTest(Archive.size(), static_cast< Arxx::Archive::size_type >(1));
		Archive.vClose();
		EqualTest(Reference.u4GetReferenceCount(), static_cast< Arxx::u4byte >(1));
		EqualTest(Reference.pGetItem(), static_cast< Arxx::Item * >(0));
		EndBlock();
	}
	EndTest();
	
	// vInternalizeData() should not change anything if data is already fetched and internal.
	StartTest("trying to internalize new item data")
	{
		Arxx::Item Item;
		
		EqualTest(Item.bIsInternal(), true);
		EqualTest(Item.bIsFetched(), true);
		Item.vInternalize();
		EqualTest(Item.bIsInternal(), true);
		EqualTest(Item.bIsFetched(), true);
	}
	EndTest();
	
	// one archive and one item and testing for including and excluding.
	StartTest("static including and excluding");
	{
		Arxx::Item Item1;
		{
			Arxx::Archive Archive;
			
			EqualTest(Archive.begin(), Archive.end());
			EqualTest(Archive.size(), static_cast< size_t >(0));
			
			// adding the item from outer frame
			TraceException(Archive.vRegisterItem(Item1));
			UnequalTest(Archive.begin(), Archive.end());
			EqualTest(Archive.size(), static_cast< size_t >(1));
			EqualTest(&(*(Archive.begin())), &Item1);
			EqualTest(Item1.pGetArchive(), &Archive);
			// adding a stacked item
			{
				Arxx::Item Item2(Archive);
				
				UnequalTest(Archive.begin(), Archive.end());
				EqualTest(Archive.size(), static_cast< size_t >(2));
				EqualTest(Item2.pGetArchive(), &Archive);
			}
			// everything's back to normal
			UnequalTest(Archive.begin(), Archive.end());
			EqualTest(Archive.size(), static_cast< size_t >(1));
			EqualTest(Item1.pGetArchive(), &Archive);
			// removing the item from outer frame
			Archive.vUnregisterItem(Item1);
			EqualTest(Archive.begin(), Archive.end());
			EqualTest(Archive.size(), static_cast< size_t >(0));
		}
		EqualTest(Item1.pGetArchive(), reinterpret_cast< Arxx::Archive * >(0));
	}
	EndTest();
	
	// setting external data to 0
	StartTest("setting external data to 0");
	{
		Arxx::Item Item;
		
		Item.vSetExternal();
	}
	EndTest();
	
	// one archive and one item and testing the saving with a bool
	StartTest("saving an archive with a single bool item")
	{
		Arxx::Archive Archive;
		Arxx::Item Item(Archive);
		Arxx::BufferWriter ItemWriter(Item);
		
		ItemWriter << true;
		Item.vSetName("Testing bools");
		Archive.vSave("file:///tmp/Test.arx");
		Archive.vSave("file:Test1.arx");
	}
	EndTest();
	
	// one archive and one item and testing the saving with a bool
	StartTest("saving an archive with a single float item")
	{
		Arxx::Archive Archive;
		Arxx::Item Item(Archive);
		Arxx::BufferWriter ItemWriter(Item);
		
		ItemWriter << 4.0f;
		Item.vSetName("a float");
		Archive.vSave("file:Test2.arx");
	}
	EndTest();
	
	// loading from a relative URL
	StartTest("loading an archive from \"file:Test.arx\"")
	{
		Arxx::Archive Archive;
		bool bLoadSuccess(Archive.bLoad("file:Test1.arx"));
		
		EqualTest(bLoadSuccess, true);
	}
	EndTest();
	
	// one archive and two items; saving a string and an u4byte
	StartTest("saving one archive with two items")
	{
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive);
		Arxx::BufferWriter ItemWriter1(Item1);
		
		ItemWriter1 << static_cast< Arxx::u4byte >(111111);
		Item1.vSetName("width");
		
		Arxx::Item Item2(Archive);
		Arxx::BufferWriter ItemWriter2(Item2);
		
		ItemWriter2 << "Save game ... ";
		Item2.vSetName("string20342");
		Item2.vSetUniqueID(20342);
		Archive.vSave("file:///tmp/Test2.arx");
	}
	EndTest();
	
	// one archive and reading from Test.arx
	// many many strange testing which all yielded an error at a time :|
	StartTest("reading from a file")
	{
		Arxx::Archive Archive;
		
		Archive.bLoad("file:///tmp/Test.arx");
		UnequalTest(Archive.begin(), Archive.end());
		
		Arxx::Item & Item(*(Archive.begin()));
		
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(Item.u4GetDecompressedLength(), static_cast< Arxx::u4byte >(1));
		EqualTest(Item.sGetName(), std::string("Testing bools"));
		EqualTest(Item.bIsCompressed(), false);
		
		bool bB(false);
		
		Item.bFetch();
		EqualTest(Item.stGetLength(), static_cast< Arxx::Buffer::size_type >(1));
		
		Arxx::BufferReader ItemReader(Item);
		
		ItemReader >> bB;
		EqualTest(bB, true);
	}
	EndTest();
	
	// one archive and reading from Test2.arx
	StartTest("reading from a file")
	{
		Arxx::Archive Archive;
		
		Archive.bLoad("Test2.arx");
		UnequalTest(Archive.begin(), Archive.end());
		
		Arxx::Item & Item(*(Archive.begin()));
		
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.stGetLength(), static_cast< Arxx::Buffer::size_type >(0));
		EqualTest(Item.u4GetDecompressedLength(), static_cast< Arxx::u4byte >(4));
		EqualTest(Item.sGetName(), std::string("a float"));
		EqualTest(Item.bIsCompressed(), false);
		
		float fValue(0.0f);
		
		Item.bFetch();
		EqualTest(Item.stGetLength(), static_cast< Arxx::Buffer::size_type >(sizeof(float)));
		
		Arxx::BufferReader ItemReader(Item);
		
		ItemReader >> fValue;
		EqualTest(fValue, 4.0f);
	}
	EndTest();
	
	// reading Test.arx and changing the uid of an item before its external data is read
	// this yielded an error while the external data was indirectly related to the item only.
	StartTest("changing unique ids observing external data persistence");
	{
		Arxx::Archive Archive;
		
		Archive.bLoad("file:///tmp/Test.arx");
		
		Arxx::Archive::iterator iItem(Archive.begin());
		
		UnequalTest(iItem, Archive.end());
		
		Arxx::Item & Item(*iItem);
		
		UnequalTest(Item.u4GetUniqueID(), static_cast< Arxx::u4byte >(20));
		Item.vSetUniqueID(20);
		EqualTest(Item.u4GetUniqueID(), static_cast< Arxx::u4byte >(20));
		
		bool bB(false);
		
		Item.bFetch();
		
		Arxx::BufferReader ItemReader(Item);
		ItemReader >> bB;
		EqualTest(bB, true);
	}
	EndTest();
	
	// reading Test2.arx
	// compressing an item and saving back
	StartTest("saving compressed items");
	{
		Arxx::Archive Archive;
		
		Archive.bLoad("file:///tmp/Test2.arx");
		
		Arxx::Item * pItem(Archive.pGetItem(20342));
		
		UnequalTest(pItem, reinterpret_cast< Arxx::Item * >(0));
		EqualTest(pItem->u4GetDecompressedLength(), static_cast< Arxx::u4byte >(15));
		EqualTest(pItem->u4GetCompressedLength(), static_cast< Arxx::u4byte >(0));
		EqualTest(pItem->bIsCompressed(), false);
		pItem->bFetch();
		pItem->vCompress();
		if(Arxx::Data::m_DefaultCompression != Arxx::Data::NONE)
		{
			EqualTest(pItem->bIsCompressed(), true);
			UnequalTest(pItem->u4GetCompressedLength(), static_cast< Arxx::u4byte >(0));
			EqualTest(pItem->stGetLength(), pItem->u4GetCompressedLength());
		}
		EqualTest(pItem->u4GetDecompressedLength(), static_cast< Arxx::u4byte >(15));
		EqualTest(pItem->bIsInternal(), true);
		Archive.vSave("file:///tmp/Test2.arx");
	}
	EndTest();
	
	// reading Test2.arx
	StartTest("file reading and decompressing");
	{
		Arxx::Archive Archive;
		
		Archive.bLoad("file:///tmp/Test2.arx");
		
		Arxx::Item * pItem(Archive.pGetItem(20342));
		
		UnequalTest(pItem, static_cast< Arxx::Item * >(0));
		EqualTest(pItem->bIsInternal(), true);
		EqualTest(pItem->sGetName(), std::string("string20342"));
		if(Arxx::Data::m_DefaultCompression != Arxx::Data::NONE)
		{
			UnequalTest(pItem->u4GetCompressedLength(), static_cast< Arxx::u4byte >(0));
		}
		EqualTest(pItem->u4GetDecompressedLength(), static_cast< Arxx::u4byte >(std::string("Save game ... ").length() + 1)); // consider ending '\0'
		pItem->bFetch();
		TraceException(pItem->vDecompress());
		
		std::stringstream ssString;
		
		ssString << *pItem;
		// the stream contains the \0 from the string and from itself (*string*stream)
		// so we have to substract 1 from the length and take a substring going to the char before the last.
		EqualTest(ssString.str().length() - 1, std::string("Save game ... ").length());
		EqualTest(ssString.str().substr(0, ssString.str().length() - 1), std::string("Save game ... "));
	}
	EndTest();
	
	// reading Test2.arx
	// setting and reading a version number
	// saving and loading items with version number
	StartTest("saving and loading of version numbers on items");
	{
		{
			Arxx::Archive Archive;
			
			Archive.bLoad("file:///tmp/Test2.arx");
			
			Arxx::Item * pItem(Archive.pGetItem(20342));
			
			UnequalTest(pItem->u4GetVersion(), static_cast< Arxx::u4byte >(0x01020304));
			pItem->vSetVersion(0x01020304);
			EqualTest(pItem->u4GetVersion(), static_cast< Arxx::u4byte >(0x01020304));
			Archive.vSave("file:///tmp/Test2.arx");
		}
		{
			Arxx::Archive Archive;
			
			Archive.bLoad("file:///tmp/Test2.arx");
			
			Arxx::Item * pItem(Archive.pGetItem(20342));
			
			EqualTest(pItem->u4GetVersion(), static_cast< Arxx::u4byte >(0x01020304));
		}
	}
	EndTest();
	
	// using a custom item factory
	StartTest("custom item factory");
	{
		CustomItemFactory ItemFactory;
		
		Arxx::Item * pItem(ItemFactory.pNewItem());
		
		pItem->vSetName("Test");
		EqualTest(ItemFactory.stGetCount(), static_cast< size_t >(1));
		{
			Arxx::Archive Archive(ItemFactory);
			
			Archive.bLoad("file:///tmp/Test2.arx");
			EqualTest(Archive.size(), static_cast< size_t >(2));
			EqualTest(ItemFactory.stGetCount(), static_cast< size_t >(3));
			Archive.vRegisterItem(*pItem);
			EqualTest(Archive.size(), static_cast< size_t >(3));
			EqualTest(ItemFactory.stGetCount(), static_cast< size_t >(3));
			EqualTest(pItem->pGetArchive(), &Archive);
			Archive.vUnregisterItem(*pItem);
			EqualTest(Archive.size(), static_cast< size_t >(2));
			EqualTest(ItemFactory.stGetCount(), static_cast< size_t >(3));
			EqualTest(pItem->pGetArchive(), static_cast< Arxx::Archive * >(0));
			// re-register the item to see whether destroying the archive
			// invalidates the item's pointer to the archive correctly
			TraceException(Archive.vRegisterItem(*pItem));
		}
		// the archive deletes them all
		EqualTest(ItemFactory.stGetCount(), static_cast< size_t >(0));
	}
	EndTest();
	
	// simple memory merge
	StartTest("memory merge");
	{
		Arxx::Archive Archive1;
		Arxx::Item Item1(Archive1, 0x123);
		Arxx::BufferWriter ItemWriter1(Item1);
		
		ItemWriter1 << static_cast< Arxx::u4byte >(123456);
		
		Arxx::Archive Archive2;
		Arxx::Item Item2(Archive2, 0x1234);
		Arxx::Item Item3(Archive2, 0x123);
		Arxx::BufferWriter ItemWriter2(Item2);
		Arxx::BufferWriter ItemWriter3(Item3);
		
		ItemWriter2 << static_cast< Arxx::u4byte >(654321);
		ItemWriter2 << static_cast< Arxx::u4byte >(789432);
		ItemWriter3 << "Hallo";
		{
			std::vector< Arxx::Rule * > Rules;
			
			Rules.push_back(&Dropper);
			Rules.push_back(&Adder);
			Archive1.bMerge(Archive2, Rules);
			EqualTest(Archive1.size(), static_cast< Arxx::Archive::size_type >(2));
			EqualTest(Archive2.size(), static_cast< Arxx::Archive::size_type >(0));
			EqualTest(Archive1.pGetItem(0x123)->u4GetDecompressedLength(), static_cast< Arxx::u4byte >(4));
			EqualTest(Archive1.pGetItem(0x1234)->u4GetDecompressedLength(), static_cast< Arxx::u4byte >(8));
		}
	}
	EndTest();
	
	// merging from two different files
	// saving to another one
	StartTest("merging two files and saving to a third");
	{
		Arxx::Archive Archive1;
		
		Archive1.bLoad("file:///tmp/Test.arx");
		
		Arxx::Archive Archive2;
		
		Archive2.bLoad("file:///tmp/Test2.arx");
		
		std::vector< Arxx::Rule * > Rules;
		
		Rules.push_back(&Dropper);
		Rules.push_back(&Adder);
		Archive1.bMerge(Archive2, Rules);
		Archive1.vSave("file:///tmp/MergeTest.arx");
	}
	EndTest()
	
	// one archive and one item and testing for dynamic including and excluding
	StartTest("dynamic including and excluding of items");
	{
		Arxx::Archive Archive;
		{
			Arxx::Item Item;
			
			Item.vSetUniqueID(0x2345);
			EqualTest(Archive.begin(), Archive.end());
			Archive.vRegisterItem(Item);
			UnequalTest(Archive.begin(), Archive.end());
			EqualTest(&(*(Archive.begin())), &Item);
			EqualTest(Item.pGetArchive(), &Archive);
			EqualTest(Item.u4GetUniqueID(), static_cast< Arxx::u4byte >(0x2345));
			Archive.vUnregisterItem(Item);
			EqualTest(Archive.begin(), Archive.end());
			EqualTest(Item.pGetArchive(), static_cast< Arxx::Archive * >(0));
			EqualTest(Item.u4GetUniqueID(), static_cast< Arxx::u4byte >(0x2345));
		}
	}
	EndTest();
	
	// creating an item and setting its external data to a file
	StartTest("setting external data to a file and testing specialized file channel");
	{
		// creating the external data first
		{
			std::ofstream OFStream("Test1.txt");
			
			OFStream << "Test1 is a nice tool to test Arxx integrity." << std::endl;
			OFStream.close();
		}
		
		LocalFileChannel Channel("file:Test1.txt");
		
		Arxx::Repository.bRegisterDataChannel(&Channel);
		{
			// setting the data
			Arxx::Item Item;
			
			Item.vSetExternal("file:Test1.txt");
			EqualTest(Item.u4GetDecompressedLength(), static_cast< Arxx::u4byte >(0));
			EqualTest(Item.u4GetCompressedLength(), static_cast< Arxx::u4byte >(0));
		}
		{
			// getting the data
			Arxx::Item Item;
			
			Item.vSetExternal("file:Test1.txt");
			Item.bFetch();
			EqualTest(Item[3], static_cast< unsigned char >('t'));
			EqualTest(Item[4], static_cast< unsigned char >('1'));
		}
		Arxx::Repository.bUnregisterDataChannel(&Channel);
	}
	EndTest();
	
	// merging libraries from files and closing one file before saving the merged result
	StartTest("merging libraries from files");
	{
		Arxx::Archive Archive1;
		
		Archive1.bLoad("file:///tmp/Test.arx");
		{
			Arxx::Archive Archive2;
			
			Archive2.bLoad("file:///tmp/Test2.arx");
			
			std::vector< Arxx::Rule * > Rules;
			
			Rules.push_back(&Dropper);
			Rules.push_back(&Adder);
			Archive1.bMerge(Archive2, Rules);
		}
		Archive1.vSave("file:///tmp/MergeTest1.arx");
	}
	EndTest();
	
	// trying out the structure. very basic
	StartTest("basic structure testing");
	{
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive, 12);
		Arxx::Item Item2(Archive, 13);
		
		Item1.GetStructure().bAdd(Item2.u4GetUniqueID());
		EqualTest(Item1.GetStructure().bHasRelation("child"), true);
		EqualTest(Item1.GetStructure().GetRelation("child").size(), static_cast< size_t >(1));
		Archive.vSave("file:///tmp/StructureTest1.arx");
		Item1.GetStructure().bDelete(Item2.u4GetUniqueID());
		EqualTest(Item1.GetStructure().bHasRelation("child"), false);
	}
	EndTest();
	
	// loading structure
	StartTest("loading structure");
	{
		Arxx::Archive Archive;
		
		Archive.bLoad("file:///tmp/StructureTest1.arx");
		
		Arxx::Item * pItem1(Archive.pGetItem(12));
		Arxx::Item * pItem2(Archive.pGetItem(13));
		
		UnequalTest(pItem1, static_cast< Arxx::Item * >(0));
		UnequalTest(pItem2, static_cast< Arxx::Item * >(0));
		EqualTest(pItem1->GetStructure().GetRelation("child").size(), static_cast< size_t >(1));
		EqualTest(pItem1->GetStructure().GetRelation("child").size(), static_cast< size_t >(1));
		
		Archive.vSave("file:///tmp/StructureTest1.arx");
	}
	EndTest();
	
	// deleting unresolved items
	StartTest("testin non-child and empty relations")
	{
		{
			Arxx::Archive Archive;
			Arxx::Item Item1(Archive, 1);
			Arxx::Item Item2(Archive, 2);
			
			Item1.GetStructure().bAdd(Item2.u4GetUniqueID(), "test");
			Archive.vSave("file:///tmp/StructureTest2.arx");
		}
		{
			Arxx::Archive Archive;
			
			Archive.bLoad("file:///tmp/StructureTest2.arx");
			
			Arxx::Item * pItem1(Archive.pGetItem(1));
			Arxx::Item * pItem2(Archive.pGetItem(2));
			
			UnequalTest(pItem1, static_cast< Arxx::Item * >(0));
			UnequalTest(pItem2, static_cast< Arxx::Item * >(0));
			EqualTest(pItem1->GetStructure().bHasRelation("test"), true);
			pItem1->GetStructure().bDelete(pItem2->u4GetUniqueID(), "test");
			EqualTest(pItem1->GetStructure().bHasRelation("test"), false);
		}
	}
	EndTest();
	
	// structure tests
	StartTest("adding unresolved item references");
	{
		Arxx::Item Item;
		
		Item.GetStructure().bAdd(0x123, "reference");
		EqualTest(Item.GetStructure().bHasRelation("reference"), false);
		Item.GetStructure().bAdd(Arxx::g_u4InvalidID, "test");
		EqualTest(Item.GetStructure().bHasRelation("test"), false);
		Item.GetStructure().bAdd(0x123, "test");
		EqualTest(Item.GetStructure().bHasRelation("test"), true);
	}
	EndTest();
	
	StartTest("accessing a reference that just got valid")
	{
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive, 0x23);
		
		Item1.GetStructure().bAdd(0x67);
		
		Arxx::Item Item2(Archive, 0x67);
		const Arxx::Structure::Relation & Relation(Item1.GetStructure().GetRelation("child"));
		
		EqualTest(Relation.size(), static_cast< size_t >(1));
		EqualTest(Relation.begin()->u4GetUniqueID(), static_cast< Arxx::u4byte >(0x67));
		EqualTest(Relation.begin()->pGetItem(), const_cast< const Arxx::Item * >(&Item2));
	}
	EndTest();
	
	StartTest("accessing a reference on an item that was just registered")
	{
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive, 0x67);
		Arxx::Item Item2(0x23);
		
		Item2.GetStructure().bAdd(0x67);
		Archive.vRegisterItem(Item2);
		
		const Arxx::Structure::Relation & Relation(Item2.GetStructure().GetRelation("child"));
		
		EqualTest(Relation.size(), static_cast< Arxx::Structure::Relation::size_type >(1));
		EqualTest(Relation.begin()->u4GetUniqueID(), static_cast< Arxx::u4byte >(0x67));
		EqualTest(Relation.begin()->pGetItem(), const_cast< const Arxx::Item * >(&Item1));
	}
	EndTest();
	
	StartTest("unresolving due to unregistering");
	{
		Arxx::Archive Archive;
		Arxx::Reference Reference(Archive.GetReference(0x987));
		{
			Arxx::Item Item(Archive, 0x987);
			
			EqualTest(Reference.pGetItem(), &Item);
		}
		EqualTest(Reference.pGetItem(), reinterpret_cast< Arxx::Item * >(0));
	}
	EndTest();
	
	StartTest("unresolving relation references due to unregistering other items");
	{
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive, 0x12);
		
		EqualTest(Item1.GetStructure().bAdd(0x23), true);
		{
			Arxx::Item Item2(Archive, 0x23);
			
			EqualTest(Item1.GetStructure().GetRelation("child").begin()->pGetItem(), &Item2);
		}
		EqualTest(Item1.GetStructure().GetRelation("child").begin()->pGetItem(), reinterpret_cast< Arxx::Item * >(0));
	}
	EndTest();
	
	StartTest("unresolving relation references due to unregistering self");
	{
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive, 0x12);
		
		EqualTest(Item1.GetStructure().bAdd(0x23), true);
		
		Arxx::Item Item2(Archive, 0x23);
		
		EqualTest(Item1.GetStructure().GetRelation("child").begin()->pGetItem(), &Item2);
		Archive.vUnregisterItem(Item1);
		EqualTest(Item1.GetStructure().GetRelation("child").begin()->pGetItem(), reinterpret_cast< Arxx::Item * >(0));
	}
	EndTest();
	
	LocalFileChannel FileChannel("file:///tmp/StructureTest2.arx");
	
	Arxx::Repository.bRegisterDataChannel(&FileChannel);
	
	// setting the same external data twice but internalizing once
	StartTest("internal and external data from same source file");
	{
		Arxx::Archive Archive;
		Arxx::Item Item1(Archive, 0x00000001);
		Arxx::Item Item2(Archive, 0x00000002);
		Arxx::Item Item3(Archive, 0x00000003);
		
		// setting the same data
		Item1.vSetExternal("file:///tmp/StructureTest2.arx");
		Item2.vSetExternal("file:///tmp/StructureTest2.arx");
		Item3.vSetExternal("file:///tmp/StructureTest2.arx");
		EqualTest(Item1.bIsExternal(), true);
		EqualTest(Item2.bIsExternal(), true);
		EqualTest(Item3.bIsExternal(), true);
		EqualTest(Item1.bIsFetched(), false);
		EqualTest(Item2.bIsFetched(), false);
		EqualTest(Item3.bIsFetched(), false);
		EqualTest(Item1.GetURI().sGetURI(), std::string("file:///tmp/StructureTest2.arx"));
		EqualTest(Item2.GetURI().sGetURI(), std::string("file:///tmp/StructureTest2.arx"));
		EqualTest(Item3.GetURI().sGetURI(), std::string("file:///tmp/StructureTest2.arx"));
		// internalizing twice
		Item2.vInternalize();
		Item3.vInternalize();
		EqualTest(Item1.bIsExternal(), true);
		EqualTest(Item2.bIsExternal(), false);
		EqualTest(Item3.bIsExternal(), false);
		EqualTest(Item1.bIsFetched(), false);
		EqualTest(Item2.bIsFetched(), false);
		EqualTest(Item3.bIsFetched(), false);
		// prefetching once
		EqualTest(Item2.bFetch(), true);
		UnequalTest(Item2.u4GetDecompressedLength(), static_cast< Arxx::u4byte >(0));
		EqualTest(Item1.bIsExternal(), true);
		EqualTest(Item2.bIsExternal(), false);
		EqualTest(Item3.bIsExternal(), false);
		EqualTest(Item1.bIsFetched(), false);
		EqualTest(Item2.bIsFetched(), true);
		EqualTest(Item3.bIsFetched(), false);
		//saving archive should fetch the other one as well
		Archive.vSave("file:///tmp/DataTest.arx");
	}
	EndTest();
	
	StartTest("testing sameness of internal and external data");
	{
		Arxx::Archive Archive;
		
		Archive.bLoad("file:///tmp/DataTest.arx");
		
		Arxx::Item * pItem1(Archive.pGetItem(0x00000001));
		Arxx::Item * pItem2(Archive.pGetItem(0x00000002));
		Arxx::Item * pItem3(Archive.pGetItem(0x00000003));
		
		EqualTest(pItem1->bIsExternal(), true);
		EqualTest(pItem2->bIsExternal(), false);
		EqualTest(pItem3->bIsExternal(), false);
		EqualTest(pItem1->bIsFetched(), false);
		EqualTest(pItem2->bIsFetched(), false);
		EqualTest(pItem3->bIsFetched(), false);
		EqualTest(pItem1->GetURI().sGetURI(), std::string("file:///tmp/StructureTest2.arx"));
		pItem1->bFetch();
		pItem2->bFetch();
		pItem3->bFetch();
		EqualTest(pItem1->u4GetDecompressedLength(), pItem2->u4GetDecompressedLength());
		EqualTest(pItem2->u4GetDecompressedLength(), pItem3->u4GetDecompressedLength());
		
		std::string sHostname1;
		std::string sHostname2;
		std::string sHostname3;
		Arxx::BufferReader ItemReader1(*pItem1);
		Arxx::BufferReader ItemReader2(*pItem2);
		Arxx::BufferReader ItemReader3(*pItem3);
		
		ItemReader1 >> sHostname1;
		ItemReader2 >> sHostname2;
		ItemReader3 >> sHostname3;
		EqualTest(sHostname1, sHostname2);
		EqualTest(sHostname2, sHostname3);
	}
	EndTest();
	
	StartTest("loading and saving local files with relative paths")
	{
		std::string sTestString("test string that is the item's data.");
		
		{
			// prepare
			Arxx::Archive Archive;
			Arxx::Item Item(Archive, 12);
			Arxx::BufferWriter BufferWriter(Item);
			
			BufferWriter << sTestString;
			
			Archive.vSave("SetUIDTest.arx");
		}
		{
			// load and change
			Arxx::Archive Archive;
			
			EqualTest(Archive.bLoad("SetUIDTest.arx"), true);
			Archive.vSave("SetUIDTest.arx");
		}
		{
			// verify
			Arxx::Archive Archive;
			
			EqualTest(Archive.bLoad("SetUIDTest.arx"), true);
			
			Arxx::Item * pItem(Archive.pGetItem(12));
			
			UnequalTest(pItem, static_cast< Arxx::Item * >(0));
			EqualTest(pItem->u4GetDecompressedLength(), static_cast< Arxx::u4byte >(sTestString.length() + 1));
		}
	}
	EndTest();
	
	Arxx::Repository.bUnregisterDataChannel(&FileChannel);
	Arxx::Repository.bUnregisterDataChannel(&FallbackChannel);
	
	return iEndTests();
}
