/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_LOCALARCHIVECHANNEL_H
#define ARXX_LOCALARCHIVECHANNEL_H

#include <fstream>

#include "../Include/DataChannel.h"

/**
 * @file LocalArchiveChannel.h
 * 
 * This file declares the Arxx::LocalArchiveChannel class, which is responsible for fetching data from a local ARX archive file.
 **/

namespace Arxx
{
	/**
	 * @brief An class to fetch data from a local ARX archive file.
	 **/
	class LocalArchiveChannel : public Arxx::DataChannel
	{
	public:
		/**
		 * @brief The standard constructor for LocalArchiveChannel.
		 * @param URI The URI of the local ARX archive file.
		 * 
		 * This creates a LocalArchiveChannel object with zero references.
		 * 
		 * The constructor opens the file.
		 **/
		LocalArchiveChannel(const Arxx::URI & URI);
		
		/**
		 * @brief The standard deonstructor for LocalArchiveChannel.
		 * 
		 * Only here to help debugging dangeling library sources.
		 **/
		virtual ~LocalArchiveChannel(void);
		
		/**
		 * @brief Returns the associated stream object.
		 * 
		 * This function is called by the Archive after the stream has been opened to retrieve the library and item headers.
		 **/
		std::istream & GetStream(void);
		
		/**
		 * @brief Calling this function will set the data offset to the current file position.
		 * 
		 * The current stream position will be written in the member variable @a m_u4DataOffset. Every fetch will seek to this offset plus the value of the "Offset" attribute in the sURI passed to vFetchData().
		 * 
		 * @sa vFetchData()
		 **/
		void vMarkDataOffset(void);
		
		/**
		 * @brief This abstract function is responsible for retrieving the data and writing it into a Arxx::Item.
		 * @param URI The URI of the data chunk in the local library to be retrieved by the data channel.
		 * @param Buffer The Arxx::Item which will be used as the target for the retrieved data.
		 * @param FetchStatus The fetcher object will update this variable according to the current status of the fetching process.
		 * @return A boolean value indicating whether the fetch was successful.
		 **/
		virtual bool bFetchData(const Arxx::URI & URI, Arxx::Buffer & Buffer, Arxx::FetchStatus & FetchStatus);
	protected:
		/**
		 * @brief Function that is called every time the reference counter goes to zero.
		 * 
		 * The function closes the stream object.
		 **/
		void vOnUnreferenced(void);
	private:
		
		/**
		 * @brief The associated file stream.
		 * 
		 * This stream is opened with the constructor and closed in vOnUnreferenced().
		 **/
		std::ifstream m_IStream;
		
		/**
		 * @brief The file offset of the data block.
		 * 
		 * Using this Offset the fetcher seeks to the correct position in the file to read data on vFetchData() calls.
		 **/
		Arxx::u4byte m_u4DataOffset;
	};
}

#endif
