/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_DEFAULTITEMFACTORY_H
#define ARXX_DEFAULTITEMFACTORY_H

#include "../Include/Common.h"
#include "../Include/ItemFactory.h"

/**
 * @file DefaultItemFactory.h
 * 
 * This file declares the Arxx::DefaultItemFactory class, the default Arxx::ItemFactory implementation that is used everytime no other Arxx::ItemFactory has been specified.
 **/

namespace Arxx
{
	class Item;
	
	class Archive;
	
	/**
	 * @brief The standard ItemFactory created by every library that has no other ItemFactory set.
	 * 
	 * The default ItemFactory is only a wrapper object for new and delete and serves as the most basic and simple ItemFactory.
	 * 
	 * There is no need to ever create a DefaultItemFactory, since a global stack object is included in the object file. If no ItemFactory is forced to be used this global factory will be chosen automatically for you.
	 * 
	 * @note As this global default DefaultItemFactory is a stack object it satisfies the lifepspan requirement of ItemFactories.
	 * 
	 * @sa Arxx::ItemFactory
	 **/
	class DefaultItemFactory : public Arxx::ItemFactory
	{
	public:
		/**
		 * @brief The default constructor fot a DefaultItemFactory.
		 * 
		 * The default constructor does no valuable work and is only implemented for debugging and logging support.
		 **/
		DefaultItemFactory(void);
		
		/**
		 * @brief The default destructor fot a DefaultItemFactory.
		 * 
		 * The default destructor does no valuable work and is only implemented for debugging and logging support.
		 **/
		virtual ~DefaultItemFactory(void);
		
		/**
		 * @brief Creates a new Arxx::Item.
		 * @param u4UniqueID The unique ID that the item should be created with.
		 * 
		 * This function creates a new Arxx::Item and does nothing else. The item is created via new and no reference counting or any other form of keeping track is performed yet.
		 * 
		 * The call that is performed here is the Arxx::Item default constructor.
		 *
		 * @note Note that the pointer returned from this function is the only reference to the newly created item so you better save it somewhere.
		 **/
		virtual Arxx::Item * pNewItem(Arxx::u4byte u4UniqueID = Arxx::g_u4InvalidID);
		
		/**
		 * @brief Creates a new Arxx::Item.
		 * @param Archive A reference to an Archive which the item is to be registered to.
		 * @param u4UniqueID The unique ID that the item should be created with.
		 * 
		 * This function creates a new Arxx::Item and registers it at the @a Archive. The Item is created via new and no reference counting or any other form of keeping track is performed yet.
		 * 
		 * The call that is performed here is the Arxx::Item constructor with an Archive and the unique ID.
		 * 
		 * @sa Arxx::Library::vRegisterItem().
		 **/
		virtual Arxx::Item * pNewItem(Arxx::Archive & Archive, Arxx::u4byte u4UniqueID = Arxx::g_u4InvalidID);
		
		/**
		 * @brief Destroys a Arxx::Item.
		 * 
		 * This function is a wrapper for delete and nothing else. It deletes the item @a pItem.
		 **/
		virtual void vDeleteItem(Arxx::Item * pItem);
	};
}

#endif
