/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include "../config.h"
#include "../Include/Data.h"
#include "../Include/DataRepository.h"

/// optional headers
#ifdef HAVE_ZLIB_H
#include <zlib.h>
#endif

#ifdef HAVE_BZLIB_H
#include <bzlib.h>
#endif

/// default compression
#ifdef HAVE_BZLIB_H
Arxx::Data::Compression Arxx::Data::m_DefaultCompression = BZLIB;
#elif HAVE_ZLIB_H
Arxx::Data::Compression Arxx::Data::m_DefaultCompression = ZLIB_9;
#else
Arxx::Data::Compression Arxx::Data::m_DefaultCompression = NONE;
#endif

Arxx::Data::Data(void) :
	m_Compression(Arxx::Data::NONE),
	m_u4DecompressedLength(0),
	m_u4CompressedLength(0),
	m_bIsExternal(false),
	m_FetchStatus(Arxx::FETCHED)
{
}

Arxx::Data::~Data(void)
{
	if(GetURI().bIsValid() == true)
	{
		Arxx::Repository.vDereferenceDataChannel(GetURI());
	}
}

void Arxx::Data::vDecompress(void)
{
	switch(m_Compression)
	{
	case Arxx::Data::NONE:
		{
			break;
		}
	case Arxx::Data::ZLIB_0:
	case Arxx::Data::ZLIB_1:
	case Arxx::Data::ZLIB_2:
	case Arxx::Data::ZLIB_3:
	case Arxx::Data::ZLIB_4:
	case Arxx::Data::ZLIB_5:
	case Arxx::Data::ZLIB_6:
	case Arxx::Data::ZLIB_7:
	case Arxx::Data::ZLIB_8:
	case Arxx::Data::ZLIB_9:
		{
#ifdef HAVE_ZLIB_H
			Arxx::Data::pointer DecompressedData(new Arxx::Data::value_type[m_u4DecompressedLength]);
			z_stream zStream;
			int iReturn = 0;
			
			zStream.zalloc = 0;
			zStream.zfree = 0;
			zStream.next_in = const_cast< Arxx::u1byte * >(GetBegin());
			/** @todo Why not use u4GetCompressedLength() here? **/
			zStream.avail_in = stGetLength();
			zStream.next_out = DecompressedData;
			/** @todo Why not use u4GetDecompressedLength() here? **/
			zStream.avail_out = m_u4DecompressedLength;
			if((iReturn = inflateInit(&zStream)) != Z_OK)
			{
				delete[] DecompressedData;
				/** @todo Correctly specify the actual error that occured in zlib. **/
				throw zlib_error("TODO: specify!");
			}
			if((iReturn = inflate(&zStream, Z_FINISH)) != Z_STREAM_END)
			{
				delete[] DecompressedData;
				/** @todo Correctly specify the actual error that occured in zlib. **/
				throw zlib_error("TODO: specify!");
			}
			if((iReturn = inflateEnd(&zStream)) != Z_OK)
			{
				delete[] DecompressedData;
				/** @todo Correctly specify the actual error that occured in zlib. **/
				throw zlib_error("TODO: specify!");
			}
			vSetLength(0);
			vInsert(0, m_u4DecompressedLength, DecompressedData);
			delete[] DecompressedData;
			m_Compression = Arxx::Data::NONE;
			m_u4CompressedLength = 0;
#else
			throw std::runtime_error("Unsupported decompression method.");
#endif
			
			break;
		}
	case Arxx::Data::BZLIB:
		{
#ifdef HAVE_BZLIB_H
			Arxx::Data::pointer DecompressedData(new Arxx::Data::value_type[m_u4DecompressedLength]);
			bz_stream BZStream;
			
			BZStream.bzalloc = 0;
			BZStream.bzfree = 0;
			BZStream.opaque = 0;
			BZStream.next_in = reinterpret_cast< char * >(const_cast< Arxx::u1byte * >(GetBegin()));
			/** @todo Why not use u4GetCompressedLength() here? **/
			BZStream.avail_in = stGetLength();
			BZStream.next_out = reinterpret_cast< char * >(DecompressedData);
			/** @todo Why not use u4GetDecompressedLength() here? **/
			BZStream.avail_out = m_u4DecompressedLength;
			
			int iBZResult;
			
			if((iBZResult = BZ2_bzDecompressInit(&BZStream, 0, 0)) != BZ_OK)
			{
				delete DecompressedData;
				/** @todo Correctly specify the actual error that occured in bzlib. **/
				throw bzlib_error("TODO: specify!");
			}
			if((iBZResult = BZ2_bzDecompress(&BZStream)) != BZ_STREAM_END)
			{
				delete DecompressedData;
				/** @todo Correctly specify the actual error that occured in bzlib. **/
				throw bzlib_error("TODO: specify!");
			}
			if((iBZResult = BZ2_bzDecompressEnd(&BZStream)) != BZ_OK)
			{
				delete DecompressedData;
				/** @todo Correctly specify the actual error that occured in bzlib. **/
				throw bzlib_error("TODO: specify!");
			}
			vSetLength(0);
			vInsert(0, m_u4DecompressedLength, DecompressedData);
			delete[] DecompressedData;
			m_Compression = Arxx::Data::NONE;
			m_u4CompressedLength = 0;
#else
			throw std::runtime_error("Unsupported decompression method.");
#endif
			break;
		}
	}
}

void Arxx::Data::vCompress(const Arxx::Data::Compression & Compression)
{
	if((bIsCompressed() == true) || (bIsFetched() == false))
	{
		return;
	}
	switch(Compression)
	{
	case Arxx::Data::NONE:
		{
			break;
		}
	case Arxx::Data::ZLIB_0:
	case Arxx::Data::ZLIB_1:
	case Arxx::Data::ZLIB_2:
	case Arxx::Data::ZLIB_3:
	case Arxx::Data::ZLIB_4:
	case Arxx::Data::ZLIB_5:
	case Arxx::Data::ZLIB_6:
	case Arxx::Data::ZLIB_7:
	case Arxx::Data::ZLIB_8:
	case Arxx::Data::ZLIB_9:
		{
#ifdef HAVE_ZLIB_H
			Arxx::Data::pointer CompressedData = new Arxx::Data::value_type[static_cast< Arxx::u4byte >(u4GetDecompressedLength() * 1.001 + 12)];
			z_stream zStream;
			signed long slReturn = 0;
			
			zStream.zalloc = 0;
			zStream.zfree = 0;
			zStream.next_in = const_cast< Arxx::u1byte * >(GetBegin());
			zStream.avail_in = u4GetDecompressedLength();
			zStream.next_out = CompressedData;
			zStream.avail_out = static_cast< Arxx::u4byte >(u4GetDecompressedLength() * 1.001 + 12);
			if((slReturn = deflateInit(&zStream, Compression - Arxx::Data::ZLIB_0)) != Z_OK)
			{
				delete[] CompressedData;
				/** @todo Correctly specify the actual error that occured in zlib. **/
				throw zlib_error("TODO: specify!");
			}
			if((slReturn = deflate(&zStream, Z_FINISH)) != Z_STREAM_END)
			{
				delete[] CompressedData;
				/** @todo Correctly specify the actual error that occured in zlib. **/
				throw zlib_error("TODO: specify!");
			}
			if((slReturn = deflateEnd(&zStream)) != Z_OK)
			{
				delete[] CompressedData;
				/** @todo Correctly specify the actual error that occured in zlib. **/
				throw zlib_error("TODO: specify!");
			}
			vSetLength(0);
			vInsert(0, zStream.total_out, CompressedData);
			delete[] CompressedData;
			m_Compression = Compression;
			m_u4CompressedLength = zStream.total_out;
#else
			throw std::runtime_error("Unsupported compression method.");
#endif
			
			break;
		}
	case Arxx::Data::BZLIB:
		{
#ifdef HAVE_BZLIB_H
			Arxx::u4byte u4CompressedDataLength(static_cast< Arxx::u4byte >(u4GetDecompressedLength() * 1.01 + 600));
			Arxx::Data::pointer CompressedData(new Arxx::Data::value_type[u4CompressedDataLength]);
			bz_stream BZStream;
			
			BZStream.bzalloc = 0;
			BZStream.bzfree = 0;
			BZStream.opaque = 0;
			BZStream.next_in = reinterpret_cast< char * >(const_cast< Arxx::u1byte * >(GetBegin()));
			BZStream.avail_in = u4GetDecompressedLength();
			BZStream.next_out = reinterpret_cast< char * >(CompressedData);
			BZStream.avail_out = u4CompressedDataLength;
			
			int iBZResult;
			
			if((iBZResult = BZ2_bzCompressInit(&BZStream, 9, 0, 0)) != BZ_OK)
			{
				delete[] CompressedData;
				/** @todo Correctly specify the actual error that occured in bzlib. **/
				throw bzlib_error("TODO: specify!");
			}
			if((iBZResult = BZ2_bzCompress(&BZStream, BZ_FINISH)) != BZ_STREAM_END)
			{
				delete[] CompressedData;
				/** @todo Correctly specify the actual error that occured in bzlib. **/
				throw bzlib_error("TODO: specify!");
			}
			if((iBZResult = BZ2_bzCompressEnd(&BZStream)) != BZ_OK)
			{
				delete[] CompressedData;
				/** @todo Correctly specify the actual error that occured in bzlib. **/
				throw bzlib_error("TODO: specify!");
			}
			vSetLength(0);
			vInsert(0, BZStream.total_out_lo32, CompressedData);
			delete[] CompressedData;
			m_Compression = Compression;
			m_u4CompressedLength = BZStream.total_out_lo32;
#else
			throw std::runtime_error("Unsupported compression method.");
#endif
			break;
		}
	}
}

bool Arxx::Data::bIsCompressed(void) const
{
	return m_Compression != Arxx::Data::NONE;
}

bool Arxx::Data::bIsDecompressed(void) const
{
	return m_Compression == Arxx::Data::NONE;
}

Arxx::Data::Compression Arxx::Data::GetCompression(void) const
{
	return m_Compression;
}

Arxx::u4byte Arxx::Data::u4GetDecompressedLength(void) const
{
	if((bIsFetched() == true) && (bIsDecompressed() == true))
	{
		return m_u4DecompressedLength = stGetLength();
	}
	else
	{
		return m_u4DecompressedLength;
	}
}

Arxx::u4byte Arxx::Data::u4GetCompressedLength(void) const
{
	if((bIsFetched() == true) && (bIsCompressed() == true))
	{
		return m_u4CompressedLength = stGetLength();
	}
	else
	{
		return m_u4CompressedLength;
	}
}

const Arxx::URI & Arxx::Data::GetURI(void) const
{
	return m_URI;
}

bool Arxx::Data::bFetch(void)
{
	if((GetURI().bIsValid() == true) && (bIsFetching() == false) && (bIsFetched() == false))
	{
		m_FetchStatus = Arxx::FETCHING;
		if(Arxx::Repository.bFetchData(GetURI(), *this, m_FetchStatus) == true)
		{
			return true;
		}
		else
		{
			m_FetchStatus = Arxx::UNFETCHED;
			
			return false;
		}
	}
	else
	{
		return false;
	}
}

void Arxx::Data::vUnfetch(void)
{
	if(bIsFetched() == true)
	{
		vDelete(0, stGetLength());
		m_FetchStatus = Arxx::UNFETCHED;
	}
}

void Arxx::Data::vReleaseDataChannel(void)
{
	Arxx::Repository.vDereferenceDataChannel(GetURI());
	m_URI.vInvalidate();
}

void Arxx::Data::vSetExternal(const Arxx::URI & URI, const Arxx::Data::Compression & Compression, Arxx::u4byte u4DecompressedLength, Arxx::u4byte u4CompressedLength)
{
	m_Compression = Compression;
	m_u4DecompressedLength = u4DecompressedLength;
	m_u4CompressedLength = u4CompressedLength;
	if(GetURI().bIsValid() == true)
	{
		Arxx::Repository.vDereferenceDataChannel(GetURI());
	}
	m_URI = URI;
	if(GetURI().bIsValid() == true)
	{
		Arxx::Repository.vReferenceDataChannel(GetURI());
	}
	m_bIsExternal = true;
	m_FetchStatus = Arxx::UNFETCHED;
}

bool Arxx::Data::bIsInternal(void) const
{
	return !m_bIsExternal;
}

bool Arxx::Data::bIsExternal(void) const
{
	return m_bIsExternal;
}

void Arxx::Data::vInternalize(void)
{
	m_bIsExternal = false;
}

Arxx::FetchStatus Arxx::Data::GetFetchStatus(void) const
{
	return m_FetchStatus;
}

bool Arxx::Data::bIsFetching(void) const
{
	return (m_FetchStatus == Arxx::CONNECTING) || (m_FetchStatus == Arxx::LOGGINGIN) || (m_FetchStatus == Arxx::LOGGINGOUT) || (m_FetchStatus == Arxx::DISCONNECTING) || (m_FetchStatus == Arxx::TRANSFERING) || (m_FetchStatus == Arxx::REQUESTING) || (m_FetchStatus == Arxx::FETCHING);
}

bool Arxx::Data::bIsFetched(void) const
{
	return m_FetchStatus == Arxx::FETCHED;
}
