/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_ITEMFACTORY_H
#define ARXX_ITEMFACTORY_H

#include "Common.h"

/**
 * @file ItemFactory.h
 * 
 * This file declares the interface class Arxx::ItemFactory.
 **/

namespace Arxx
{
	class Item;
	
	class Archive;
	
	/**
	 * @brief An abstract definition of an item factory.
	 * 
	 * This class serves as an abstract entity to create Arxx::Item objects. It is not necessary to create items via an item factory. On the contrary it is quite possible to create them using new or as stack objects but whenever a library creates an item (for example while loading a library file) it will use an item factory.
	 * 
	 * @note A difficult issue is connected with the ItemFactory. Since ItemFactories are meant to be for constructing and destructing items it is quite obvious that the ItemFactory used to create an item has to be existent when the time comes to delete it. Therefore the lifespan of the Item must be contained in the lifespan of the ItemFactory. This requires you to make sure that an ItemFactory is only destructed when there are no items referring to it anymore. If need arises a functionality to support some kind of reference counting will be implemented inside this abstract base class but until then you have to do this yourself.
	 **/
	class ItemFactory
	{
	public:
		/**
		 * @brief The default constructor.
		 * 
		 * This default constructor is only here for debugging and documentation purposes. It serves no functionality.
		 **/
		ItemFactory(void);
		
		/**
		 * @brief The deconstructor.
		 * 
		 * This default deconstructor is only here for debugging and documentation purposes. It serves no functionality.
		 **/
		virtual ~ItemFactory(void);
		
		/**
		 * @brief Creates a new item.
		 * 
		 * This function is called whenever there is need for a new item. It is required to return the address of an item that is ready to be used by the application or the library component. A call to this function has to result in a constuctor call to an Arxx::Item.
		 **/
		virtual Arxx::Item * pNewItem(Arxx::u4byte u4UniqueID = Arxx::g_u4InvalidID) = 0;
	
		/**
		 * @brief Creates a new item.
		 * 
		 * This function is called whenever there is need for a new item. It is required to return the address of an item that is ready to be used by the application or the library component. A call to this function has to result in a constuctor call to an Arxx::Item.
		 **/
		virtual Arxx::Item * pNewItem(Arxx::Archive & Archive, Arxx::u4byte u4UniqueID = Arxx::g_u4InvalidID) = 0;
		
		/**
		 * @brief Destroys an item.
		 * 
		 * This function will destroy an item by calling the items destructor and doing anything necessary to release resources or unregister other things.
		 **/
		virtual void vDeleteItem(Arxx::Item * pItem) = 0;
	};
}

#endif
