/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_ITEM_H
#define ARXX_ITEM_H

#include "Data.h"
#include "Structure.h"

/**
 * @file Item.h
 * 
 * This file declares the interface of Arxx::Item.
 **/

namespace Arxx
{
	class ItemFactory;
	
	class Archive;
	
	/**
	 * @brief The Arxx::Item is the basic class for any data storaging.
	 *
	 * The Item is a thin wrapper around Arxx::Data but the inheritance ensures you can also treat it as an Arxx::Data object.
	 *
	 * It adds the basics of ARX to the data storage: Names, IDs, Types, SubTypes and a connection to a parenting Arxx::Archive.
	 *
	 * Additionally, at the moment it provides basic functionality for the structure, but that is about to change. Structure should only be handled by external structure classes.
	 **/
	class Item : public Arxx::Data
	{
		friend class Arxx::Archive;
	public:
		/**
		 * @brief The "Archive Constructor" creates an Item that is associated to a certain Arxx::Archive.
		 * @param Archive The Arxx::Archive which will be used to register the Item after it has been set up.
		 * @param u4UniqueID The item's unique ID that will be tried for the Item.
		 *
		 * After all the member variables have been initialized Archive.vRegisterItem(this) is issued.
		 **/
		Item(Arxx::Archive & Archive, Arxx::u4byte u4UniqueID = g_u4InvalidID);
		
		/**
		 * @brief The "Archive & Factory Constructor" creates an Item that is associated to a certain Arxx::Archive and to a certain ItemFactory.
		 * @param ItemFactory The Arxx::ItemFactory that this item is created from and will be destroyed from.
		 * @param Archive The Arxx::Archive which will be used to register the Item after it has been set up.
		 * @param u4UniqueID The item's unique ID that will be tried for the Item.
		 *
		 * After all the member variables have been initialized Archive.vRegisterItem(this) is issued.
		 **/
		Item(Arxx::ItemFactory & ItemFactory, Arxx::Archive & Archive, Arxx::u4byte u4UniqueID = g_u4InvalidID);
		
		/**
		 * @brief The default constructor, with the additional ability to define a unique identifier.
		 * @param u4UniqueID The unique ID you wish to assign to the new Arxx::Item.
		 * 
		 * This constructor is available so you can create items which don't belong to a library initially.
		 **/
		explicit Item(Arxx::u4byte u4UniqueID = g_u4InvalidID);
		
		/**
		 * @brief The Factory constructor, with the additional ability to define a unique identifier.
		 * @param ItemFactory The Arxx::ItemFactory that this item is created from and will be destroyed from.
		 * @param u4UniqueID The unique ID you wish to assign to the new Arxx::Item.
		 * 
		 * This constructor is available so you can create items which don't belong to a library initially.
		 **/
		Item(Arxx::ItemFactory & ItemFactory, Arxx::u4byte u4UniqueID = g_u4InvalidID);
		
		/**
		 * @brief The destructor for items.
		 *
		 * This destructor is virtual so that you can savely derieve from the Arxx::Item.
		 **/
		virtual ~Item();
		
		/**
		 * @brief This function returns the unique identifier of @em this Arxx::Item.
		 * 
		 * Returns the unique identifier of @em this item. Remember that it is only unique inside a certain ARX archive.
		 **/
		Arxx::u4byte u4GetUniqueID(void) const;
		
		/**
		 * @brief Sets @em this item's unique identifier inside the library.
		 * @param u4UniqueID The new unique identifier of this item.
		 * 
		 * This function is trivial for items that don't belong to any library. The unique identifier has no meaning there and will just get assigned.
		 * 
		 * If @em this Arxx::Item is inside an Arxx:Archive the function is responsible for checking that the new identifier is not used yet.
		 **/
		void vSetUniqueID(Arxx::u4byte u4UniqueID);
		
		/**
		 * @brief Returns a const reference to the items name.
		 *
		 * This function is called for const items and returns a const reference to the item's name.
		 **/
		const std::string & sGetName(void) const;
		
		/**
		 * @brief Set the item's name.
		 * @param sName The new name of the item.
		 *
		 * This function lets you change the name of an item to any value representable in a std::string.
		 **/
		void vSetName(const std::string & sName);
		
		/**
		 * @brief Returns the type identifier of @em this item.
		 * 
		 * The type identifier has no sematic meaning to ARX or libarxx. It only serves the purpose of getting a meaning by the application.
		 *
		 * The default value for the type is 0xFFFFFFFF.
		 **/
		Arxx::u4byte u4GetType(void) const;
		
		/**
		 * @brief Returns the sub-type identifier of @em this item.
		 * 
		 * As the type the sub-type identifier has no sematic meaning to ARX or libarxx. It is meant to be placed into meaning by the application.
		 *
		 * The default value for the sub type is 0xFFFFFFFF.
		 **/
		Arxx::u4byte u4GetSubType(void) const;
		
		/**
		 * @brief Returns the version number of @em this item.
		 *
		 * The version number has no semantic meaning within ARX or libarxx and is only related to the content. It may represent ANY kind of information however it SHOULD be useful as a indicator for the up-to-dateness of the item's content.
		 *
		 * If the value is actually a version number it should be treated like xx.xx.xx.xx with each component ranging from 0 to 255. However you may also like to store the timestamp when the item was created.
		 *
		 * The default value for the version number is 0.0.0.0.
		 **/
		Arxx::u4byte u4GetVersion(void) const;
		
		/**
		 * @brief Enables you to set the type identifier of @em this item.
		 * @param u4Type The type identifier you whish to set for @em this item.
		 * 
		 * The type identifier has no sematic meaning to ARX or libarxx. It only serves the purpose of getting a meaning by the application.
		 **/
		void vSetType(Arxx::u4byte u4Type);
		
		/**
		 * @brief Enables you to set the sub-type identifier of @em this item.
		 * @param u4SubType The sub-type identifier you whish to set for @em this item.
		 * 
		 * As the type the sub-type identifier has no sematic meaning to ARX or libarxx. It is meant to be placed into meaning by the application.
		 **/
		void vSetSubType(Arxx::u4byte u4SubType);
		
		/**
		 * @brief Sets the version number of this @em item.
		 * @param u4Version The new version number of the item.
		 *
		 * Sets the versin number of the item. Note that this information is purely related to the content and in no way related to the components or structure of the Arxx::Item.
		 **/
		void vSetVersion(Arxx::u4byte u4Version);
		
		/**
		 * @brief Returns a pointer to the registered Archive.
		 *
		 * The function will return `0` if there is no Archive set for the Item.
		 **/
		Arxx::Archive * pGetArchive(void) const;
		
		/**
		 * @brief Returns a pointer to the ItemFactory that @em this item is a part of.
		 *
		 * The function will return 0 if there is no item factory set for this item.
		 **/
		Arxx::ItemFactory * pGetItemFactory(void) const;
		
		/**
		 * @brief Returns the Structure component associated with this Item.
		 **/
		Arxx::Structure & GetStructure(void);
		
		/**
		 * @brief Returns the const Structure component associated with this Item.
		 **/
		const Arxx::Structure & GetStructure(void) const;
	private:
		/**
		 * @brief Hide the copy constructor from user space. We won't need it in any tool here.
		 * @sa If you ever should be in need of this functionality I need to figure out the semantics.
		 **/
		Item(const Item & Item);
		
		/**
		 * @brief Hide to assignment operator from user space.
		 * @sa If you are in need of this functionality, tell me so I can figure out the semantis needed for this.
		 **/
		Item & operator=(const Item & Item);
		
		/**
		 * @brief The item's unique ID to identify it inside a library.
		 * 
		 * The item's unique ID will identify the item in a library. The ID allows access to the item in O(log(n)).
		 * 
		 * Depending on the constructor this field is either initialized with 0xFFFFFFFF or with the unique ID passed to the constructor.
		 **/
		u4byte m_u4UniqueID;
		
		/**
		 * @brief The item's name in string form.
		 * 
		 * The name is in no way meant to identify the item. It's just used to have a short description of the item (although no length restriction is given).
		 * 
		 * Despite this, the name can be used to build paths over the items of the library.
		 * 
		 * Paths look like this:
		 * @code
		 * /
		 * /some
		 * /some/path
		 * /[child]some/[child]path
		 * /[child]some/path/[parent]some/path
		 * /[child]some/[other]path
		 * @endcode
		 * 
		 * After creating an Item this value is intentionally "".
		 **/
		std::string m_sName;
		
		/**
		 * @brief A type identifier for @em this item.
		 * 
		 * The type of an item has no semantic meaning inside ARX or libarxx. It exists only to be used by applications to identify the type of the data of @em this item. Think of it as a replacement for extensions.
		 * 
		 * After creating an Item this value is intentionally 0xFFFFFFFF.
		 **/
		Arxx::u4byte m_u4Type;
		
		/**
		 * @brief A subtype identifier for @em this item.
		 * 
		 * Like the type the subtype has no semantic meaning  inside ARX or libarxx. Ite exists only to be used by the application to identify subtypes of types. This is useful for example to distinguish between different versions of data format that belong to the same type. It can also be used to highlight semantic meaning of a general data format on filesystem layer, like identifying all kinds of log files with a certain type and distinguishing between error, info, debug and critical with different subtypes.
		 * 
		 * After creating an Item this value is intentionally 0xFFFFFFFF.
		 **/
		Arxx::u4byte m_u4SubType;
		
		/**
		 * @brief This field stores versioning information about the data.
		 * 
		 * This version information, given in xx.xx.xx.xx format, allows applications to assign a running version number of @em this item's data. It should strictly refere to data content only, not the data format. Again this property has no semantic meaning inside ARX or libarxx, however the more recently the Item is been changed the higher the version number should generally be.
		 * 
		 * After creating an Item this value is intentionally 0x00000000.
		 **/
		Arxx::u4byte m_u4Version;
		
		/**
		 * @brief A pointer to the Archive.
		 * 
		 * Depending on the constructor this field is either initialized with 0 or with the address of the Archive passed to the constructor.
		 * 
		 * It is changed in conjuction with the Arxx::Archive::vRegisterItem() and Arxx::Archive::vUnregisterItem() functions.
		 **/
		Arxx::Archive * m_pArchive;
		
		/**
		 * @brief A pointer to the library that @a this Item was created with.
		 * 
		 * Depending on the constructor this field is either initialized with 0 or with the address of the ItemFactory passed to the constructor.
		 **/
		Arxx::ItemFactory * m_pItemFactory;
		
		/**
		 * @brief The structural information of an item.
		 **/
		Arxx::Structure m_Structure;
	};
}

#endif
