/**
 * Copyright (c) OpenLens Authors. All rights reserved.
 * Licensed under MIT License. See LICENSE in root directory for more information.
 */

import fse from "fs-extra";
import type { ClusterModel } from "../../common/cluster-types";
import type { MigrationDeclaration } from "../helpers";
import { getLegacyGlobalDiForExtensionApi } from "../../extensions/as-legacy-globals-for-extension-api/legacy-global-di-for-extension-api";
import directoryForUserDataInjectable from "../../common/app-paths/directory-for-user-data/directory-for-user-data.injectable";
import { isErrnoException } from "../../common/utils";
import joinPathsInjectable from "../../common/path/join-paths.injectable";

interface Pre500WorkspaceStoreModel {
  workspaces: {
    id: string;
    name: string;
  }[];
}

export default {
  version: "5.0.0-beta.10",
  run(store) {
    const di = getLegacyGlobalDiForExtensionApi();

    const userDataPath = di.inject(directoryForUserDataInjectable);
    const joinPaths = di.inject(joinPathsInjectable);

    try {
      const workspaceData: Pre500WorkspaceStoreModel = fse.readJsonSync(joinPaths(userDataPath, "lens-workspace-store.json"));
      const workspaces = new Map<string, string>(); // mapping from WorkspaceId to name

      for (const { id, name } of workspaceData.workspaces) {
        workspaces.set(id, name);
      }

      const clusters: ClusterModel[] = store.get("clusters") ?? [];

      for (const cluster of clusters) {
        if (cluster.workspace) {
          const workspace = workspaces.get(cluster.workspace);

          if (workspace) {
            (cluster.labels ??= {}).workspace = workspace;
          }
        }
      }

      store.set("clusters", clusters);
    } catch (error) {
      if (isErrnoException(error) && !(error.code === "ENOENT" && error.path?.endsWith("lens-workspace-store.json"))) {
        // ignore lens-workspace-store.json being missing
        throw error;
      }
    }
  },
} as MigrationDeclaration;
